package pageroute

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"time"

	"gioui.org/io/clipboard"
	"gioui.org/layout"
	"gioui.org/unit"
	"gioui.org/widget/material"

	"git.sr.ht/~mil/mobroute"
	"git.sr.ht/~mil/transito/src/util/utilfuncs"
	"git.sr.ht/~mil/transito/src/util/utiltheme"
)

func (page *PageRoute) widgetRowActionButtons(gtx layout.Context) layout.Dimensions {
	gtx.Constraints.Min.Y = gtx.Dp(unit.Dp(40))
	gtx.Constraints.Max.Y = gtx.Constraints.Min.Y
	return layout.Flex{}.Layout(gtx, layout.Rigid(dimensionsToWidget(layout.Flex{}.Layout(
		gtx,
		layout.Flexed(0.2, page.widgetBtnParams),
		layout.Flexed(0.01, layout.Spacer{Width: unit.Dp(2)}.Layout),
		layout.Flexed(0.2, page.widgetButtonRoute),
	))))
}

func (page *PageRoute) widgetButtonRoute(gtx layout.Context) layout.Dimensions {
	// Update result area with mobroute timer value when running
	if page.TimerStartMobroute != nil {
		extraMessage := ""

		page.ResultMRRouteErrorDbglog = true
		page.ResultMRRouteError = fmt.Errorf("**%s**: %.1fs\n\n%s",
			page.TimerStage,
			time.Since(*page.TimerStartMobroute).Seconds(),
			extraMessage)
		page.GlobalState.Window.Invalidate()
	}

	if page.clickableRoute.Clicked(gtx) {
		go func() {
			if page.TimerStartMobroute != nil {
				return
			}

			now := time.Now()
			page.TimerStartMobroute = &now
			defer func() {
				page.TimerStartMobroute = nil
				page.GlobalState.Window.Invalidate()
			}()

			if len(page.GlobalState.Save.FeedIDsEnabled) == 0 {
				page.ResultMRRouteError = errors.New("Error: **Must enable atleast 1 source to route**")
				page.ResultMRRouteErrorDbglog = false
				page.ResultMRRoute = nil
				return
			} else if len(page.GlobalState.LocationFromCoords) < 2 || len(page.GlobalState.LocationToCoords) < 2 {
				page.ResultMRRouteError = errors.New("Error: **Set From/To Coordinates on Stops/POI Tab**")
				page.ResultMRRouteErrorDbglog = false
				page.ResultMRRoute = nil
				return
			}

			// SCHEDULE_DEPART_NOW: Just use the current day/time (default)
			// SCHEDULE_DEPART_AT:  Use custom defined day/time
			routeDepartureTime := time.Now()
			if page.scheduleType.Value == SCHEDULE_DEPART_AT {
				departTimeRaw := page.textfieldDepartureTime.Text()
				if departTime, err := time.Parse("15:04", departTimeRaw); err != nil {
					page.ResultMRRouteError = errors.New("Error: **Invalid departure time**")
					page.ResultMRRouteErrorDbglog = false
					page.ResultMRRoute = nil
					return
				} else {
					// Merge departure date and departure time.
					routeDepartureTime = time.Date(
						page.scheduleDay.Year(), page.scheduleDay.Month(), page.scheduleDay.Day(),
						departTime.Hour(), departTime.Minute(),
						0, 0, now.Location())

				}
			}
			fmt.Println("Using departure time:", routeDepartureTime)

			// Route
			var mrParams = &mobroute.RouteParams{
				OutputFormats:      []string{"legs", "diagnostics", "request", "mapurl"},
				FeedIDs:            page.GlobalState.Save.FeedIDsEnabled,
				From:               [2]float64{page.GlobalState.LocationFromCoords[0], page.GlobalState.LocationFromCoords[1]},
				To:                 [2]float64{page.GlobalState.LocationToCoords[0], page.GlobalState.LocationToCoords[1]},
				Time:               &routeDepartureTime,
				MaxTripSeconds:     parseUint(page.textfieldMaxTripMinutes.Text()) * 60,
				MinTransferSeconds: parseUint(page.textfieldMinTransferMinutes.Text()) * 60,
				MaxTransferSeconds: parseUint(page.textfieldMaxTransferMinutes.Text()) * 60,
				MaxWalkSeconds:     parseUint(page.textfieldMaxWalkMinutes.Text()) * 60,
				WalkspeedKmHr:      parseFloat64(page.textfieldWalkspeedKmHr.Text()),
				TransferCategories: func() []string {
					categories := []string{}
					if page.checkboxTransfersFeed.Value {
						categories = append(categories, "f")
					}
					if page.checkboxTransfersImplicit.Value {
						categories = append(categories, "i")
					}
					if page.checkboxTransfersGenerated.Value {
						categories = append(categories, "g")
					}
					return categories
				}(),
			}

			page.TimerStage = fmt.Sprintf("Route Calculation for %v", page.GlobalState.Save.FeedIDsEnabled)
			page.ResultMRRouteErrorDbglog = true
			page.ResultMRRoute = nil
			page.GlobalState.MobrouteLog.Reset()
			if mobrouteResult, err := mobroute.RTRoute(page.GlobalState.MobrouteRuntime, mrParams); err != nil {
				// Copy MR request to clipboard for replication
				j, _ := json.Marshal(mrParams)
				gtx.Execute(clipboard.WriteCmd{Data: io.NopCloser(bytes.NewReader(j))})
				page.ResultMRRouteErrorDbglog = true
				page.ResultMRRouteError = fmt.Errorf(
					"Error: **Failed Route Calculation**\n\n(Raw Error: **%s**)",
					err,
				)
				page.ResultMRRoute = nil
			} else {
				page.ResultMRRoute = mobrouteResult
				page.ResultMRRouteError = err
				page.ResultMRRouteErrorDbglog = true
			}
		}()
	}

	return material.Button(
		utiltheme.MaterialTheme(
			*page.GlobalState.Theme,
			utilfuncs.Fif(page.TimerStartMobroute != nil, utiltheme.ThemecolorDisabled, utiltheme.ThemecolorDefault),
			16,
		),
		&page.clickableRoute,
		"Route",
	).Layout(gtx)
}

func (page *PageRoute) widgetBtnParams(gtx layout.Context) layout.Dimensions {
	if page.clickableParams.Clicked(gtx) {
		page.showParams = !page.showParams
	}

	var label = "Params"
	var color = utiltheme.ThemecolorFgBlackBgGray
	if page.showParams {
		label = "Hide Params"
		color = utiltheme.ThemecolorFgBlackBgGray
	}

	return material.Button(
		utiltheme.MaterialTheme(*page.GlobalState.Theme, color, 14),
		&page.clickableParams, label,
	).Layout(gtx)
}
