package pageroute

import (
	"bytes"
	"encoding/json"
	"io"
	"log"
	"net/url"
	"time"

	"gioui.org/io/clipboard"
	"gioui.org/io/key"
	"gioui.org/layout"
	"gioui.org/unit"
	"gioui.org/widget"
	"gioui.org/widget/material"
	"gioui.org/x/component"
	"gioui.org/x/richtext"
	"git.sr.ht/~mil/mobroute"
	"git.sr.ht/~mil/transito/src/components/buttonrow"
	"git.sr.ht/~mil/transito/src/globalstate"
	"git.sr.ht/~mil/transito/src/util/utiltheme"
	"github.com/gioui-plugins/gio-plugins/hyperlink/giohyperlink"
	"github.com/gioui-plugins/gio-plugins/plugin/gioplugins"
)

type PageRoute struct {
	GlobalState *globalstate.GlobalState
	list        widget.List

	richtextFrom       richtext.InteractiveText
	richtextTo         richtext.InteractiveText
	richtextResultArea richtext.InteractiveText

	ResultMRRoute            *mobroute.RouteResponse
	ResultMRRouteErrorDbglog bool
	ResultMRRouteError       error

	buttonRow buttonrow.ButtonRow

	showParams bool

	// Params
	checkboxTransfersFeed       widget.Bool
	checkboxTransfersImplicit   widget.Bool
	checkboxTransfersGenerated  widget.Bool
	textfieldMaxWalkMinutes     component.TextField
	textfieldMaxTripMinutes     component.TextField
	textfieldMaxTransferMinutes component.TextField
	textfieldMinTransferMinutes component.TextField
	textfieldWalkspeedKmHr      component.TextField
	textfieldDepartureTime      component.TextField // time without date. See scheduleDay.

	clickableRoute   widget.Clickable
	clickableParams  widget.Clickable
	clickableDayNext widget.Clickable
	clickableDayPrev widget.Clickable

	scheduleType widget.Enum
	scheduleDay  time.Time // Day without time. See textfieldDepartureTime.

	showLog              bool
	clickableShowHideLog widget.Clickable

	TimerStage         string
	TimerStartMobroute *time.Time
}

func Initialize(u *globalstate.GlobalState) *PageRoute {
	page := PageRoute{
		GlobalState:        u,
		list:               widget.List{List: layout.List{Axis: layout.Vertical}},
		ResultMRRoute:      nil,
		ResultMRRouteError: nil,
		TimerStartMobroute: nil,

		buttonRow: buttonrow.ButtonRow{
			Theme: u.Theme,
			Buttons: []buttonrow.Button{
				{Label: "Overview", CanActivate: true, Themecolor: utiltheme.ThemecolorFgBlackBgGray},
				{Label: "Debug", CanActivate: true, Themecolor: utiltheme.ThemecolorFgBlackBgGray},
				{Label: "Clear", CanActivate: false, Themecolor: utiltheme.ThemecolorFgBlackBgGray},
				{Align: buttonrow.Align(buttonrow.AlignRight), Label: "Map", CanActivate: false, Themecolor: utiltheme.ThemecolorFgBlackBgGreen},
			},
			ActiveButtonLabel:      "Overview",
			ActiveButtonThemecolor: utiltheme.ThemecolorFgBlackBgBlue,
		},

		textfieldMaxWalkMinutes:     component.TextField{Editor: widget.Editor{Filter: "1234567890", InputHint: key.HintNumeric}},
		textfieldMaxTripMinutes:     component.TextField{Editor: widget.Editor{Filter: "1234567890", InputHint: key.HintNumeric}},
		textfieldMaxTransferMinutes: component.TextField{Editor: widget.Editor{Filter: "1234567890", InputHint: key.HintNumeric}},
		textfieldMinTransferMinutes: component.TextField{Editor: widget.Editor{Filter: "1234567890", InputHint: key.HintNumeric}},
		textfieldWalkspeedKmHr:      component.TextField{Editor: widget.Editor{Filter: "1234567890.", InputHint: key.HintNumeric}},
		textfieldDepartureTime:      component.TextField{Editor: widget.Editor{Filter: "1234567890:", InputHint: key.HintNumeric}, CharLimit: 5},
	}

	// Defaults
	page.textfieldMaxWalkMinutes.SetText("20")
	page.textfieldMaxTripMinutes.SetText("240")
	page.textfieldMinTransferMinutes.SetText("3")
	page.textfieldMaxTransferMinutes.SetText("40")
	page.textfieldWalkspeedKmHr.SetText("3.5")
	page.checkboxTransfersFeed.Value = true
	page.checkboxTransfersImplicit.Value = true
	page.textfieldDepartureTime.SetText(time.Now().Format("15:04"))
	for _, tf := range []*component.TextField{
		&page.textfieldMaxWalkMinutes,
		&page.textfieldMaxTripMinutes,
		&page.textfieldMinTransferMinutes,
		&page.textfieldMaxTransferMinutes,
		&page.textfieldWalkspeedKmHr,
		&page.textfieldDepartureTime,
	} {
		tf.MoveCaret(9999, 99999)
	}
	page.scheduleType.Value = SCHEDULE_DEPART_NOW
	page.scheduleDay = time.Now()

	pageptr := &page
	// Clipboard copy
	page.buttonRow.Buttons[1].Callback = func(gtx layout.Context) {
		if textBytes, err := json.Marshal(page.ResultMRRoute.Request); err == nil {
			gtx.Execute(clipboard.WriteCmd{Data: io.NopCloser(bytes.NewReader(textBytes))})
		} else {
			log.Println("Error marshaling MR request to JSON: ", err)
		}
	}
	// Clear
	page.buttonRow.Buttons[2].Callback = func(gtx layout.Context) {
		pageptr.ResultMRRoute = nil
	}
	// Open map URL for map button
	page.buttonRow.Buttons[3].Callback = func(gtx layout.Context) {
		if u, err := url.Parse(*pageptr.ResultMRRoute.MapURL); err != nil {
			log.Printf("Error parsing link <%s>: %v", *pageptr.ResultMRRoute.MapURL, err)
		} else {
			gioplugins.Execute(gtx, giohyperlink.OpenCmd{URI: u})
		}
	}
	return &page
}

func (page *PageRoute) Render(gtx layout.Context) layout.Dimensions {
	// E.g. Update for needed elements immediatly rerenders state as necessary
	page.scheduleType.Update(gtx)

	var widgetsScroll = []layout.Widget{}
	if len(page.GlobalState.Save.FeedIDsEnabled) == 0 {
		widgetsScroll = append(
			widgetsScroll,
			material.Body1(page.GlobalState.Theme, "Must enable atleast 1 GTFS source to route").Layout,
		)
	} else {
		widgetsScroll = append(widgetsScroll, page.widgetRowSplitToFrom, page.widgetRowActionButtons)
		if page.showParams {
			widgetsScroll = append(widgetsScroll, page.widgetParams()...)
		}
		widgetsScroll = append(widgetsScroll, page.widgetRowResultMarkdown(gtx)...)
		widgetsScroll = append(widgetsScroll, page.widgetRowLogToggle(gtx)...)
	}

	return material.List(page.GlobalState.Theme, &page.list).Layout(
		gtx,
		len(widgetsScroll),
		func(gtx layout.Context, i int) layout.Dimensions {
			return layout.UniformInset(unit.Dp(16)).Layout(gtx, widgetsScroll[i])
		},
	)
}
