package pageload

import (
	"fmt"
	"time"

	"gioui.org/font"
	"gioui.org/layout"
	"gioui.org/unit"
	"gioui.org/widget"
	"gioui.org/widget/material"
	"git.sr.ht/~mil/transito/src/components/styledbutton"
	"git.sr.ht/~mil/transito/src/globalstate"
	"git.sr.ht/~mil/transito/src/util/utilfuncs"
)

type PageLoad struct {
	GlobalState          *globalstate.GlobalState
	list                 widget.List
	showLog              bool
	clickableShowHideLog widget.Clickable
}

func Initialize(u *globalstate.GlobalState) *PageLoad {
	page := PageLoad{
		GlobalState: u,
		list:        widget.List{List: layout.List{Axis: layout.Vertical}},
	}
	return &page
}

func (page *PageLoad) Render(gtx layout.Context) layout.Dimensions {
	var widgetsScroll = []layout.Widget{}

	widgetsScroll = append(widgetsScroll, func(gtx layout.Context) layout.Dimensions {
		var (
			pad         = unit.Dp(2)
			textLine    = material.Body1(page.GlobalState.Theme, page.GlobalState.PendingLoadInfo.Message)
			subtextLine = material.Body2(page.GlobalState.Theme, page.GlobalState.PendingLoadInfo.SubMessage)
		)
		textLine.Font.Weight = font.Bold
		subtextLine.Font.Style = font.Italic

		return layout.Flex{Axis: layout.Vertical}.Layout(
			gtx,
			layout.Rigid(func(gtx layout.Context) layout.Dimensions {
				inset := layout.Inset{Top: pad, Bottom: pad}
				return inset.Layout(gtx, func(gtx layout.Context) layout.Dimensions {
					return textLine.Layout(gtx)
				})
			}),
			layout.Rigid(func(gtx layout.Context) layout.Dimensions {
				inset := layout.Inset{Top: 0, Bottom: 0}
				return inset.Layout(gtx, func(gtx layout.Context) layout.Dimensions {
					return subtextLine.Layout(gtx)
				})
			}),
		)
	})

	if !page.GlobalState.PendingLoadInfo.IsError && page.GlobalState.PendingLoadInfo.TimerStart != nil {
		diff := time.Since(*page.GlobalState.PendingLoadInfo.TimerStart).Seconds()
		progressText := material.Body2(page.GlobalState.Theme, fmt.Sprintf("Progress: %.1fs ", diff))
		progressText.Font.Style = font.Italic
		widgetsScroll = append(widgetsScroll, progressText.Layout)
	}

	showHideLogButton := styledbutton.StyledButtonSmall(
		gtx, *page.GlobalState.Theme, &page.clickableShowHideLog,
		utilfuncs.Fif(page.showLog, "Hide Log", "Show Log"),
	)

	if page.clickableShowHideLog.Clicked(gtx) {
		page.showLog = !page.showLog
	}
	widgetsScroll = append(widgetsScroll, showHideLogButton)

	if page.showLog {
		logText := material.Body2(page.GlobalState.Theme, page.GlobalState.MobrouteLog.String())
		logText.Font.Style = font.Italic
		widgetsScroll = append(widgetsScroll, logText.Layout)
	}

	return material.List(page.GlobalState.Theme, &page.list).Layout(
		gtx,
		len(widgetsScroll),
		func(gtx layout.Context, i int) layout.Dimensions {
			return layout.UniformInset(unit.Dp(17)).Layout(gtx, widgetsScroll[i])
		},
	)
}
