package integrations

import (
	"fmt"
	"sort"
	"strings"
	"time"

	"git.sr.ht/~mil/mobroute"
	"git.sr.ht/~mil/transito/src/util/utilfuncs"
)

var CacheTabledataSearch = map[string]*[]string{}

func cacheKey(
	feedsearchFilter *mobroute.FeedsearchFilter,
	excludeFeedIDs []int,
	markdownify, offline bool,
) string {
	feedIDs := feedsearchFilter.FeedID
	sort.Ints(feedIDs)
	return fmt.Sprintf(
		"cachekey_%s_%s_%s_%v_%v",
		strings.Join(utilfuncs.Strvals(feedIDs), "_"),
		strings.Join(utilfuncs.Strvals(excludeFeedIDs), "_"),
		feedsearchFilter.Glob,
		markdownify, offline,
	)

}

func MobrouteIntegrationGetTableDataForFeedIDs(
	mobrouteRuntime *mobroute.MobrouteRuntime,
	feedsearchFilter *mobroute.FeedsearchFilter,
	excludeFeedIDs []int,
	markdownify, offline bool,
) (*[]string, error) {
	requestCacheKey := cacheKey(feedsearchFilter, excludeFeedIDs, markdownify, offline)

	if len(feedsearchFilter.FeedID) == 0 && feedsearchFilter.Glob == "" {
		return &[]string{}, nil
	} else if cachedReturn, exists := CacheTabledataSearch[requestCacheKey]; exists {
		return cachedReturn, nil
	} else if searchResults, err := mobroute.RTDatabaseq(mobrouteRuntime, &mobroute.DatabaseqParams{
		FeedSearchFilter: feedsearchFilter,
	}); err != nil {
		return nil, err
	} else {
		limitedSearchResults := []*mobroute.FeedStatusInfo{}
		for _, sr := range *searchResults {
			if !utilfuncs.Contains(excludeFeedIDs, sr.FeedID) {
				limitedSearchResults = append(limitedSearchResults, &sr)
			}
			if len(limitedSearchResults) > 10 {
				break
			}
		}
		returnItems := []string{}
		for _, r := range limitedSearchResults {
			returnItems = append(returnItems, constructFeedInfoLine(r, markdownify, offline))
		}
		CacheTabledataSearch[requestCacheKey] = &returnItems
		return &returnItems, nil
	}
}

func constructFeedInfoLine(r *mobroute.FeedStatusInfo, markdownify, offline bool) string {
	locationField := func() string {
		nonNilLocationFields := []string{}
		for _, l := range []*string{
			r.LocationCountry, r.Name, r.Provider, r.LocationMunicipality, r.LocationMunicipality, r.LocationSubdivision,
		} {
			if l != nil {
				nonNilLocationFields = append(nonNilLocationFields, *l)
			}
		}
		return strings.Join(nonNilLocationFields, " / ")
	}()

	var appendString string
	if markdownify {
		appendString = fmt.Sprintf("**%d**: %s", r.FeedID, strings.TrimSpace(utilfuncs.Truncate(locationField, 50)))
		if r.MinDate != nil && r.MaxDate != nil {
			appendString = fmt.Sprintf("%s (*%s → %s*)", appendString, prettifyDate(*r.MinDate), prettifyDate(*r.MaxDate))
		}
		if r.TimestampComputed != nil {
			appendString = fmt.Sprintf("%s (Updated: *%s*)", appendString, *r.TimestampComputed)
		}
		if r.URLLicense != nil {
			appendString = fmt.Sprintf("%s | [License](%s)", appendString, *r.URLLicense)
		}
		if !offline && r.FeedID > 0 { // E.g. only show if not offline & for feed IDs that can be 'updated' (e.g. MDB feeds; feedID>0)
			appendString = fmt.Sprintf("%s | [Update](transito://feedid_update/%d)", appendString, r.FeedID)
		}
		appendString = fmt.Sprintf("%s | [Remove](transito://feedid_remove/%d)", appendString, r.FeedID)
	} else {
		appendString = fmt.Sprintf("%d: %s", r.FeedID, utilfuncs.Truncate(locationField, 50))
		if r.TimestampComputed != nil {
			appendString = fmt.Sprintf("%s (cached: %v)", appendString, *r.TimestampComputed)
		}
	}
	return appendString
}

func prettifyDate(yyyymmdd string) string {
	if p, err := time.Parse("20060102", yyyymmdd); err != nil {
		return yyyymmdd
	} else {
		return p.Format("2006-01-02")
	}
}
