package ui

import (
	"log"
	"os"

	"gioui.org/app"
	"gioui.org/font/gofont"
	"gioui.org/op"
	"gioui.org/op/clip"
	"gioui.org/text"
	"gioui.org/unit"
	"gioui.org/widget/material"
	"gioui.org/x/explorer"
	"gioui.org/x/markdown"
	"git.sr.ht/~mil/transito/src/globalstate"
	"git.sr.ht/~mil/transito/src/uipages/pageconfig"
	"git.sr.ht/~mil/transito/src/uipages/pagefave"
	"git.sr.ht/~mil/transito/src/uipages/pageload"
	"git.sr.ht/~mil/transito/src/uipages/pagepois"
	"git.sr.ht/~mil/transito/src/uipages/pageroute"
	"git.sr.ht/~mil/transito/src/uipages/pagestops"
	"git.sr.ht/~mil/transito/src/util/utilplatform"
	"github.com/gioui-plugins/gio-plugins/plugin/gioplugins"
)

func CreateUI() {
	utilplatform.TimezoneFixAndroid()

	// (1) Create UI struct
	uiInstance := &UI{
		GlobalState: globalstate.GlobalState{
			MobrouteRuntime:  nil,
			MarkdownRenderer: markdown.NewRenderer(),
			Theme:            material.NewTheme(),
			Save:             globalstate.FileLoadWarnError(),
		},
	}
	uiInstance.GlobalState.Window = &app.Window{}
	uiInstance.GlobalState.Window.Option(app.Title("Transito"), app.Size(unit.Dp(420), unit.Dp(780)))
	uiInstance.GlobalState.Explorer = explorer.NewExplorer(uiInstance.GlobalState.Window)
	uiInstance.GlobalState.Theme.Shaper = text.NewShaper(text.WithCollection(gofont.Collection()))
	uiInstance.Tabs = []UITab{
		{Tabname: "Route", Page: pageroute.Initialize(&uiInstance.GlobalState)},
		{Tabname: "Stops", Page: pagestops.Initialize(&uiInstance.GlobalState)},
		{Tabname: "POI", Page: pagepois.Initialize(&uiInstance.GlobalState)},
		{Tabname: "Fave", Page: pagefave.Initialize(&uiInstance.GlobalState)},
		{Tabname: "Config", Page: pageconfig.Initialize(&uiInstance.GlobalState)},
		{Tabname: "Load", Page: pageload.Initialize(&uiInstance.GlobalState)},
	}

	// (2) Triggers initial loading of MRRuntime (see uiservices.go)
	uiInstance.GlobalState.AppQueueload = &globalstate.AppQueueload{RuntimeInitialize: true}

	// (3) Pass to Run (main gio loop)
	go func() {
		if err := uiInstance.Run(uiInstance.GlobalState.Window); err != nil {
			log.Println(err)
			os.Exit(1)
		}
		os.Exit(0)
	}()

	// (4) Gio main
	app.Main()
}

func (ui *UI) Run(w *app.Window) error {
	var ops op.Ops
	for {
		//e := w.Event()
		e := gioplugins.Hijack(w)
		ui.GlobalState.Explorer.ListenEvents(e)
		switch e := e.(type) {
		case app.FrameEvent:
			gtx := app.NewContext(&ops, e)
			area := clip.Rect{Max: gtx.Constraints.Max}.Push(gtx.Ops)
			if ui.GlobalState.AppQueueload != nil {
				ui.ServiceQueueload(ui.GlobalState.AppQueueload)
				ui.GlobalState.AppQueueload = nil
			}
			go globalstate.IntervalSaveWatch(&ui.GlobalState.Save)
			ui.RenderTabbedUI(gtx)
			area.Pop()
			e.Frame(gtx.Ops)
		case app.DestroyEvent:
			return e.Err
		}
	}

	return nil
}
