package pagestops

import (
	"fmt"
	"log"
	"regexp"
	"strings"

	"gioui.org/layout"
	"gioui.org/unit"
	"gioui.org/widget/material"
	"gioui.org/x/richtext"
	"git.sr.ht/~mil/transito/src/uipages/pagestops/integrations"
	"git.sr.ht/~mil/transito/src/util/utillinkhandler"
)

func (widget *PageStops) renderSearchWidgetsSelected(gtx layout.Context) []layout.Widget {
	renderSelectedLabel, renderSelectedMDData := widget.determineSelectedData()
	widget.eventhandlersSearchWidgetsSelected(gtx, renderSelectedLabel)

	// Selected text markdown
	widgetSelectedText := func(gtx layout.Context) layout.Dimensions {
		if renderSelectedMDData == "" {
			return layout.Spacer{Height: unit.Dp(0)}.Layout(gtx)
		}
		selectedSpans := []richtext.SpanStyle{}
		if s, err := widget.GlobalState.MarkdownRenderer.Render([]byte(renderSelectedMDData)); err == nil {
			selectedSpans = s
		}
		return richtext.Text(&widget.richtextSelected, widget.GlobalState.Theme.Shaper, selectedSpans...).Layout(gtx)
	}

	// Buttons split
	widgetButtonsSplit := func(gtx layout.Context) layout.Dimensions {
		if renderSelectedLabel == "" {
			return layout.Spacer{Height: unit.Dp(0)}.Layout(gtx)
		}
		gtx.Constraints.Min.Y = gtx.Dp(unit.Dp(50))
		gtx.Constraints.Max.Y = gtx.Constraints.Min.Y
		return layout.Flex{}.Layout(gtx, layout.Rigid(dimensionsToWidget(layout.Flex{}.Layout(
			gtx,
			layout.Flexed(0.2, func(gtx layout.Context) layout.Dimensions {
				return material.Button(widget.GlobalState.Theme, &widget.clickableSetFrom, "Set as From").Layout(gtx)
			}),
			layout.Flexed(0.01, layout.Spacer{Width: unit.Dp(2)}.Layout),
			layout.Flexed(0.2, func(gtx layout.Context) layout.Dimensions {
				return material.Button(widget.GlobalState.Theme, &widget.clickableSetTo, "Set as To").Layout(gtx)
			}),
			layout.Flexed(0.01, layout.Spacer{Width: unit.Dp(2)}.Layout),
			layout.Flexed(0.2, func(gtx layout.Context) layout.Dimensions {
				return material.Button(widget.GlobalState.Theme, &widget.clickableSetFav, "Favorite").Layout(gtx)
			}),
		))))
	}

	widgets := []layout.Widget{
		widgetSelectedText,
		widgetButtonsSplit,
	}
	return widgets
}

func (widget *PageStops) determineSelectedData() (string, string) {
	var (
		renderSelectedLabel  string
		renderSelectedMDData string
	)

	selectListSelected := widget.stopsSelectList.Selected
	selectListItems := widget.stopsSelectListItems
	if selectListSelected <= len(selectListItems)-1 {
		if label, coords, err := parseLatLon(selectListItems[selectListSelected]); err != nil {
			log.Println(err)
		} else {
			renderSelectedLabel = selectListItems[selectListSelected]
			renderSelectedMDData = fmt.Sprintf(
				"**Selected:**\n\n%s ([%.5f, %.5f](geo:%.5f,%.5f))\n\n",
				label, coords[0], coords[1], coords[0], coords[1],
			)

			if !widget.GlobalState.Save.CfgDisableSchedules {
				stopIDSelectedMatch := regexp.MustCompile(`StopID:\s*([^/]+)\s*/\s*Latlon:`).FindStringSubmatch(renderSelectedLabel)
				if len(stopIDSelectedMatch) > 1 {
					stopDepartures, err := integrations.MobrouteStoptimesForStopMarkdowndata(
						widget.GlobalState.MobrouteRuntime,
						widget.GlobalState.Save.FeedIDsEnabled,
						strings.TrimSpace(stopIDSelectedMatch[1]),
					)
					if err != nil {
						log.Println(err)
					} else {
						for _, s := range *stopDepartures {
							renderSelectedMDData = fmt.Sprintf("%s\n-  %v", renderSelectedMDData, s)
						}
					}
				}
			}
		}
	}

	return renderSelectedLabel, renderSelectedMDData
}

func (widget *PageStops) eventhandlersSearchWidgetsSelected(gtx layout.Context, renderSelectedLabel string) {
	utillinkhandler.HandleLinkEvents(widget.GlobalState.Window, &widget.richtextSelected, gtx)

	if widget.clickableSetFrom.Clicked(gtx) {
		if renderSelectedLabel != "" {
			setFromTo(widget, "from", renderSelectedLabel)
		}
	}
	if widget.clickableSetTo.Clicked(gtx) {
		if renderSelectedLabel != "" {
			setFromTo(widget, "to", renderSelectedLabel)
		}
	}
	if widget.clickableSetFav.Clicked(gtx) {
		if renderSelectedLabel != "" {
			exists := false
			for _, favorite := range widget.GlobalState.Save.Favorites {
				if favorite == renderSelectedLabel {
					exists = true
				}
			}
			if !exists {
				widget.GlobalState.Save.Favorites = append(widget.GlobalState.Save.Favorites, renderSelectedLabel)
			}

		}
	}
}
