package ui

import (
	"bytes"
	"fmt"
	"log"
	"os"
	"time"

	"git.sr.ht/~mil/mobroute"
	"git.sr.ht/~mil/transito/src/assets"
	"git.sr.ht/~mil/transito/src/globalstate"
	"git.sr.ht/~mil/transito/src/uipages/pageconfig/integrations"
	"git.sr.ht/~mil/transito/src/util/utilfuncs"
	"git.sr.ht/~mil/transito/src/util/utilplatform"
)

func (ui *UI) ServiceQueueload(queueload *globalstate.AppQueueload) {
	if queueload.RuntimeInitialize {
		go ui.ServiceLoadMobrouteRuntime()
	} else if queueload.FeedIDLoadAndCompute != nil {
		go ui.ServiceLoadComputeFeeds([]int{*queueload.FeedIDLoadAndCompute}, false, queueload.ReturnToTab)
	} else if queueload.FeedIDUpdate != nil {
		go ui.ServiceLoadComputeFeeds([]int{*queueload.FeedIDUpdate}, true, queueload.ReturnToTab)
	}
}

func (ui *UI) ServiceLoadMobrouteRuntime() {
	ui.GlobalState.TabSelected = "Load"
	var (
		now               = time.Now()
		mobrouteLogBuffer = bytes.Buffer{}
	)
	ui.GlobalState.PendingLoadInfo.TimerStart = &now
	ui.GlobalState.PendingLoadInfo.Message = "Mobroute Runtime Initialization"
	ui.GlobalState.PendingLoadInfo.IsError = false
	go ui.RefreshUIWhileOnLoadTab()

	mdbCSVFile := fmt.Sprintf("%s/%s", utilplatform.DataDir(), "mdb_csv.csv")
	if err := os.WriteFile(mdbCSVFile, []byte(assets.MobilityDBCSVData), 0755); err != nil {
		ui.DisplayLoadingError(fmt.Sprintf("Error Initializing Mobroute Runtime mdb csv: %v", err))
	} else if mrRuntime, err := mobroute.RTInitialize(&mobroute.MobrouteRuntimeConfig{
		Logger: log.New(&mobrouteLogBuffer, "", log.Ltime),
		MobsqlRuntimeConfig: mobroute.MobsqlRuntimeConfig{
			MobilityDBCatalogCSVURI: fmt.Sprintf("file://%s", mdbCSVFile),
		},
	}); err != nil {
		ui.DisplayLoadingError(fmt.Sprintf(
			`Error Initializing Mobroute Runtime

Note: If this message appears after you updated between versions, the DB schema may have changed and you should clear cache or uninstall/reinstall

Raw error:
%v`, err,
		))
	} else {
		ui.GlobalState.MobrouteRuntime = mrRuntime
		ui.GlobalState.MobrouteLog = &mobrouteLogBuffer
		if len(ui.GlobalState.Save.FeedIDsEnabled) > 0 {
			ui.GlobalState.TabSelected = "Route"
		} else {
			ui.GlobalState.TabSelected = "Config"
		}
		ui.GlobalState.Window.Invalidate()
	}
}

func (ui *UI) ServiceLoadComputeFeeds(gtfsFeedIDs []int, purgeUpdate bool, returnToTab string) {
	ui.GlobalState.TabSelected = "Load"
	now := time.Now()
	ui.GlobalState.PendingLoadInfo.TimerStart = &now
	ui.GlobalState.PendingLoadInfo.SubMessage = "(Be patient, 1-time process per GTFS feed)"
	ui.GlobalState.MobrouteLog.Reset()
	go ui.RefreshUIWhileOnLoadTab()

	// Purge GTFS data
	if purgeUpdate {
		ui.GlobalState.PendingLoadInfo.Message = fmt.Sprintf("Purge GTFS data for %v", gtfsFeedIDs)
		ui.GlobalState.Window.Invalidate()
		if _, err := mobroute.RTDatabase(
			ui.GlobalState.MobrouteRuntime,
			&mobroute.DatabaseParams{Op: "purge_all", FeedIDs: gtfsFeedIDs},
		); err != nil {
			ui.DisplayLoadingError(fmt.Sprintf("Error puring: %v", err))
			return
		}
	}

	// Load GTFS stage
	ui.GlobalState.PendingLoadInfo.Message = fmt.Sprintf("Load GTFS data for %v", gtfsFeedIDs)
	ui.GlobalState.Window.Invalidate()
	if _, err := mobroute.RTDatabase(
		ui.GlobalState.MobrouteRuntime,
		&mobroute.DatabaseParams{
			Op:                    "loadmdbgtfs",
			FeedIDs:               gtfsFeedIDs,
			LoadMDBGTFSUpdate:     true,
			LoadMDBGTFSDirectURLs: ui.GlobalState.Save.CfgUseAgencyGTFSURLs,
		},
	); err != nil {
		ui.DisplayLoadingError(fmt.Sprintf("Error loading: %v", err))
		return
	}

	// Compute stage
	ui.GlobalState.PendingLoadInfo.Message = fmt.Sprintf("Compute Routing Optimized Tables for %v", gtfsFeedIDs)
	ui.GlobalState.Window.Invalidate()
	if _, err := mobroute.RTDatabase(
		ui.GlobalState.MobrouteRuntime,
		&mobroute.DatabaseParams{Op: "compute", FeedIDs: gtfsFeedIDs},
	); err != nil {
		ui.DisplayLoadingError(fmt.Sprintf("Error computing: %v", err))
		return
	}

	ui.GlobalState.Save.FeedIDsEnabled = utilfuncs.UniqueInts(append(ui.GlobalState.Save.FeedIDsEnabled, gtfsFeedIDs...))

	// Crossfeed Compute stage
	if ui.GlobalState.Save.CfgCrossfeedCompute && len(ui.GlobalState.Save.FeedIDsEnabled) > 0 {
		ui.GlobalState.PendingLoadInfo.Message = fmt.Sprintf("Compute Crossfeed Transfers for %v", gtfsFeedIDs)
		ui.GlobalState.Window.Invalidate()
		if _, err := mobroute.RTDatabase(
			ui.GlobalState.MobrouteRuntime,
			&mobroute.DatabaseParams{Op: "compute", FeedIDs: ui.GlobalState.Save.FeedIDsEnabled},
		); err != nil {
			ui.DisplayLoadingError(fmt.Sprintf("Error computing crossfeed transfers: %v", err))
			return
		}
	}

	ui.GlobalState.TabSelected = returnToTab
	integrations.CacheTabledataSearch = map[string]*[]string{}
	ui.GlobalState.Window.Invalidate()
}

func (ui *UI) DisplayLoadingError(err string) {
	log.Printf("Error: %v", err)
	ui.GlobalState.PendingLoadInfo.Message = err
	ui.GlobalState.PendingLoadInfo.IsError = true
	ui.GlobalState.PendingLoadInfo.SubMessage = "(Please file a bug report & force-quit application)"
	ui.GlobalState.Window.Invalidate()
}

func (ui *UI) RefreshUIWhileOnLoadTab() {
	for {
		if ui.GlobalState.TabSelected != "Load" {
			break
		} else {
			time.Sleep(time.Second / 10)
			ui.GlobalState.Window.Invalidate()
		}
	}
}
