package integrations

import (
	"fmt"
	"strings"
	"time"

	"git.sr.ht/~mil/mobroute"
)

var (
	MaxNStops              = 200
	cacheTabledataStops    = map[string]*[]string{}
	cacheTabledataSchedule = map[string]*[]string{}
)

func cacheKeyStops(feedIDs []int, searchTerm string) string {
	return fmt.Sprintf("%v_%s", feedIDs, searchTerm)
}

func cacheKeyStoptimes(feedIDs []int, stopID string, now *time.Time) string {
	minutes := now.Unix() / 60
	return fmt.Sprintf("%v_%s_%d", feedIDs, stopID, minutes)
}

func MobrouteStopsTableData(
	mobrouteRuntime *mobroute.MobrouteRuntime,
	feedIDs []int,
	searchTerm string,
) (*[]string, error) {
	key := cacheKeyStops(feedIDs, searchTerm)

	if _, exists := cacheTabledataStops[key]; exists {
		return cacheTabledataStops[key], nil
	}

	if stopsResponse, err := mobroute.RTStops(mobrouteRuntime, &mobroute.StopsParams{
		FeedIDs: feedIDs,
		NLimit:  MaxNStops,
		Glob: searchTerm,
	}); err != nil {
		return nil, err
	} else {
		ret := []string{}
		for _, s := range *stopsResponse.Stops {
			ret = append(ret, fmt.Sprintf("%s - StopID: %s / Latlon: %.05f %.05f", s.StopName, s.StopSIDU, s.StopLat, s.StopLon))
		}
		cacheTabledataStops[key] = &ret
		return &ret, nil
	}
}

func MobrouteStoptimesForStopMarkdowndata(
	mobrouteRuntime *mobroute.MobrouteRuntime,
	feedIDs []int,
	stopID string,
) (*[]string, error) {
	now := time.Now()
	key := cacheKeyStoptimes(feedIDs, stopID, &now)

	if _, exists := cacheTabledataSchedule[key]; exists {
		return cacheTabledataSchedule[key], nil
	}

	if stoptimesResponse, err := mobroute.RTSchedule(
		mobrouteRuntime,
		&mobroute.ScheduleParams{
			FeedIDs:    feedIDs,
			StopIDs:    []string{stopID},
			MaxSeconds: 60 * 60, // Next hr
		},
	); err != nil {
		return nil, err
	} else {
		ret := []string{}
		for i, st := range *stoptimesResponse.Schedule {
			if i > 2 {
				// TODO: limit nresults in mobroute.RTStoptimes lib call?
				break
			}
			now := time.Now()
			delta := (*st.DepartureTime).UTC().Sub(now.UTC())

			var route string
			if st.RouteShortName != nil && st.RouteLongName != nil {
				route = fmt.Sprintf("%s / %s", *st.RouteShortName, *st.RouteLongName)
			} else if st.RouteShortName != nil {
				route = *st.RouteShortName
			} else if st.RouteLongName != nil {
				route = *st.RouteLongName
			}
			if st.TripHeadsign != nil {
				route = fmt.Sprintf("%s _toward %s_", route, strings.Trim(*st.TripHeadsign, "\n"))
			}
			str := fmt.Sprintf("%d min (at **%s**): %s", int(delta/time.Minute), now.Add(delta).Format("15:04"), route)
			ret = append(ret, str)
		}
		cacheTabledataSchedule[key] = &ret
		return &ret, nil
	}
}
