package pageroute

import (
	"bytes"
	"encoding/json"
	"io"
	"log"
	"time"

	"gioui.org/font/gofont"
	"gioui.org/io/clipboard"
	"gioui.org/io/key"
	"gioui.org/layout"
	"gioui.org/text"
	"gioui.org/unit"
	"gioui.org/widget"
	"gioui.org/widget/material"
	"gioui.org/x/component"
	"gioui.org/x/richtext"
	"git.sr.ht/~mil/mobroute"
	"git.sr.ht/~mil/transito/src/components/buttonrow"
	"git.sr.ht/~mil/transito/src/globalstate"
	"git.sr.ht/~mil/transito/src/util/utiltheme"
	"github.com/inkeliz/giohyperlink"
)

type PageRoute struct {
	GlobalState *globalstate.GlobalState
	list        widget.List

	richtextFrom       richtext.InteractiveText
	richtextTo         richtext.InteractiveText
	richtextResultArea richtext.InteractiveText

	ResultMRRoute            *mobroute.RouteResponse
	ResultMRRouteErrorDbglog bool
	ResultMRRouteError       error

	buttonRow buttonrow.ButtonRow

	showParams bool

	// Params
	checkboxTransfersFeed       widget.Bool
	checkboxTransfersImplicit   widget.Bool
	checkboxTransfersGenerated  widget.Bool
	textfieldMaxWalkMinutes     component.TextField
	textfieldMaxTripMinutes     component.TextField
	textfieldMaxTransferMinutes component.TextField
	textfieldMinTransferMinutes component.TextField
	textfieldWalkspeedKmHr      component.TextField

	clickableRoute  widget.Clickable
	clickableParams widget.Clickable

	showLog              bool
	clickableShowHideLog widget.Clickable

	TimerStage         string
	TimerStartMobroute *time.Time
}

func Initialize(u *globalstate.GlobalState) *PageRoute {
	page := PageRoute{
		GlobalState:        u,
		list:               widget.List{List: layout.List{Axis: layout.Vertical}},
		ResultMRRoute:      nil,
		ResultMRRouteError: nil,
		TimerStartMobroute: nil,

		buttonRow: buttonrow.ButtonRow{
			Theme: u.Theme,
			Buttons: []buttonrow.Button{
				{Label: "Overview", CanActivate: true, Themecolor: utiltheme.ThemecolorFgBlackBgGray},
				{Label: "Debug", CanActivate: true, Themecolor: utiltheme.ThemecolorFgBlackBgGray},
				{Label: "Clear", CanActivate: false, Themecolor: utiltheme.ThemecolorFgBlackBgGray},
				{Align: buttonrow.Align(buttonrow.AlignRight), Label: "Map", CanActivate: false, Themecolor: utiltheme.ThemecolorFgBlackBgGreen},
			},
			ActiveButtonLabel:      "Overview",
			ActiveButtonThemecolor: utiltheme.ThemecolorFgBlackBgBlue,
		},

		textfieldMaxWalkMinutes:     component.TextField{Editor: widget.Editor{Filter: "1234567890", InputHint: key.HintNumeric}},
		textfieldMaxTripMinutes:     component.TextField{Editor: widget.Editor{Filter: "1234567890", InputHint: key.HintNumeric}},
		textfieldMaxTransferMinutes: component.TextField{Editor: widget.Editor{Filter: "1234567890", InputHint: key.HintNumeric}},
		textfieldMinTransferMinutes: component.TextField{Editor: widget.Editor{Filter: "1234567890", InputHint: key.HintNumeric}},
		textfieldWalkspeedKmHr:      component.TextField{Editor: widget.Editor{Filter: "1234567890.", InputHint: key.HintNumeric}},
	}

	// Defaults
	page.textfieldMaxWalkMinutes.SetText("20")
	page.textfieldMaxTripMinutes.SetText("240")
	page.textfieldMinTransferMinutes.SetText("3")
	page.textfieldMaxTransferMinutes.SetText("40")
	page.textfieldWalkspeedKmHr.SetText("3.5")
	page.checkboxTransfersFeed.Value = true
	page.checkboxTransfersImplicit.Value = true
	for _, tf := range []*component.TextField{
		&page.textfieldMaxWalkMinutes,
		&page.textfieldMaxTripMinutes,
		&page.textfieldMinTransferMinutes,
		&page.textfieldMaxTransferMinutes,
		&page.textfieldWalkspeedKmHr,
	} {
		tf.MoveCaret(9999, 99999)
	}

	pageptr := &page
	// Clipboard copy
	page.buttonRow.Buttons[1].Callback = func(gtx layout.Context) {
		if textBytes, err := json.Marshal(page.ResultMRRoute.Request); err == nil {
			gtx.Execute(clipboard.WriteCmd{Data: io.NopCloser(bytes.NewReader(textBytes))})
		} else {
			log.Println("Error marshaling MR request to JSON: ", err)
		}
	}
	// Clear
	page.buttonRow.Buttons[2].Callback = func(gtx layout.Context) {
		pageptr.ResultMRRoute = nil
	}
	// Open map URL for map button
	page.buttonRow.Buttons[3].Callback = func(gtx layout.Context) {
		if err := giohyperlink.Open(*pageptr.ResultMRRoute.MapURL); err != nil {
			log.Println("Error opening map URL:", err)
		}
	}
	page.GlobalState.Theme.Shaper = text.NewShaper(text.WithCollection(gofont.Collection()))
	return &page
}

func (page *PageRoute) Render(gtx layout.Context) layout.Dimensions {
	var widgetsScroll = []layout.Widget{}
	if len(page.GlobalState.Save.FeedIDsEnabled) == 0 {
		widgetsScroll = append(
			widgetsScroll,
			material.Body1(page.GlobalState.Theme, "Must enable atleast 1 GTFS source to route").Layout,
		)
	} else {
		widgetsScroll = append(widgetsScroll, page.widgetRowSplitToFrom, page.widgetRowActionButtons)
		if page.showParams {
			widgetsScroll = append(widgetsScroll, page.widgetRowParams(gtx)...)
		}
		widgetsScroll = append(widgetsScroll, page.widgetRowResultMarkdown(gtx)...)
		widgetsScroll = append(widgetsScroll, page.widgetRowLogToggle(gtx)...)
	}

	return material.List(page.GlobalState.Theme, &page.list).Layout(
		gtx,
		len(widgetsScroll),
		func(gtx layout.Context, i int) layout.Dimensions {
			return layout.UniformInset(unit.Dp(16)).Layout(gtx, widgetsScroll[i])
		},
	)
}
