package mrtomarkdown

import (
	"fmt"

	"git.sr.ht/~mil/mobroute"
)

const TimeFormat = "15:04"

func MROverviewFormatMarkdown(mrResponse *mobroute.RouteResponse) string {
	steps := ""
	for legI, leg := range *mrResponse.RouteLegs {
		stepText := legToStepText(&leg, legI)
		steps = fmt.Sprintf(
			"%s- %s\n",
			steps,
			stepText,
		)
	}
	routeSeconds, routeNTransfers := routeMetadataSecondsAndTransfers(mrResponse.RouteLegs)

	return fmt.Sprintf(
		"###### **Steps** ‑ %dm, %dt\n%s",
		routeSeconds/60,
		routeNTransfers,
		steps,
	)
}

func routeMetadataSecondsAndTransfers(legs *[]mobroute.RouteLeg) (uint, uint) {
	var (
		secondsTotal   uint = 0
		transfersTotal uint = 0
	)
	for _, leg := range *legs {
		secondsTotal = secondsTotal + uint(leg.LegEndTime.Sub(leg.LegBeginTime).Seconds())
		if leg.LegType == "transfer" {
			transfersTotal = transfersTotal + 1
		}
	}
	return secondsTotal, transfersTotal
}

func legToStepText(leg *mobroute.RouteLeg, legI int) string {
	if leg.LegType == "walk" {
		return fmt.Sprintf(
			"%s: **Walk** :: [%s](%s) → [%s](%s) (%s)",
			leg.LegBeginTime.Format(TimeFormat),

			*leg.WalkFrom,
			latLonToGeoURI(leg.LegFromCoords),

			*leg.WalkTo,
			latLonToGeoURI(leg.LegToCoords),

			leg.LegDuration,
		)
	} else if leg.LegType == "trip" {
		return fmt.Sprintf(
			"%s: **Trip** :: [%s](%s) → [%s](%s) (%s) (%d stops / %s)",
			leg.LegBeginTime.Format(TimeFormat),

			*leg.TripFrom,
			latLonToGeoURI(leg.LegFromCoords),

			*leg.TripTo,
			latLonToGeoURI(leg.LegToCoords),

			routeText(leg.TripRoute, leg.TripRouteHeadsign),

			len(*leg.TripStops),
			leg.LegDuration,
		)
	} else if leg.LegType == "transfer" {
		return fmt.Sprintf(
			"%s: **Transfer** :: [%s](%s) → [%s](%s) (%s)",
			leg.LegBeginTime.Format(TimeFormat),

			*leg.TransferFrom,
			latLonToGeoURI(leg.LegFromCoords),

			*leg.TransferTo,
			latLonToGeoURI(leg.LegToCoords),

			leg.LegDuration,
		)
	}

	return ""
}

func latLonToGeoURI(latLon [2]float64) string {
	return fmt.Sprintf("geo:%.5f,%.5f", latLon[0], latLon[1])
}

func routeText(route, headsign *string) string {
	if route != nil && headsign != nil {
		return fmt.Sprintf("via %s toward %s", *route, *headsign)
	} else if route != nil && headsign == nil {
		return *route
	} else {
		return "Unknown"
	}
}
