package integrations

import (
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"strconv"
	"strings"
)

type NominatimAPIResult struct {
	DisplayName string `json:"display_name"`
	Name        string `json:"name"`
	Lat         string `json:"lat"`
	Lon         string `json:"lon"`
}

func FetchNominatimWithBbox(searchQuery string, bbox *[4]float64) (*[]NominatimAPIResult, error) {
	var (
		httpClient        = &http.Client{}
		req               *http.Request
		nominatimEndpoint = "https://nominatim.openstreetmap.org/search"
		results           []NominatimAPIResult
	)

	if parsedUrl, err := url.Parse(nominatimEndpoint); err != nil {
		return nil, err
	} else {
		params := url.Values{}
		params.Add("q", searchQuery)
		if bbox != nil {
			params.Add("bounded", "1")
			params.Add("viewbox", strings.Join(
				[]string{
					strconv.FormatFloat(bbox[1], 'f', -1, 64),
					strconv.FormatFloat(bbox[0], 'f', -1, 64),
					strconv.FormatFloat(bbox[3], 'f', -1, 64),
					strconv.FormatFloat(bbox[2], 'f', -1, 64),
				},
				",",
			))
		}
		params.Add("format", "json")
		parsedUrl.RawQuery = params.Encode()

		if req, err = http.NewRequest("GET", parsedUrl.String(), nil); err != nil {
			return nil, err
		}
		req.Header.Set("User-Agent", "Transito/0.8.4")

		if resp, err := httpClient.Do(req); err != nil {
			return nil, err
		} else if body, err := io.ReadAll(resp.Body); err != nil {
			return nil, err
		} else if resp.StatusCode != http.StatusOK {
			return nil, fmt.Errorf("Errorcode %d from Nominatim / body: %s", resp.StatusCode, body)
		} else if err := json.Unmarshal(body, &results); err != nil {
			return nil, err
		} else {
			return &results, nil
		}
	}
}
