package integrations

import (
	"errors"
	"fmt"
	"sort"
	"strings"
	"time"

	"git.sr.ht/~mil/mobsql"
	"git.sr.ht/~mil/transito/src/util/utilfuncs"
)

var CacheTabledataSearch = map[string]*[]string{}

func cacheKeyInts(ints []int, markdownify, offline bool) string {
	sort.Ints(ints)
	return fmt.Sprintf(
		"cachekey_feedids_%s_%b_%b",
		strings.Join(utilfuncs.Strvals(ints), "_"), markdownify, offline,
	)
}

func MobsqlIntegrationGetTableDataForFeedIDs(
	mobsqlRuntime *mobsql.MobsqlRuntime,
	searchText *string,
	feedIDs *[]int,
	markdownify, offline bool,
) (*[]string, error) {
	var filter *mobsql.FeedsearchFilter
	if searchText != nil {
		if *searchText == "" {
			return &[]string{}, nil
		} else if cachedReturn, exists := CacheTabledataSearch[*searchText]; exists {
			return cachedReturn, nil
		} else {
			filter = &mobsql.FeedsearchFilter{Glob: *searchText, Status: "active"}
		}
	} else if feedIDs != nil {
		if len(*feedIDs) == 0 {
			return &[]string{}, nil
		} else if cachedReturn, exists := CacheTabledataSearch[cacheKeyInts(*feedIDs, markdownify, offline)]; exists {
			return cachedReturn, nil
		} else {
			filter = &mobsql.FeedsearchFilter{FeedID: *feedIDs, Status: "active"}
		}
	} else {
		return nil, errors.New("Must pass searchtext or feedIDs as nonnil")
	}

	if feedIDs, err := mobsql.FeedsearchFilterToFeedIDs(mobsqlRuntime, filter); err != nil {
		return nil, err
	} else if searchResults, err := mobsql.FeedStatus(mobsqlRuntime, feedIDs); err != nil {
		return nil, err
	} else {
		if len(searchResults) > 10 {
			searchResults = (searchResults)[0:10]
		}
		returnItems := []string{}
		for _, r := range searchResults {
			returnItems = append(returnItems, constructFeedInfoLine(&r, markdownify, offline))
		}
		if searchText != nil {
			CacheTabledataSearch[*searchText] = &returnItems
		} else if feedIDs != nil {
			CacheTabledataSearch[cacheKeyInts(feedIDs, markdownify, offline)] = &returnItems
		}
		return &returnItems, nil
	}
}

func constructFeedInfoLine(r *mobsql.FeedStatusInfo, markdownify, offline bool) string {
	locationField := func() string {
		nonNilLocationFields := []string{}
		for _, l := range []*string{
			r.LocationCountry, r.Provider, r.Name, r.LocationMunicipality, r.LocationMunicipality, r.LocationSubdivision,
		} {
			if l != nil {
				nonNilLocationFields = append(nonNilLocationFields, *l)
			}
		}
		return strings.Join(nonNilLocationFields, " / ")
	}()

	var appendString string
	if markdownify {
		appendString = fmt.Sprintf("**%d**: %s", r.FeedID, strings.TrimSpace(utilfuncs.Truncate(locationField, 50)))
		if r.MinDate != nil && r.MaxDate != nil {
			appendString = fmt.Sprintf("%s (*%s → %s*)", appendString, prettifyDate(*r.MinDate), prettifyDate(*r.MaxDate))
		}
		if r.TimestampComputed != nil {
			appendString = fmt.Sprintf("%s (Updated: *%s*)", appendString, *r.TimestampComputed)
		}
		if r.URLLicense != nil {
			appendString = fmt.Sprintf("%s | [License](%s)", appendString, *r.URLLicense)
		}
		if !offline {
			appendString = fmt.Sprintf("%s | [Update](transito://feedid_update:%d)", appendString, r.FeedID)
		}
		appendString = fmt.Sprintf("%s | [Remove](transito://feedid_remove:%d)", appendString, r.FeedID)
	} else {
		appendString = fmt.Sprintf("%d: %s", r.FeedID, utilfuncs.Truncate(locationField, 50))
		if r.TimestampComputed != nil {
			appendString = fmt.Sprintf("%s (cached: %v)", appendString, *r.TimestampComputed)
		}
	}
	return appendString
}

func prettifyDate(yyyymmdd string) string {
	if p, err := time.Parse("20060102", yyyymmdd); err != nil {
		return yyyymmdd
	} else {
		return p.Format("2006-01-02")
	}
}
