package ui

import (
	"image"
	"image/color"
	"math"
	"strings"
	"time"

	"gioui.org/layout"
	"gioui.org/op/clip"
	"gioui.org/op/paint"
	"gioui.org/unit"
	"gioui.org/widget"
	"gioui.org/widget/material"
	"git.sr.ht/~mil/transito/src/components/divider"
	"git.sr.ht/~mil/transito/src/components/headerbar"
	"git.sr.ht/~mil/transito/src/util/utilfuncs"
)

func (ui *UI) RenderTabbedUI(gtx layout.Context) layout.Dimensions {
	tabButtons := layout.Rigid(func(gtx layout.Context) layout.Dimensions {
		activeTabs := []*UITab{}
		for i, _ := range ui.Tabs {
			// Only show load tab by itself when active
			if ui.Tabs[i].Tabname == "Load" && ui.GlobalState.TabSelected == "Load" {
				activeTabs = []*UITab{&ui.Tabs[i]}
				break
			}

			// For all tabs that are not load, dont show the load tab
			if ui.Tabs[i].Tabname == "Load" {
				continue
			}

			// In offline mode, hide POI tab
			if ui.Tabs[i].Tabname == "POI" && ui.GlobalState.Save.CfgOfflineMode {
				continue
			}

			// If no faves, hide fave ab
			if ui.Tabs[i].Tabname == "Fave" && len(ui.GlobalState.Save.Favorites) == 0 {
				continue
			}

			// If config tab, no feedids configured, hide all other tabs
			if ui.Tabs[i].Tabname != "Config" && len(ui.GlobalState.Save.FeedIDsEnabled) == 0 {
				continue
			}
			activeTabs = append(activeTabs, &ui.Tabs[i])
		}

		return ui.tabContentList.Layout(gtx, len(activeTabs), func(gtx layout.Context, tabIdx int) layout.Dimensions {
			var labelString string = activeTabs[tabIdx].Tabname
			var tabButton *widget.Clickable = &activeTabs[tabIdx].Clickable

			if tabButton.Clicked(gtx) {
				ui.GlobalState.TabSelected = activeTabs[tabIdx].Tabname
			}

			var tabWidth int
			var tabHeight int
			return layout.Stack{Alignment: layout.S}.Layout(gtx,
				// Bg
				layout.Stacked(func(gtx layout.Context) layout.Dimensions {
					if ui.GlobalState.TabSelected != activeTabs[tabIdx].Tabname {
						return layout.Dimensions{}
					}
					dims := material.Clickable(gtx, tabButton, func(gtx layout.Context) layout.Dimensions {
						return layout.UniformInset(unit.Dp(12)).Layout(gtx,
							material.H6(ui.GlobalState.Theme, labelString).Layout,
						)
					})
					tabRect := image.Rect(1, 0, dims.Size.X, gtx.Dp(unit.Dp(dims.Size.Y)))
					paint.FillShape(gtx.Ops, color.NRGBA{R: 0xf0, G: 0xf0, B: 0xf0, A: 0xf0}, clip.Rect(tabRect).Op())
					return layout.Dimensions{Size: image.Point{X: dims.Size.X, Y: dims.Size.Y}}
				}),
				// Draw tab font
				layout.Stacked(func(gtx layout.Context) layout.Dimensions {
					dims := material.Clickable(gtx, tabButton, func(gtx layout.Context) layout.Dimensions {
						return layout.UniformInset(unit.Dp(12)).Layout(gtx,
							material.H6(ui.GlobalState.Theme, labelString).Layout,
						)
					})
					tabWidth = dims.Size.X
					tabHeight = dims.Size.Y
					return dims
				}),
				// Border between tabs
				layout.Stacked(func(gtx layout.Context) layout.Dimensions {
					tabRect := image.Rect(0, 0, 1, gtx.Dp(unit.Dp(tabHeight)))
					paint.FillShape(gtx.Ops, color.NRGBA{R: 0xcf, G: 0xcf, B: 0xcf, A: 0xFF}, clip.Rect(tabRect).Op())
					return layout.Dimensions{Size: image.Point{X: -tabWidth, Y: tabHeight}}
				}),
			)
		})
	})

	// Tab content
	tabContents := layout.Flexed(1, func(gtx layout.Context) layout.Dimensions {
		for _, t := range ui.Tabs {
			if t.Tabname == ui.GlobalState.TabSelected {
				return t.Page.Render(gtx)
			}
		}
		return layout.Center.Layout(gtx, material.H1(ui.GlobalState.Theme, "Unknown").Layout)
	})

	headerBar := layout.Rigid(func(gtx layout.Context) layout.Dimensions {
		var isLoading = ui.GlobalState.TabSelected == "Load"
		if isLoading && ui.GlobalState.PendingLoadInfo.TimerStart != nil && !ui.GlobalState.PendingLoadInfo.IsError {
			diff := time.Now().Sub(*ui.GlobalState.PendingLoadInfo.TimerStart).Seconds()
			var progressPercent float32
			progressPercent = float32(math.Mod(diff, 1))
			return headerbar.Headerbar(
				gtx, ui.GlobalState.Theme, progressPercent, "Transito", "",
			)
		} else if isLoading && ui.GlobalState.PendingLoadInfo.IsError {
			return headerbar.Headerbar(gtx, ui.GlobalState.Theme, -1, "Transito", "")
		} else {
			return headerbar.Headerbar(
				gtx, ui.GlobalState.Theme, -1, "Transito", strings.Join(utilfuncs.Strvals(ui.GlobalState.Save.FeedIDsEnabled), ", "),
			)
		}
	})

	return layout.Flex{Axis: layout.Vertical}.Layout(
		gtx,
		headerBar,
		layout.Rigid(divider.Divider(ui.GlobalState.Theme).Layout),
		tabButtons,
		layout.Rigid(divider.Divider(ui.GlobalState.Theme).Layout),
		tabContents,
	)
}
