package globalstate

import (
	"encoding/json"
	"fmt"
	"log"
	"os"
	"time"

	"git.sr.ht/~mil/transito/src/util/utilplatform"
)

var (
	savestateFileCached = ""
	saveIntervalSeconds = 5.0
	lastSaveTime        time.Time
)

type GlobalStateSave struct {
	FeedIDsEnabled        []int    `json:"feed_ids_enabled"`
	Favorites             []string `json:"favorites"`
	CfgDisableSchedules   bool     `json:"cfg_disable_schedules"`
	CfgOfflineMode        bool     `json:"cfg_offline_mode"`
	CfgCrossfeedCompute   bool     `json:"cfg_crossfeed_compute"`
}

// savestateFile determines the file used for storing the savestate
// It wraps gioui.org/app's DataDir() & creates the transito/ folder.
//
// Multiple calls will cache the result in savestateFileCached which
// will be directly returned
func savestateFile() string {
	if savestateFileCached == "" {
		savestateFileCached = fmt.Sprintf("%s/%s", utilplatform.DataDir(), "transito_savestate.json")
		log.Printf("Set savestatefile: %v", savestateFileCached)
	}
	return savestateFileCached
}

func FileSave(s *GlobalStateSave) error {
	if bytes, err := json.MarshalIndent(s, "", "  "); err != nil {
		return err
	} else if err := os.WriteFile(savestateFile(), bytes, 0644); err != nil {
		return err
	}
	return nil
}

func FileLoad() (*GlobalStateSave, error) {
	var s GlobalStateSave

	if dat, err := os.ReadFile(savestateFile()); err != nil {
		return nil, err
	} else if err := json.Unmarshal(dat, &s); err != nil {
		return nil, err
	} else {
		return &s, nil
	}
}

func FileLoadWarnError() GlobalStateSave {
	if s, err := FileLoad(); err != nil {
		log.Printf("Problem loading from savestate file: %v", err)
		return GlobalStateSave{}
	} else {
		return *s
	}
}

// IntervalSaveWatch is expected to be called from the main rendering loop
// of the application and just deltas lastSaveTime, if the delta exceeds
// the saveIntervalSeconds threshold, FileSave() is rerun
func IntervalSaveWatch(s *GlobalStateSave) error {
	now := time.Now()
	if now.Sub(lastSaveTime).Seconds() > saveIntervalSeconds {
		lastSaveTime = now
		return FileSave(s)
	}
	return nil
}
