// Copyright (c) 2024, the Dart project authors. Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

import 'package:test/test.dart';
import 'package:test_reflective_loader/test_reflective_loader.dart';

import '../../dart/resolution/node_text_expectations.dart';
import '../elements_base.dart';

main() {
  defineReflectiveSuite(() {
    defineReflectiveTests(ClassElementTest_keepLinking);
    defineReflectiveTests(ClassElementTest_fromBytes);
    defineReflectiveTests(ClassElementTest_augmentation_keepLinking);
    defineReflectiveTests(ClassElementTest_augmentation_fromBytes);
    defineReflectiveTests(UpdateNodeTextExpectations);
  });
}

abstract class ClassElementTest extends ElementsBaseTest {
  test_class_abstract() async {
    var library = await buildLibrary('abstract class C {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        abstract class C @15
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @15
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    abstract class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_base() async {
    var library = await buildLibrary('base class C {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        base class C @11
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @11
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    base class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_const() async {
    var library = await buildLibrary('class C { const C(); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            const @16
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 16
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_const_external() async {
    var library = await buildLibrary('class C { external const C(); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            external const @25
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            external const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 25
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        external const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_documented() async {
    var library = await buildLibrary('''
class C {
  /**
   * Docs
   */
  C();
}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            @34
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              documentationComment: /**\n   * Docs\n   */
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              documentationComment: /**\n   * Docs\n   */
              typeName: C
              typeNameOffset: 34
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          documentationComment: /**\n   * Docs\n   */
''');
  }

  test_class_constructor_explicit_named() async {
    var library = await buildLibrary('class C { C.foo(); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            foo @12
              reference: <testLibraryFragment>::@class::C::@constructor::foo
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 11
              nameEnd: 15
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            foo @12
              reference: <testLibraryFragment>::@class::C::@constructor::foo
              element: <testLibraryFragment>::@class::C::@constructor::foo#element
              typeName: C
              typeNameOffset: 10
              periodOffset: 11
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        foo
          firstFragment: <testLibraryFragment>::@class::C::@constructor::foo
''');
  }

  test_class_constructor_explicit_type_params() async {
    var library = await buildLibrary('class C<T, U> { C(); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              defaultType: dynamic
            covariant U @11
              defaultType: dynamic
          constructors
            @16
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
            U @11
              element: <not-implemented>
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 16
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_explicit_unnamed() async {
    var library = await buildLibrary('class C { C(); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            @10
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 10
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_external() async {
    var library = await buildLibrary('class C { external C(); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            external @19
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            external new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 19
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        external new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_factory() async {
    var library = await buildLibrary('class C { factory C() => throw 0; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            factory @18
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 18
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_field_formal_dynamic_dynamic() async {
    var library =
        await buildLibrary('class C { dynamic x; C(dynamic this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            @21
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional final this.x @36
                  type: dynamic
                  field: <testLibraryFragment>::@class::C::@field::x
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: dynamic
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 21
              formalParameters
                this.x @36
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final x
              type: dynamic
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
''');
  }

  test_class_constructor_field_formal_dynamic_typed() async {
    var library = await buildLibrary('class C { dynamic x; C(int this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            @21
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional final this.x @32
                  type: int
                  field: <testLibraryFragment>::@class::C::@field::x
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: dynamic
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 21
              formalParameters
                this.x @32
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final x
              type: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
''');
  }

  test_class_constructor_field_formal_dynamic_untyped() async {
    var library = await buildLibrary('class C { dynamic x; C(this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            @21
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional final hasImplicitType this.x @28
                  type: dynamic
                  field: <testLibraryFragment>::@class::C::@field::x
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: dynamic
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 21
              formalParameters
                this.x @28
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType x
              type: dynamic
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
''');
  }

  test_class_constructor_field_formal_functionTyped_noReturnType() async {
    var library = await buildLibrary(r'''
class C {
  var x;
  C(this.x(double b));
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @16
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            @21
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional final this.x @28
                  type: dynamic Function(double)
                  parameters
                    requiredPositional b @37
                      type: double
                  field: <testLibraryFragment>::@class::C::@field::x
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: dynamic
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @16
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 21
              formalParameters
                this.x @28
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final x
              type: dynamic Function(double)
              formalParameters
                requiredPositional b
                  type: double
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
''');
  }

  test_class_constructor_field_formal_functionTyped_withReturnType() async {
    var library = await buildLibrary(r'''
class C {
  var x;
  C(int this.x(double b));
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @16
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            @21
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional final this.x @32
                  type: int Function(double)
                  parameters
                    requiredPositional b @41
                      type: double
                  field: <testLibraryFragment>::@class::C::@field::x
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: dynamic
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @16
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 21
              formalParameters
                this.x @32
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final x
              type: int Function(double)
              formalParameters
                requiredPositional b
                  type: double
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
''');
  }

  test_class_constructor_field_formal_functionTyped_withReturnType_generic() async {
    var library = await buildLibrary(r'''
class C {
  Function() f;
  C(List<U> this.f<T, U>(T t));
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            f @23
              reference: <testLibraryFragment>::@class::C::@field::f
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic Function()
          constructors
            @28
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional final this.f @43
                  type: List<U> Function<T, U>(T)
                  typeParameters
                    covariant T @45
                    covariant U @48
                  parameters
                    requiredPositional t @53
                      type: T
                  field: <testLibraryFragment>::@class::C::@field::f
          accessors
            synthetic get f @-1
              reference: <testLibraryFragment>::@class::C::@getter::f
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic Function()
            synthetic set f= @-1
              reference: <testLibraryFragment>::@class::C::@setter::f
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _f @-1
                  type: dynamic Function()
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            f @23
              reference: <testLibraryFragment>::@class::C::@field::f
              element: <testLibraryFragment>::@class::C::@field::f#element
              getter2: <testLibraryFragment>::@class::C::@getter::f
              setter2: <testLibraryFragment>::@class::C::@setter::f
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 28
              formalParameters
                this.f @43
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::f#element
          getters
            synthetic get f
              reference: <testLibraryFragment>::@class::C::@getter::f
              element: <testLibraryFragment>::@class::C::@getter::f#element
          setters
            synthetic set f
              reference: <testLibraryFragment>::@class::C::@setter::f
              element: <testLibraryFragment>::@class::C::@setter::f#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::f::@parameter::_f#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        f
          firstFragment: <testLibraryFragment>::@class::C::@field::f
          type: dynamic Function()
          getter: <testLibraryFragment>::@class::C::@getter::f#element
          setter: <testLibraryFragment>::@class::C::@setter::f#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final f
              type: List<U> Function<T, U>(T)
              formalParameters
                requiredPositional t
                  type: T
      getters
        synthetic get f
          firstFragment: <testLibraryFragment>::@class::C::@getter::f
      setters
        synthetic set f
          firstFragment: <testLibraryFragment>::@class::C::@setter::f
          formalParameters
            requiredPositional _f
              type: dynamic Function()
''');
  }

  test_class_constructor_field_formal_multiple_matching_fields() async {
    // This is a compile-time error but it should still analyze consistently.
    var library = await buildLibrary('class C { C(this.x); int x; String x; }');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @25
              reference: <testLibraryFragment>::@class::C::@field::x::@def::0
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
            x @35
              reference: <testLibraryFragment>::@class::C::@field::x::@def::1
              enclosingElement3: <testLibraryFragment>::@class::C
              type: String
          constructors
            @10
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional final hasImplicitType this.x @17
                  type: int
                  field: <testLibraryFragment>::@class::C::@field::x::@def::0
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x::@def::0
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x::@def::0
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: int
              returnType: void
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x::@def::1
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: String
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x::@def::1
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: String
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @25
              reference: <testLibraryFragment>::@class::C::@field::x::@def::0
              element: <testLibraryFragment>::@class::C::@field::x::@def::0#element
              getter2: <testLibraryFragment>::@class::C::@getter::x::@def::0
              setter2: <testLibraryFragment>::@class::C::@setter::x::@def::0
            x @35
              reference: <testLibraryFragment>::@class::C::@field::x::@def::1
              element: <testLibraryFragment>::@class::C::@field::x::@def::1#element
              getter2: <testLibraryFragment>::@class::C::@getter::x::@def::1
              setter2: <testLibraryFragment>::@class::C::@setter::x::@def::1
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 10
              formalParameters
                this.x @17
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x::@def::0
              element: <testLibraryFragment>::@class::C::@getter::x::@def::0#element
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x::@def::1
              element: <testLibraryFragment>::@class::C::@getter::x::@def::1#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x::@def::0
              element: <testLibraryFragment>::@class::C::@setter::x::@def::0#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@def::0::@parameter::_x#element
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x::@def::1
              element: <testLibraryFragment>::@class::C::@setter::x::@def::1#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@def::1::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x::@def::0
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x::@def::0#element
          setter: <testLibraryFragment>::@class::C::@setter::x::@def::0#element
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x::@def::1
          type: String
          getter: <testLibraryFragment>::@class::C::@getter::x::@def::1#element
          setter: <testLibraryFragment>::@class::C::@setter::x::@def::1#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType x
              type: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x::@def::0
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x::@def::1
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x::@def::0
          formalParameters
            requiredPositional _x
              type: int
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x::@def::1
          formalParameters
            requiredPositional _x
              type: String
''');
  }

  test_class_constructor_field_formal_no_matching_field() async {
    // This is a compile-time error but it should still analyze consistently.
    var library = await buildLibrary('class C { C(this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            @10
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional final hasImplicitType this.x @17
                  type: dynamic
                  field: <null>
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 10
              formalParameters
                this.x @17
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType x
              type: dynamic
''');
  }

  test_class_constructor_field_formal_typed_dynamic() async {
    var library = await buildLibrary('class C { num x; C(dynamic this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: num
          constructors
            @17
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional final this.x @32
                  type: dynamic
                  field: <testLibraryFragment>::@class::C::@field::x
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: num
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: num
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                this.x @32
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: num
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final x
              type: dynamic
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: num
''');
  }

  test_class_constructor_field_formal_typed_typed() async {
    var library = await buildLibrary('class C { num x; C(int this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: num
          constructors
            @17
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional final this.x @28
                  type: int
                  field: <testLibraryFragment>::@class::C::@field::x
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: num
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: num
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                this.x @28
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: num
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final x
              type: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: num
''');
  }

  test_class_constructor_field_formal_typed_untyped() async {
    var library = await buildLibrary('class C { num x; C(this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: num
          constructors
            @17
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional final hasImplicitType this.x @24
                  type: num
                  field: <testLibraryFragment>::@class::C::@field::x
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: num
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: num
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                this.x @24
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: num
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType x
              type: num
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: num
''');
  }

  test_class_constructor_field_formal_untyped_dynamic() async {
    var library = await buildLibrary('class C { var x; C(dynamic this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            @17
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional final this.x @32
                  type: dynamic
                  field: <testLibraryFragment>::@class::C::@field::x
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: dynamic
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                this.x @32
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final x
              type: dynamic
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
''');
  }

  test_class_constructor_field_formal_untyped_typed() async {
    var library = await buildLibrary('class C { var x; C(int this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            @17
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional final this.x @28
                  type: int
                  field: <testLibraryFragment>::@class::C::@field::x
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: dynamic
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                this.x @28
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final x
              type: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
''');
  }

  test_class_constructor_field_formal_untyped_untyped() async {
    var library = await buildLibrary('class C { var x; C(this.x); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            @17
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional final hasImplicitType this.x @24
                  type: dynamic
                  field: <testLibraryFragment>::@class::C::@field::x
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: dynamic
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                this.x @24
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType x
              type: dynamic
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
''');
  }

  test_class_constructor_fieldFormal_named_noDefault() async {
    var library = await buildLibrary('class C { int x; C({this.x}); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            @17
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                optionalNamed default final hasImplicitType this.x @25
                  reference: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x
                  type: int
                  field: <testLibraryFragment>::@class::C::@field::x
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                default this.x @25
                  reference: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            optionalNamed final hasImplicitType x
              firstFragment: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x
              type: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: int
''');
  }

  test_class_constructor_fieldFormal_named_withDefault() async {
    var library = await buildLibrary('class C { int x; C({this.x: 42}); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            @17
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                optionalNamed default final hasImplicitType this.x @25
                  reference: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x
                  type: int
                  constantInitializer
                    IntegerLiteral
                      literal: 42 @28
                      staticType: int
                  field: <testLibraryFragment>::@class::C::@field::x
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                default this.x @25
                  reference: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            optionalNamed final hasImplicitType x
              firstFragment: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x
              type: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: int
''');
  }

  test_class_constructor_fieldFormal_optional_noDefault() async {
    var library = await buildLibrary('class C { int x; C([this.x]); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            @17
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                optionalPositional default final hasImplicitType this.x @25
                  type: int
                  field: <testLibraryFragment>::@class::C::@field::x
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                default this.x @25
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            optionalPositional final hasImplicitType x
              type: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: int
''');
  }

  test_class_constructor_fieldFormal_optional_withDefault() async {
    var library = await buildLibrary('class C { int x; C([this.x = 42]); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            @17
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                optionalPositional default final hasImplicitType this.x @25
                  type: int
                  constantInitializer
                    IntegerLiteral
                      literal: 42 @29
                      staticType: int
                  field: <testLibraryFragment>::@class::C::@field::x
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 17
              formalParameters
                default this.x @25
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            optionalPositional final hasImplicitType x
              type: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: int
''');
  }

  test_class_constructor_implicit_type_params() async {
    var library = await buildLibrary('class C<T, U> {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              defaultType: dynamic
            covariant U @11
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
            U @11
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_initializers_assertInvocation() async {
    var library = await buildLibrary('''
class C {
  const C(int x) : assert(x >= 42);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            const @18
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional x @24
                  type: int
              constantInitializers
                AssertInitializer
                  assertKeyword: assert @29
                  leftParenthesis: ( @35
                  condition: BinaryExpression
                    leftOperand: SimpleIdentifier
                      token: x @36
                      staticElement: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x
                      element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
                      staticType: int
                    operator: >= @38
                    rightOperand: IntegerLiteral
                      literal: 42 @41
                      staticType: int
                    staticElement: dart:core::<fragment>::@class::num::@method::>=
                    element: dart:core::<fragment>::@class::num::@method::>=#element
                    staticInvokeType: bool Function(num)
                    staticType: bool
                  rightParenthesis: ) @43
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 18
              formalParameters
                x @24
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
              constantInitializers
                AssertInitializer
                  assertKeyword: assert @29
                  leftParenthesis: ( @35
                  condition: BinaryExpression
                    leftOperand: SimpleIdentifier
                      token: x @36
                      staticElement: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x
                      element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
                      staticType: int
                    operator: >= @38
                    rightOperand: IntegerLiteral
                      literal: 42 @41
                      staticType: int
                    staticElement: dart:core::<fragment>::@class::num::@method::>=
                    element: dart:core::<fragment>::@class::num::@method::>=#element
                    staticInvokeType: bool Function(num)
                    staticType: bool
                  rightParenthesis: ) @43
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional x
              type: int
''');
  }

  test_class_constructor_initializers_assertInvocation_message() async {
    var library = await buildLibrary('''
class C {
  const C(int x) : assert(x >= 42, 'foo');
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            const @18
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional x @24
                  type: int
              constantInitializers
                AssertInitializer
                  assertKeyword: assert @29
                  leftParenthesis: ( @35
                  condition: BinaryExpression
                    leftOperand: SimpleIdentifier
                      token: x @36
                      staticElement: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x
                      element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
                      staticType: int
                    operator: >= @38
                    rightOperand: IntegerLiteral
                      literal: 42 @41
                      staticType: int
                    staticElement: dart:core::<fragment>::@class::num::@method::>=
                    element: dart:core::<fragment>::@class::num::@method::>=#element
                    staticInvokeType: bool Function(num)
                    staticType: bool
                  comma: , @43
                  message: SimpleStringLiteral
                    literal: 'foo' @45
                  rightParenthesis: ) @50
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 18
              formalParameters
                x @24
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
              constantInitializers
                AssertInitializer
                  assertKeyword: assert @29
                  leftParenthesis: ( @35
                  condition: BinaryExpression
                    leftOperand: SimpleIdentifier
                      token: x @36
                      staticElement: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x
                      element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
                      staticType: int
                    operator: >= @38
                    rightOperand: IntegerLiteral
                      literal: 42 @41
                      staticType: int
                    staticElement: dart:core::<fragment>::@class::num::@method::>=
                    element: dart:core::<fragment>::@class::num::@method::>=#element
                    staticInvokeType: bool Function(num)
                    staticType: bool
                  comma: , @43
                  message: SimpleStringLiteral
                    literal: 'foo' @45
                  rightParenthesis: ) @50
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional x
              type: int
''');
  }

  test_class_constructor_initializers_field() async {
    var library = await buildLibrary('''
class C {
  final x;
  const C() : x = 42;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            final x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            const @29
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              constantInitializers
                ConstructorFieldInitializer
                  fieldName: SimpleIdentifier
                    token: x @35
                    staticElement: <testLibraryFragment>::@class::C::@field::x
                    element: <testLibraryFragment>::@class::C::@field::x#element
                    staticType: null
                  equals: = @37
                  expression: IntegerLiteral
                    literal: 42 @39
                    staticType: int
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 29
              constantInitializers
                ConstructorFieldInitializer
                  fieldName: SimpleIdentifier
                    token: x @35
                    staticElement: <testLibraryFragment>::@class::C::@field::x
                    element: <testLibraryFragment>::@class::C::@field::x#element
                    staticType: null
                  equals: = @37
                  expression: IntegerLiteral
                    literal: 42 @39
                    staticType: int
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
''');
  }

  test_class_constructor_initializers_field_notConst() async {
    var library = await buildLibrary('''
class C {
  final x;
  const C() : x = foo();
}
int foo() => 42;
''');
    // It is OK to keep non-constant initializers.
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            final x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            const @29
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              constantInitializers
                ConstructorFieldInitializer
                  fieldName: SimpleIdentifier
                    token: x @35
                    staticElement: <testLibraryFragment>::@class::C::@field::x
                    element: <testLibraryFragment>::@class::C::@field::x#element
                    staticType: null
                  equals: = @37
                  expression: MethodInvocation
                    methodName: SimpleIdentifier
                      token: foo @39
                      staticElement: <testLibraryFragment>::@function::foo
                      element: <testLibrary>::@function::foo
                      staticType: int Function()
                    argumentList: ArgumentList
                      leftParenthesis: ( @42
                      rightParenthesis: ) @43
                    staticInvokeType: int Function()
                    staticType: int
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
      functions
        foo @52
          reference: <testLibraryFragment>::@function::foo
          enclosingElement3: <testLibraryFragment>
          returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 29
              constantInitializers
                ConstructorFieldInitializer
                  fieldName: SimpleIdentifier
                    token: x @35
                    staticElement: <testLibraryFragment>::@class::C::@field::x
                    element: <testLibraryFragment>::@class::C::@field::x#element
                    staticType: null
                  equals: = @37
                  expression: MethodInvocation
                    methodName: SimpleIdentifier
                      token: foo @39
                      staticElement: <testLibraryFragment>::@function::foo
                      element: <testLibrary>::@function::foo
                      staticType: int Function()
                    argumentList: ArgumentList
                      leftParenthesis: ( @42
                      rightParenthesis: ) @43
                    staticInvokeType: int Function()
                    staticType: int
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
      functions
        foo @52
          reference: <testLibraryFragment>::@function::foo
          element: <testLibrary>::@function::foo
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
  functions
    foo
      reference: <testLibrary>::@function::foo
      firstFragment: <testLibraryFragment>::@function::foo
      returnType: int
''');
  }

  test_class_constructor_initializers_field_optionalPositionalParameter() async {
    var library = await buildLibrary('''
class A {
  final int _f;
  const A([int f = 0]) : _f = f;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final promotable _f @22
              reference: <testLibraryFragment>::@class::A::@field::_f
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
          constructors
            const @34
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                optionalPositional default f @41
                  type: int
                  constantInitializer
                    IntegerLiteral
                      literal: 0 @45
                      staticType: int
              constantInitializers
                ConstructorFieldInitializer
                  fieldName: SimpleIdentifier
                    token: _f @51
                    staticElement: <testLibraryFragment>::@class::A::@field::_f
                    element: <testLibraryFragment>::@class::A::@field::_f#element
                    staticType: null
                  equals: = @54
                  expression: SimpleIdentifier
                    token: f @56
                    staticElement: <testLibraryFragment>::@class::A::@constructor::new::@parameter::f
                    element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::f#element
                    staticType: int
          accessors
            synthetic get _f @-1
              reference: <testLibraryFragment>::@class::A::@getter::_f
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _f @22
              reference: <testLibraryFragment>::@class::A::@field::_f
              element: <testLibraryFragment>::@class::A::@field::_f#element
              getter2: <testLibraryFragment>::@class::A::@getter::_f
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 34
              formalParameters
                default f @41
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::f#element
              constantInitializers
                ConstructorFieldInitializer
                  fieldName: SimpleIdentifier
                    token: _f @51
                    staticElement: <testLibraryFragment>::@class::A::@field::_f
                    element: <testLibraryFragment>::@class::A::@field::_f#element
                    staticType: null
                  equals: = @54
                  expression: SimpleIdentifier
                    token: f @56
                    staticElement: <testLibraryFragment>::@class::A::@constructor::new::@parameter::f
                    element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::f#element
                    staticType: int
          getters
            synthetic get _f
              reference: <testLibraryFragment>::@class::A::@getter::_f
              element: <testLibraryFragment>::@class::A::@getter::_f#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _f
          firstFragment: <testLibraryFragment>::@class::A::@field::_f
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::_f#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            optionalPositional f
              type: int
      getters
        synthetic get _f
          firstFragment: <testLibraryFragment>::@class::A::@getter::_f
''');
  }

  test_class_constructor_initializers_field_recordLiteral() async {
    var library = await buildLibrary('''
class C {
  final Object x;
  const C(int a) : x = (0, a);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            final x @25
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: Object
          constructors
            const @36
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional a @42
                  type: int
              constantInitializers
                ConstructorFieldInitializer
                  fieldName: SimpleIdentifier
                    token: x @47
                    staticElement: <testLibraryFragment>::@class::C::@field::x
                    element: <testLibraryFragment>::@class::C::@field::x#element
                    staticType: null
                  equals: = @49
                  expression: RecordLiteral
                    leftParenthesis: ( @51
                    fields
                      IntegerLiteral
                        literal: 0 @52
                        staticType: int
                      SimpleIdentifier
                        token: a @55
                        staticElement: <testLibraryFragment>::@class::C::@constructor::new::@parameter::a
                        element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::a#element
                        staticType: int
                    rightParenthesis: ) @56
                    staticType: (int, int)
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: Object
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @25
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 36
              formalParameters
                a @42
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::a#element
              constantInitializers
                ConstructorFieldInitializer
                  fieldName: SimpleIdentifier
                    token: x @47
                    staticElement: <testLibraryFragment>::@class::C::@field::x
                    element: <testLibraryFragment>::@class::C::@field::x#element
                    staticType: null
                  equals: = @49
                  expression: RecordLiteral
                    leftParenthesis: ( @51
                    fields
                      IntegerLiteral
                        literal: 0 @52
                        staticType: int
                      SimpleIdentifier
                        token: a @55
                        staticElement: <testLibraryFragment>::@class::C::@constructor::new::@parameter::a
                        element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::a#element
                        staticType: int
                    rightParenthesis: ) @56
                    staticType: (int, int)
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: Object
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional a
              type: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
''');
  }

  test_class_constructor_initializers_field_withParameter() async {
    var library = await buildLibrary('''
class C {
  final x;
  const C(int p) : x = 1 + p;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            final x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            const @29
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional p @35
                  type: int
              constantInitializers
                ConstructorFieldInitializer
                  fieldName: SimpleIdentifier
                    token: x @40
                    staticElement: <testLibraryFragment>::@class::C::@field::x
                    element: <testLibraryFragment>::@class::C::@field::x#element
                    staticType: null
                  equals: = @42
                  expression: BinaryExpression
                    leftOperand: IntegerLiteral
                      literal: 1 @44
                      staticType: int
                    operator: + @46
                    rightOperand: SimpleIdentifier
                      token: p @48
                      staticElement: <testLibraryFragment>::@class::C::@constructor::new::@parameter::p
                      element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::p#element
                      staticType: int
                    staticElement: dart:core::<fragment>::@class::num::@method::+
                    element: dart:core::<fragment>::@class::num::@method::+#element
                    staticInvokeType: num Function(num)
                    staticType: int
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 29
              formalParameters
                p @35
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::p#element
              constantInitializers
                ConstructorFieldInitializer
                  fieldName: SimpleIdentifier
                    token: x @40
                    staticElement: <testLibraryFragment>::@class::C::@field::x
                    element: <testLibraryFragment>::@class::C::@field::x#element
                    staticType: null
                  equals: = @42
                  expression: BinaryExpression
                    leftOperand: IntegerLiteral
                      literal: 1 @44
                      staticType: int
                    operator: + @46
                    rightOperand: SimpleIdentifier
                      token: p @48
                      staticElement: <testLibraryFragment>::@class::C::@constructor::new::@parameter::p
                      element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::p#element
                      staticType: int
                    staticElement: dart:core::<fragment>::@class::num::@method::+
                    element: dart:core::<fragment>::@class::num::@method::+#element
                    staticInvokeType: num Function(num)
                    staticType: int
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional p
              type: int
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
''');
  }

  test_class_constructor_initializers_genericFunctionType() async {
    var library = await buildLibrary('''
class A<T> {
  const A();
}
class B {
  const B(dynamic x);
  const B.f()
   : this(A<Function()>());
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              defaultType: dynamic
          constructors
            const @21
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
        class B @34
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          constructors
            const @46
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                requiredPositional x @56
                  type: dynamic
            const f @70
              reference: <testLibraryFragment>::@class::B::@constructor::f
              enclosingElement3: <testLibraryFragment>::@class::B
              periodOffset: 69
              nameEnd: 71
              constantInitializers
                RedirectingConstructorInvocation
                  thisKeyword: this @79
                  argumentList: ArgumentList
                    leftParenthesis: ( @83
                    arguments
                      InstanceCreationExpression
                        constructorName: ConstructorName
                          type: NamedType
                            name: A @84
                            typeArguments: TypeArgumentList
                              leftBracket: < @85
                              arguments
                                GenericFunctionType
                                  functionKeyword: Function @86
                                  parameters: FormalParameterList
                                    leftParenthesis: ( @94
                                    rightParenthesis: ) @95
                                  declaredElement: GenericFunctionTypeElement
                                    parameters
                                    returnType: dynamic
                                    type: dynamic Function()
                                  type: dynamic Function()
                              rightBracket: > @96
                            element: <testLibraryFragment>::@class::A
                            element2: <testLibrary>::@class::A
                            type: A<dynamic Function()>
                          staticElement: ConstructorMember
                            base: <testLibraryFragment>::@class::A::@constructor::new
                            substitution: {T: dynamic Function()}
                          element: <testLibraryFragment>::@class::A::@constructor::new#element
                        argumentList: ArgumentList
                          leftParenthesis: ( @97
                          rightParenthesis: ) @98
                        staticType: A<dynamic Function()>
                    rightParenthesis: ) @99
                  staticElement: <testLibraryFragment>::@class::B::@constructor::new
                  element: <testLibraryFragment>::@class::B::@constructor::new#element
              redirectedConstructor: <testLibraryFragment>::@class::B::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 21
        class B @34
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            const new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 46
              formalParameters
                x @56
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::x#element
            const f @70
              reference: <testLibraryFragment>::@class::B::@constructor::f
              element: <testLibraryFragment>::@class::B::@constructor::f#element
              typeName: B
              typeNameOffset: 68
              periodOffset: 69
              constantInitializers
                RedirectingConstructorInvocation
                  thisKeyword: this @79
                  argumentList: ArgumentList
                    leftParenthesis: ( @83
                    arguments
                      InstanceCreationExpression
                        constructorName: ConstructorName
                          type: NamedType
                            name: A @84
                            typeArguments: TypeArgumentList
                              leftBracket: < @85
                              arguments
                                GenericFunctionType
                                  functionKeyword: Function @86
                                  parameters: FormalParameterList
                                    leftParenthesis: ( @94
                                    rightParenthesis: ) @95
                                  declaredElement: GenericFunctionTypeElement
                                    parameters
                                    returnType: dynamic
                                    type: dynamic Function()
                                  type: dynamic Function()
                              rightBracket: > @96
                            element: <testLibraryFragment>::@class::A
                            element2: <testLibrary>::@class::A
                            type: A<dynamic Function()>
                          staticElement: ConstructorMember
                            base: <testLibraryFragment>::@class::A::@constructor::new
                            substitution: {T: dynamic Function()}
                          element: <testLibraryFragment>::@class::A::@constructor::new#element
                        argumentList: ArgumentList
                          leftParenthesis: ( @97
                          rightParenthesis: ) @98
                        staticType: A<dynamic Function()>
                    rightParenthesis: ) @99
                  staticElement: <testLibraryFragment>::@class::B::@constructor::new
                  element: <testLibraryFragment>::@class::B::@constructor::new#element
              redirectedConstructor: <testLibraryFragment>::@class::B::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional x
              type: dynamic
        const f
          firstFragment: <testLibraryFragment>::@class::B::@constructor::f
          redirectedConstructor: <testLibraryFragment>::@class::B::@constructor::new#element
''');
  }

  test_class_constructor_initializers_superInvocation_argumentContextType() async {
    var library = await buildLibrary('''
class A {
  const A(List<String> values);
}
class B extends A {
  const B() : super(const []);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            const @18
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional values @33
                  type: List<String>
        class B @50
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            const @72
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @78
                  argumentList: ArgumentList
                    leftParenthesis: ( @83
                    arguments
                      ListLiteral
                        constKeyword: const @84
                        leftBracket: [ @90
                        rightBracket: ] @91
                        staticType: List<String>
                    rightParenthesis: ) @92
                  staticElement: <testLibraryFragment>::@class::A::@constructor::new
                  element: <testLibraryFragment>::@class::A::@constructor::new#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 18
              formalParameters
                values @33
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::values#element
        class B @50
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            const new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 72
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @78
                  argumentList: ArgumentList
                    leftParenthesis: ( @83
                    arguments
                      ListLiteral
                        constKeyword: const @84
                        leftBracket: [ @90
                        rightBracket: ] @91
                        staticType: List<String>
                    rightParenthesis: ) @92
                  staticElement: <testLibraryFragment>::@class::A::@constructor::new
                  element: <testLibraryFragment>::@class::A::@constructor::new#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional values
              type: List<String>
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_initializers_superInvocation_named() async {
    var library = await buildLibrary('''
class A {
  const A.aaa(int p);
}
class C extends A {
  const C() : super.aaa(42);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            const aaa @20
              reference: <testLibraryFragment>::@class::A::@constructor::aaa
              enclosingElement3: <testLibraryFragment>::@class::A
              periodOffset: 19
              nameEnd: 23
              parameters
                requiredPositional p @28
                  type: int
        class C @40
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            const @62
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @68
                  period: . @73
                  constructorName: SimpleIdentifier
                    token: aaa @74
                    staticElement: <testLibraryFragment>::@class::A::@constructor::aaa
                    element: <testLibraryFragment>::@class::A::@constructor::aaa#element
                    staticType: null
                  argumentList: ArgumentList
                    leftParenthesis: ( @77
                    arguments
                      IntegerLiteral
                        literal: 42 @78
                        staticType: int
                    rightParenthesis: ) @80
                  staticElement: <testLibraryFragment>::@class::A::@constructor::aaa
                  element: <testLibraryFragment>::@class::A::@constructor::aaa#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::aaa
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            const aaa @20
              reference: <testLibraryFragment>::@class::A::@constructor::aaa
              element: <testLibraryFragment>::@class::A::@constructor::aaa#element
              typeName: A
              typeNameOffset: 18
              periodOffset: 19
              formalParameters
                p @28
                  element: <testLibraryFragment>::@class::A::@constructor::aaa::@parameter::p#element
        class C @40
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 62
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @68
                  period: . @73
                  constructorName: SimpleIdentifier
                    token: aaa @74
                    staticElement: <testLibraryFragment>::@class::A::@constructor::aaa
                    element: <testLibraryFragment>::@class::A::@constructor::aaa#element
                    staticType: null
                  argumentList: ArgumentList
                    leftParenthesis: ( @77
                    arguments
                      IntegerLiteral
                        literal: 42 @78
                        staticType: int
                    rightParenthesis: ) @80
                  staticElement: <testLibraryFragment>::@class::A::@constructor::aaa
                  element: <testLibraryFragment>::@class::A::@constructor::aaa#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::aaa
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        const aaa
          firstFragment: <testLibraryFragment>::@class::A::@constructor::aaa
          formalParameters
            requiredPositional p
              type: int
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: A
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::aaa#element
''');
  }

  test_class_constructor_initializers_superInvocation_named_underscore() async {
    var library = await buildLibrary('''
class A {
  const A._();
}
class B extends A {
  const B() : super._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            const _ @20
              reference: <testLibraryFragment>::@class::A::@constructor::_
              enclosingElement3: <testLibraryFragment>::@class::A
              periodOffset: 19
              nameEnd: 21
        class B @33
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            const @55
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @61
                  period: . @66
                  constructorName: SimpleIdentifier
                    token: _ @67
                    staticElement: <testLibraryFragment>::@class::A::@constructor::_
                    element: <testLibraryFragment>::@class::A::@constructor::_#element
                    staticType: null
                  argumentList: ArgumentList
                    leftParenthesis: ( @68
                    rightParenthesis: ) @69
                  staticElement: <testLibraryFragment>::@class::A::@constructor::_
                  element: <testLibraryFragment>::@class::A::@constructor::_#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::_
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            const _ @20
              reference: <testLibraryFragment>::@class::A::@constructor::_
              element: <testLibraryFragment>::@class::A::@constructor::_#element
              typeName: A
              typeNameOffset: 18
              periodOffset: 19
        class B @33
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            const new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 55
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @61
                  period: . @66
                  constructorName: SimpleIdentifier
                    token: _ @67
                    staticElement: <testLibraryFragment>::@class::A::@constructor::_
                    element: <testLibraryFragment>::@class::A::@constructor::_#element
                    staticType: null
                  argumentList: ArgumentList
                    leftParenthesis: ( @68
                    rightParenthesis: ) @69
                  staticElement: <testLibraryFragment>::@class::A::@constructor::_
                  element: <testLibraryFragment>::@class::A::@constructor::_#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::_
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        const _
          firstFragment: <testLibraryFragment>::@class::A::@constructor::_
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::_#element
''');
  }

  test_class_constructor_initializers_superInvocation_namedExpression() async {
    var library = await buildLibrary('''
class A {
  const A.aaa(a, {int b});
}
class C extends A {
  const C() : super.aaa(1, b: 2);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            const aaa @20
              reference: <testLibraryFragment>::@class::A::@constructor::aaa
              enclosingElement3: <testLibraryFragment>::@class::A
              periodOffset: 19
              nameEnd: 23
              parameters
                requiredPositional hasImplicitType a @24
                  type: dynamic
                optionalNamed default b @32
                  reference: <testLibraryFragment>::@class::A::@constructor::aaa::@parameter::b
                  type: int
        class C @45
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            const @67
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @73
                  period: . @78
                  constructorName: SimpleIdentifier
                    token: aaa @79
                    staticElement: <testLibraryFragment>::@class::A::@constructor::aaa
                    element: <testLibraryFragment>::@class::A::@constructor::aaa#element
                    staticType: null
                  argumentList: ArgumentList
                    leftParenthesis: ( @82
                    arguments
                      IntegerLiteral
                        literal: 1 @83
                        staticType: int
                      NamedExpression
                        name: Label
                          label: SimpleIdentifier
                            token: b @86
                            staticElement: <testLibraryFragment>::@class::A::@constructor::aaa::@parameter::b
                            element: <testLibraryFragment>::@class::A::@constructor::aaa::@parameter::b#element
                            staticType: null
                          colon: : @87
                        expression: IntegerLiteral
                          literal: 2 @89
                          staticType: int
                    rightParenthesis: ) @90
                  staticElement: <testLibraryFragment>::@class::A::@constructor::aaa
                  element: <testLibraryFragment>::@class::A::@constructor::aaa#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::aaa
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            const aaa @20
              reference: <testLibraryFragment>::@class::A::@constructor::aaa
              element: <testLibraryFragment>::@class::A::@constructor::aaa#element
              typeName: A
              typeNameOffset: 18
              periodOffset: 19
              formalParameters
                a @24
                  element: <testLibraryFragment>::@class::A::@constructor::aaa::@parameter::a#element
                default b @32
                  reference: <testLibraryFragment>::@class::A::@constructor::aaa::@parameter::b
                  element: <testLibraryFragment>::@class::A::@constructor::aaa::@parameter::b#element
        class C @45
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 67
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @73
                  period: . @78
                  constructorName: SimpleIdentifier
                    token: aaa @79
                    staticElement: <testLibraryFragment>::@class::A::@constructor::aaa
                    element: <testLibraryFragment>::@class::A::@constructor::aaa#element
                    staticType: null
                  argumentList: ArgumentList
                    leftParenthesis: ( @82
                    arguments
                      IntegerLiteral
                        literal: 1 @83
                        staticType: int
                      NamedExpression
                        name: Label
                          label: SimpleIdentifier
                            token: b @86
                            staticElement: <testLibraryFragment>::@class::A::@constructor::aaa::@parameter::b
                            element: <testLibraryFragment>::@class::A::@constructor::aaa::@parameter::b#element
                            staticType: null
                          colon: : @87
                        expression: IntegerLiteral
                          literal: 2 @89
                          staticType: int
                    rightParenthesis: ) @90
                  staticElement: <testLibraryFragment>::@class::A::@constructor::aaa
                  element: <testLibraryFragment>::@class::A::@constructor::aaa#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::aaa
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        const aaa
          firstFragment: <testLibraryFragment>::@class::A::@constructor::aaa
          formalParameters
            requiredPositional hasImplicitType a
              type: dynamic
            optionalNamed b
              firstFragment: <testLibraryFragment>::@class::A::@constructor::aaa::@parameter::b
              type: int
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: A
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::aaa#element
''');
  }

  test_class_constructor_initializers_superInvocation_unnamed() async {
    var library = await buildLibrary('''
class A {
  const A(int p);
}
class C extends A {
  const C.ccc() : super(42);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            const @18
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional p @24
                  type: int
        class C @36
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            const ccc @60
              reference: <testLibraryFragment>::@class::C::@constructor::ccc
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 59
              nameEnd: 63
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @68
                  argumentList: ArgumentList
                    leftParenthesis: ( @73
                    arguments
                      IntegerLiteral
                        literal: 42 @74
                        staticType: int
                    rightParenthesis: ) @76
                  staticElement: <testLibraryFragment>::@class::A::@constructor::new
                  element: <testLibraryFragment>::@class::A::@constructor::new#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 18
              formalParameters
                p @24
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::p#element
        class C @36
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const ccc @60
              reference: <testLibraryFragment>::@class::C::@constructor::ccc
              element: <testLibraryFragment>::@class::C::@constructor::ccc#element
              typeName: C
              typeNameOffset: 58
              periodOffset: 59
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @68
                  argumentList: ArgumentList
                    leftParenthesis: ( @73
                    arguments
                      IntegerLiteral
                        literal: 42 @74
                        staticType: int
                    rightParenthesis: ) @76
                  staticElement: <testLibraryFragment>::@class::A::@constructor::new
                  element: <testLibraryFragment>::@class::A::@constructor::new#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional p
              type: int
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: A
      constructors
        const ccc
          firstFragment: <testLibraryFragment>::@class::C::@constructor::ccc
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_initializers_thisInvocation_argumentContextType() async {
    var library = await buildLibrary('''
class A {
  const A(List<String> values);
  const A.empty() : this(const []);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            const @18
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional values @33
                  type: List<String>
            const empty @52
              reference: <testLibraryFragment>::@class::A::@constructor::empty
              enclosingElement3: <testLibraryFragment>::@class::A
              periodOffset: 51
              nameEnd: 57
              constantInitializers
                RedirectingConstructorInvocation
                  thisKeyword: this @62
                  argumentList: ArgumentList
                    leftParenthesis: ( @66
                    arguments
                      ListLiteral
                        constKeyword: const @67
                        leftBracket: [ @73
                        rightBracket: ] @74
                        staticType: List<String>
                    rightParenthesis: ) @75
                  staticElement: <testLibraryFragment>::@class::A::@constructor::new
                  element: <testLibraryFragment>::@class::A::@constructor::new#element
              redirectedConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 18
              formalParameters
                values @33
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::values#element
            const empty @52
              reference: <testLibraryFragment>::@class::A::@constructor::empty
              element: <testLibraryFragment>::@class::A::@constructor::empty#element
              typeName: A
              typeNameOffset: 50
              periodOffset: 51
              constantInitializers
                RedirectingConstructorInvocation
                  thisKeyword: this @62
                  argumentList: ArgumentList
                    leftParenthesis: ( @66
                    arguments
                      ListLiteral
                        constKeyword: const @67
                        leftBracket: [ @73
                        rightBracket: ] @74
                        staticType: List<String>
                    rightParenthesis: ) @75
                  staticElement: <testLibraryFragment>::@class::A::@constructor::new
                  element: <testLibraryFragment>::@class::A::@constructor::new#element
              redirectedConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional values
              type: List<String>
        const empty
          firstFragment: <testLibraryFragment>::@class::A::@constructor::empty
          redirectedConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_initializers_thisInvocation_named() async {
    var library = await buildLibrary('''
class C {
  const C() : this.named(1, 'bbb');
  const C.named(int a, String b);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            const @18
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              constantInitializers
                RedirectingConstructorInvocation
                  thisKeyword: this @24
                  period: . @28
                  constructorName: SimpleIdentifier
                    token: named @29
                    staticElement: <testLibraryFragment>::@class::C::@constructor::named
                    element: <testLibraryFragment>::@class::C::@constructor::named#element
                    staticType: null
                  argumentList: ArgumentList
                    leftParenthesis: ( @34
                    arguments
                      IntegerLiteral
                        literal: 1 @35
                        staticType: int
                      SimpleStringLiteral
                        literal: 'bbb' @38
                    rightParenthesis: ) @43
                  staticElement: <testLibraryFragment>::@class::C::@constructor::named
                  element: <testLibraryFragment>::@class::C::@constructor::named#element
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named
            const named @56
              reference: <testLibraryFragment>::@class::C::@constructor::named
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 55
              nameEnd: 61
              parameters
                requiredPositional a @66
                  type: int
                requiredPositional b @76
                  type: String
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 18
              constantInitializers
                RedirectingConstructorInvocation
                  thisKeyword: this @24
                  period: . @28
                  constructorName: SimpleIdentifier
                    token: named @29
                    staticElement: <testLibraryFragment>::@class::C::@constructor::named
                    element: <testLibraryFragment>::@class::C::@constructor::named#element
                    staticType: null
                  argumentList: ArgumentList
                    leftParenthesis: ( @34
                    arguments
                      IntegerLiteral
                        literal: 1 @35
                        staticType: int
                      SimpleStringLiteral
                        literal: 'bbb' @38
                    rightParenthesis: ) @43
                  staticElement: <testLibraryFragment>::@class::C::@constructor::named
                  element: <testLibraryFragment>::@class::C::@constructor::named#element
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named
            const named @56
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 54
              periodOffset: 55
              formalParameters
                a @66
                  element: <testLibraryFragment>::@class::C::@constructor::named::@parameter::a#element
                b @76
                  element: <testLibraryFragment>::@class::C::@constructor::named::@parameter::b#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named#element
        const named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
          formalParameters
            requiredPositional a
              type: int
            requiredPositional b
              type: String
''');
  }

  test_class_constructor_initializers_thisInvocation_namedExpression() async {
    var library = await buildLibrary('''
class C {
  const C() : this.named(1, b: 2);
  const C.named(a, {int b});
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            const @18
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              constantInitializers
                RedirectingConstructorInvocation
                  thisKeyword: this @24
                  period: . @28
                  constructorName: SimpleIdentifier
                    token: named @29
                    staticElement: <testLibraryFragment>::@class::C::@constructor::named
                    element: <testLibraryFragment>::@class::C::@constructor::named#element
                    staticType: null
                  argumentList: ArgumentList
                    leftParenthesis: ( @34
                    arguments
                      IntegerLiteral
                        literal: 1 @35
                        staticType: int
                      NamedExpression
                        name: Label
                          label: SimpleIdentifier
                            token: b @38
                            staticElement: <testLibraryFragment>::@class::C::@constructor::named::@parameter::b
                            element: <testLibraryFragment>::@class::C::@constructor::named::@parameter::b#element
                            staticType: null
                          colon: : @39
                        expression: IntegerLiteral
                          literal: 2 @41
                          staticType: int
                    rightParenthesis: ) @42
                  staticElement: <testLibraryFragment>::@class::C::@constructor::named
                  element: <testLibraryFragment>::@class::C::@constructor::named#element
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named
            const named @55
              reference: <testLibraryFragment>::@class::C::@constructor::named
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 54
              nameEnd: 60
              parameters
                requiredPositional hasImplicitType a @61
                  type: dynamic
                optionalNamed default b @69
                  reference: <testLibraryFragment>::@class::C::@constructor::named::@parameter::b
                  type: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 18
              constantInitializers
                RedirectingConstructorInvocation
                  thisKeyword: this @24
                  period: . @28
                  constructorName: SimpleIdentifier
                    token: named @29
                    staticElement: <testLibraryFragment>::@class::C::@constructor::named
                    element: <testLibraryFragment>::@class::C::@constructor::named#element
                    staticType: null
                  argumentList: ArgumentList
                    leftParenthesis: ( @34
                    arguments
                      IntegerLiteral
                        literal: 1 @35
                        staticType: int
                      NamedExpression
                        name: Label
                          label: SimpleIdentifier
                            token: b @38
                            staticElement: <testLibraryFragment>::@class::C::@constructor::named::@parameter::b
                            element: <testLibraryFragment>::@class::C::@constructor::named::@parameter::b#element
                            staticType: null
                          colon: : @39
                        expression: IntegerLiteral
                          literal: 2 @41
                          staticType: int
                    rightParenthesis: ) @42
                  staticElement: <testLibraryFragment>::@class::C::@constructor::named
                  element: <testLibraryFragment>::@class::C::@constructor::named#element
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named
            const named @55
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 53
              periodOffset: 54
              formalParameters
                a @61
                  element: <testLibraryFragment>::@class::C::@constructor::named::@parameter::a#element
                default b @69
                  reference: <testLibraryFragment>::@class::C::@constructor::named::@parameter::b
                  element: <testLibraryFragment>::@class::C::@constructor::named::@parameter::b#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named#element
        const named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
          formalParameters
            requiredPositional hasImplicitType a
              type: dynamic
            optionalNamed b
              firstFragment: <testLibraryFragment>::@class::C::@constructor::named::@parameter::b
              type: int
''');
  }

  test_class_constructor_initializers_thisInvocation_unnamed() async {
    var library = await buildLibrary('''
class C {
  const C.named() : this(1, 'bbb');
  const C(int a, String b);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            const named @20
              reference: <testLibraryFragment>::@class::C::@constructor::named
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 19
              nameEnd: 25
              constantInitializers
                RedirectingConstructorInvocation
                  thisKeyword: this @30
                  argumentList: ArgumentList
                    leftParenthesis: ( @34
                    arguments
                      IntegerLiteral
                        literal: 1 @35
                        staticType: int
                      SimpleStringLiteral
                        literal: 'bbb' @38
                    rightParenthesis: ) @43
                  staticElement: <testLibraryFragment>::@class::C::@constructor::new
                  element: <testLibraryFragment>::@class::C::@constructor::new#element
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new
            const @54
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional a @60
                  type: int
                requiredPositional b @70
                  type: String
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const named @20
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 18
              periodOffset: 19
              constantInitializers
                RedirectingConstructorInvocation
                  thisKeyword: this @30
                  argumentList: ArgumentList
                    leftParenthesis: ( @34
                    arguments
                      IntegerLiteral
                        literal: 1 @35
                        staticType: int
                      SimpleStringLiteral
                        literal: 'bbb' @38
                    rightParenthesis: ) @43
                  staticElement: <testLibraryFragment>::@class::C::@constructor::new
                  element: <testLibraryFragment>::@class::C::@constructor::new#element
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 54
              formalParameters
                a @60
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::a#element
                b @70
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::b#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        const named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new#element
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional a
              type: int
            requiredPositional b
              type: String
''');
  }

  test_class_constructor_parameters_super_explicitType_function() async {
    var library = await buildLibrary('''
class A {
  A(Object? a);
}

class B extends A {
  B(int super.a<T extends num>(T d)?);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            @12
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional a @22
                  type: Object?
        class B @35
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            @51
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                requiredPositional final super.a @63
                  type: int Function<T extends num>(T)?
                  typeParameters
                    covariant T @65
                      bound: num
                  parameters
                    requiredPositional d @82
                      type: T
                  superConstructorParameter: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                a @22
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
        class B @35
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 51
              formalParameters
                super.a @63
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional a
              type: Object?
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional final a
              type: int Function<T extends num>(T)?
              formalParameters
                requiredPositional d
                  type: T
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_explicitType_interface() async {
    var library = await buildLibrary('''
class A {
  A(num a);
}

class B extends A {
  B(int super.a);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            @12
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional a @18
                  type: num
        class B @31
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            @47
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                requiredPositional final super.a @59
                  type: int
                  superConstructorParameter: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                a @18
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
        class B @31
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 47
              formalParameters
                super.a @59
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional a
              type: num
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional final a
              type: int
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_explicitType_interface_nullable() async {
    var library = await buildLibrary('''
class A {
  A(num? a);
}

class B extends A {
  B(int? super.a);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            @12
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional a @19
                  type: num?
        class B @32
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            @48
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                requiredPositional final super.a @61
                  type: int?
                  superConstructorParameter: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                a @19
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
        class B @32
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 48
              formalParameters
                super.a @61
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional a
              type: num?
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional final a
              type: int?
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_invalid_topFunction() async {
    var library = await buildLibrary('''
void f(super.a) {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      functions
        f @5
          reference: <testLibraryFragment>::@function::f
          enclosingElement3: <testLibraryFragment>
          parameters
            requiredPositional final hasImplicitType super.a @13
              type: dynamic
              superConstructorParameter: <null>
          returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      functions
        f @5
          reference: <testLibraryFragment>::@function::f
          element: <testLibrary>::@function::f
          formalParameters
            super.a @13
              element: <testLibraryFragment>::@function::f::@parameter::a#element
  functions
    f
      reference: <testLibrary>::@function::f
      firstFragment: <testLibraryFragment>::@function::f
      formalParameters
        requiredPositional final hasImplicitType a
          type: dynamic
      returnType: void
''');
  }

  test_class_constructor_parameters_super_optionalNamed() async {
    var library = await buildLibrary('''
class A {
  A({required int a, required double b});
}

class B extends A {
  B({String o1, super.a, String o2, super.b}) : super();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            @12
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredNamed default a @28
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                  type: int
                requiredNamed default b @47
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b
                  type: double
        class B @61
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            @77
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                optionalNamed default o1 @87
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o1
                  type: String
                optionalNamed default final hasImplicitType super.a @97
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
                  type: int
                  superConstructorParameter: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                optionalNamed default o2 @107
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o2
                  type: String
                optionalNamed default final hasImplicitType super.b @117
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b
                  type: double
                  superConstructorParameter: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                default a @28
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
                default b @47
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b#element
        class B @61
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 77
              formalParameters
                default o1 @87
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o1
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o1#element
                default super.a @97
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
                default o2 @107
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o2
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o2#element
                default super.b @117
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredNamed a
              firstFragment: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              type: int
            requiredNamed b
              firstFragment: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b
              type: double
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            optionalNamed o1
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o1
              type: String
            optionalNamed final hasImplicitType a
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
              type: int
            optionalNamed o2
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o2
              type: String
            optionalNamed final hasImplicitType b
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b
              type: double
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_optionalNamed_defaultValue() async {
    var library = await buildLibrary('''
class A {
  A({int a = 0});
}

class B extends A {
  B({super.a});
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            @12
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                optionalNamed default a @19
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                  type: int
                  constantInitializer
                    IntegerLiteral
                      literal: 0 @23
                      staticType: int
        class B @37
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            @53
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                optionalNamed default final hasImplicitType hasDefaultValue super.a @62
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
                  type: int
                  superConstructorParameter: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                default a @19
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
        class B @37
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 53
              formalParameters
                default super.a @62
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            optionalNamed a
              firstFragment: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              type: int
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            optionalNamed final hasImplicitType a
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
              type: int
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_optionalNamed_unresolved() async {
    var library = await buildLibrary('''
class A {
  A({required int a});
}

class B extends A {
  B({super.b});
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            @12
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredNamed default a @28
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                  type: int
        class B @42
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            @58
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                optionalNamed default final hasImplicitType super.b @67
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b
                  type: dynamic
                  superConstructorParameter: <null>
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                default a @28
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
        class B @42
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 58
              formalParameters
                default super.b @67
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredNamed a
              firstFragment: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              type: int
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            optionalNamed final hasImplicitType b
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b
              type: dynamic
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_optionalNamed_unresolved2() async {
    var library = await buildLibrary('''
class A {
  A(int a);
}

class B extends A {
  B({super.a});
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            @12
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional a @18
                  type: int
        class B @31
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            @47
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                optionalNamed default final hasImplicitType super.a @56
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
                  type: dynamic
                  superConstructorParameter: <null>
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                a @18
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
        class B @31
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 47
              formalParameters
                default super.a @56
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional a
              type: int
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            optionalNamed final hasImplicitType a
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
              type: dynamic
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_optionalPositional() async {
    var library = await buildLibrary('''
class A {
  A(int a, double b);
}

class B extends A {
  B([String o1, super.a, String o2, super.b]) : super();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            @12
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional a @18
                  type: int
                requiredPositional b @28
                  type: double
        class B @41
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            @57
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                optionalPositional default o1 @67
                  type: String
                optionalPositional default final hasImplicitType super.a @77
                  type: int
                  superConstructorParameter: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                optionalPositional default o2 @87
                  type: String
                optionalPositional default final hasImplicitType super.b @97
                  type: double
                  superConstructorParameter: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                a @18
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
                b @28
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b#element
        class B @41
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 57
              formalParameters
                default o1 @67
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o1#element
                default super.a @77
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
                default o2 @87
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o2#element
                default super.b @97
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional a
              type: int
            requiredPositional b
              type: double
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            optionalPositional o1
              type: String
            optionalPositional final hasImplicitType a
              type: int
            optionalPositional o2
              type: String
            optionalPositional final hasImplicitType b
              type: double
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_requiredNamed() async {
    var library = await buildLibrary('''
class A {
  A({required int a, required double b});
}

class B extends A {
  B({
    required String o1,
    required super.a,
    required String o2,
    required super.b,
  }) : super();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            @12
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredNamed default a @28
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                  type: int
                requiredNamed default b @47
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b
                  type: double
        class B @61
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            @77
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                requiredNamed default o1 @101
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o1
                  type: String
                requiredNamed default final hasImplicitType super.a @124
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
                  type: int
                  superConstructorParameter: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                requiredNamed default o2 @147
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o2
                  type: String
                requiredNamed default final hasImplicitType super.b @170
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b
                  type: double
                  superConstructorParameter: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                default a @28
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
                default b @47
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b#element
        class B @61
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 77
              formalParameters
                default o1 @101
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o1
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o1#element
                default super.a @124
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
                default o2 @147
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o2
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o2#element
                default super.b @170
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredNamed a
              firstFragment: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              type: int
            requiredNamed b
              firstFragment: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b
              type: double
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredNamed o1
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o1
              type: String
            requiredNamed final hasImplicitType a
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
              type: int
            requiredNamed o2
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o2
              type: String
            requiredNamed final hasImplicitType b
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b
              type: double
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_requiredNamed_defaultValue() async {
    var library = await buildLibrary('''
class A {
  A({int a = 0});
}

class B extends A {
  B({required super.a});
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            @12
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                optionalNamed default a @19
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                  type: int
                  constantInitializer
                    IntegerLiteral
                      literal: 0 @23
                      staticType: int
        class B @37
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            @53
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                requiredNamed default final hasImplicitType super.a @71
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
                  type: int
                  superConstructorParameter: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                default a @19
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
        class B @37
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 53
              formalParameters
                default super.a @71
                  reference: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            optionalNamed a
              firstFragment: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              type: int
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredNamed final hasImplicitType a
              firstFragment: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
              type: int
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_requiredPositional() async {
    var library = await buildLibrary('''
class A {
  A(int a, double b);
}

class B extends A {
  B(String o1, super.a, String o2, super.b) : super();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            @12
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional a @18
                  type: int
                requiredPositional b @28
                  type: double
        class B @41
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            @57
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                requiredPositional o1 @66
                  type: String
                requiredPositional final hasImplicitType super.a @76
                  type: int
                  superConstructorParameter: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                requiredPositional o2 @86
                  type: String
                requiredPositional final hasImplicitType super.b @96
                  type: double
                  superConstructorParameter: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                a @18
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
                b @28
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::b#element
        class B @41
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 57
              formalParameters
                o1 @66
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o1#element
                super.a @76
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
                o2 @86
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::o2#element
                super.b @96
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::b#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional a
              type: int
            requiredPositional b
              type: double
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional o1
              type: String
            requiredPositional final hasImplicitType a
              type: int
            requiredPositional o2
              type: String
            requiredPositional final hasImplicitType b
              type: double
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_requiredPositional_inferenceOrder() async {
    // It is important that `B` is declared after `C`, so that we check that
    // inference happens in order - first `B`, then `C`.
    var library = await buildLibrary('''
abstract class A {
  A(int a);
}

class C extends B {
  C(super.a);
}

class B extends A {
  B(super.a);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        abstract class A @15
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            @21
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional a @27
                  type: int
        class C @40
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: B
          constructors
            @56
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional final hasImplicitType super.a @64
                  type: int
                  superConstructorParameter: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
              superConstructor: <testLibraryFragment>::@class::B::@constructor::new
        class B @77
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            @93
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                requiredPositional final hasImplicitType super.a @101
                  type: int
                  superConstructorParameter: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @15
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 21
              formalParameters
                a @27
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
        class C @40
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 56
              formalParameters
                super.a @64
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::a#element
              superConstructor: <testLibraryFragment>::@class::B::@constructor::new
        class B @77
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 93
              formalParameters
                super.a @101
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    abstract class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional a
              type: int
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: B
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType a
              type: int
          superConstructor: <testLibraryFragment>::@class::B::@constructor::new#element
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType a
              type: int
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_requiredPositional_inferenceOrder_generic() async {
    // It is important that `C` is declared before `B`, so that we check that
    // inference happens in order - first `B`, then `C`.
    var library = await buildLibrary('''
class A {
  A(int a);
}

class C extends B<String> {
  C(super.a);
}

class B<T> extends A {
  B(super.a);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            @12
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional a @18
                  type: int
        class C @31
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: B<String>
          constructors
            @55
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional final hasImplicitType super.a @63
                  type: int
                  superConstructorParameter: SuperFormalParameterMember
                    base: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a
                    substitution: {T: String}
              superConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::B::@constructor::new
                substitution: {T: String}
        class B @76
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @78
              defaultType: dynamic
          supertype: A
          constructors
            @95
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                requiredPositional final hasImplicitType super.a @103
                  type: int
                  superConstructorParameter: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                a @18
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
        class C @31
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 55
              formalParameters
                super.a @63
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::a#element
              superConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::B::@constructor::new
                substitution: {T: String}
        class B @76
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          typeParameters
            T @78
              element: <not-implemented>
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 95
              formalParameters
                super.a @103
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional a
              type: int
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: B<String>
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType a
              type: int
          superConstructor: <testLibraryFragment>::@class::B::@constructor::new#element
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      typeParameters
        T
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType a
              type: int
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_requiredPositional_unresolved() async {
    var library = await buildLibrary('''
class A {}

class B extends A {
  B(super.a);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
        class B @18
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            @34
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                requiredPositional final hasImplicitType super.a @42
                  type: dynamic
                  superConstructorParameter: <null>
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @18
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 34
              formalParameters
                super.a @42
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType a
              type: dynamic
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_parameters_super_requiredPositional_unresolved2() async {
    var library = await buildLibrary('''
class A {
  A({required int a})
}

class B extends A {
  B(super.a);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            @12
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredNamed default a @28
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                  type: int
        class B @41
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            @57
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                requiredPositional final hasImplicitType super.a @65
                  type: dynamic
                  superConstructorParameter: <null>
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                default a @28
                  reference: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a#element
        class B @41
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 57
              formalParameters
                super.a @65
                  element: <testLibraryFragment>::@class::B::@constructor::new::@parameter::a#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredNamed a
              firstFragment: <testLibraryFragment>::@class::A::@constructor::new::@parameter::a
              type: int
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType a
              type: dynamic
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_params() async {
    var library = await buildLibrary('class C { C(x, int y); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            @10
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional hasImplicitType x @12
                  type: dynamic
                requiredPositional y @19
                  type: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 10
              formalParameters
                x @12
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::x#element
                y @19
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::y#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional hasImplicitType x
              type: dynamic
            requiredPositional y
              type: int
''');
  }

  test_class_constructor_redirected_factory_named() async {
    var library = await buildLibrary('''
class C {
  factory C() = D.named;
  C._();
}
class D extends C {
  D.named() : super._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            factory @20
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              redirectedConstructor: <testLibraryFragment>::@class::D::@constructor::named
            _ @39
              reference: <testLibraryFragment>::@class::C::@constructor::_
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 38
              nameEnd: 40
        class D @52
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          supertype: C
          constructors
            named @70
              reference: <testLibraryFragment>::@class::D::@constructor::named
              enclosingElement3: <testLibraryFragment>::@class::D
              periodOffset: 69
              nameEnd: 75
              superConstructor: <testLibraryFragment>::@class::C::@constructor::_
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 20
              redirectedConstructor: <testLibraryFragment>::@class::D::@constructor::named
            _ @39
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 37
              periodOffset: 38
        class D @52
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            named @70
              reference: <testLibraryFragment>::@class::D::@constructor::named
              element: <testLibraryFragment>::@class::D::@constructor::named#element
              typeName: D
              typeNameOffset: 68
              periodOffset: 69
              superConstructor: <testLibraryFragment>::@class::C::@constructor::_
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: <testLibraryFragment>::@class::D::@constructor::named#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      supertype: C
      constructors
        named
          firstFragment: <testLibraryFragment>::@class::D::@constructor::named
          superConstructor: <testLibraryFragment>::@class::C::@constructor::_#element
''');
  }

  test_class_constructor_redirected_factory_named_generic() async {
    var library = await buildLibrary('''
class C<T, U> {
  factory C() = D<U, T>.named;
  C._();
}
class D<T, U> extends C<U, T> {
  D.named() : super._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              defaultType: dynamic
            covariant U @11
              defaultType: dynamic
          constructors
            factory @26
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              redirectedConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::D::@constructor::named
                substitution: {T: U, U: T}
            _ @51
              reference: <testLibraryFragment>::@class::C::@constructor::_
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 50
              nameEnd: 52
        class D @64
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @66
              defaultType: dynamic
            covariant U @69
              defaultType: dynamic
          supertype: C<U, T>
          constructors
            named @94
              reference: <testLibraryFragment>::@class::D::@constructor::named
              enclosingElement3: <testLibraryFragment>::@class::D
              periodOffset: 93
              nameEnd: 99
              superConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::C::@constructor::_
                substitution: {T: U, U: T}
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
            U @11
              element: <not-implemented>
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 26
              redirectedConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::D::@constructor::named
                substitution: {T: U, U: T}
            _ @51
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 49
              periodOffset: 50
        class D @64
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          typeParameters
            T @66
              element: <not-implemented>
            U @69
              element: <not-implemented>
          constructors
            named @94
              reference: <testLibraryFragment>::@class::D::@constructor::named
              element: <testLibraryFragment>::@class::D::@constructor::named#element
              typeName: D
              typeNameOffset: 92
              periodOffset: 93
              superConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::C::@constructor::_
                substitution: {T: U, U: T}
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: <testLibraryFragment>::@class::D::@constructor::named#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      typeParameters
        T
        U
      supertype: C<U, T>
      constructors
        named
          firstFragment: <testLibraryFragment>::@class::D::@constructor::named
          superConstructor: <testLibraryFragment>::@class::C::@constructor::_#element
''');
  }

  test_class_constructor_redirected_factory_named_generic_viaTypeAlias() async {
    var library = await buildLibrary('''
typedef A<T, U> = C<T, U>;
class B<T, U> {
  factory B() = A<U, T>.named;
  B._();
}
class C<T, U> extends A<U, T> {
  C.named() : super._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class B @33
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @35
              defaultType: dynamic
            covariant U @38
              defaultType: dynamic
          constructors
            factory @53
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              redirectedConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::C::@constructor::named
                substitution: {T: U, U: T}
            _ @78
              reference: <testLibraryFragment>::@class::B::@constructor::_
              enclosingElement3: <testLibraryFragment>::@class::B
              periodOffset: 77
              nameEnd: 79
        class C @91
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @93
              defaultType: dynamic
            covariant U @96
              defaultType: dynamic
          supertype: C<U, T>
            alias: <testLibraryFragment>::@typeAlias::A
              typeArguments
                U
                T
          constructors
            named @121
              reference: <testLibraryFragment>::@class::C::@constructor::named
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 120
              nameEnd: 126
      typeAliases
        A @8
          reference: <testLibraryFragment>::@typeAlias::A
          typeParameters
            covariant T @10
              defaultType: dynamic
            covariant U @13
              defaultType: dynamic
          aliasedType: C<T, U>
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class B @33
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          typeParameters
            T @35
              element: <not-implemented>
            U @38
              element: <not-implemented>
          constructors
            factory new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 53
              redirectedConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::C::@constructor::named
                substitution: {T: U, U: T}
            _ @78
              reference: <testLibraryFragment>::@class::B::@constructor::_
              element: <testLibraryFragment>::@class::B::@constructor::_#element
              typeName: B
              typeNameOffset: 76
              periodOffset: 77
        class C @91
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @93
              element: <not-implemented>
            U @96
              element: <not-implemented>
          constructors
            named @121
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 119
              periodOffset: 120
      typeAliases
        A @8
          reference: <testLibraryFragment>::@typeAlias::A
          element: <testLibrary>::@typeAlias::A
          typeParameters
            T @10
              element: <not-implemented>
            U @13
              element: <not-implemented>
  classes
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      typeParameters
        T
        U
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named#element
        _
          firstFragment: <testLibraryFragment>::@class::B::@constructor::_
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      supertype: C<U, T>
        alias: <testLibraryFragment>::@typeAlias::A
          typeArguments
            U
            T
      constructors
        named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
  typeAliases
    A
      firstFragment: <testLibraryFragment>::@typeAlias::A
      typeParameters
        T
        U
      aliasedType: C<T, U>
''');
  }

  test_class_constructor_redirected_factory_named_imported() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D extends C {
  D.named() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart';
class C {
  factory C() = D.named;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      libraryImports
        package:test/foo.dart
          enclosingElement3: <testLibraryFragment>
      classes
        class C @25
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            factory @39
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::named
            _ @58
              reference: <testLibraryFragment>::@class::C::@constructor::_
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 57
              nameEnd: 59
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart
      classes
        class C @25
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 39
              redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::named
            _ @58
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 56
              periodOffset: 57
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::named#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_named_imported_generic() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D<T, U> extends C<U, T> {
  D.named() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart';
class C<T, U> {
  factory C() = D<U, T>.named;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      libraryImports
        package:test/foo.dart
          enclosingElement3: <testLibraryFragment>
      classes
        class C @25
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @27
              defaultType: dynamic
            covariant U @30
              defaultType: dynamic
          constructors
            factory @45
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              redirectedConstructor: ConstructorMember
                base: package:test/foo.dart::<fragment>::@class::D::@constructor::named
                substitution: {T: U, U: T}
            _ @70
              reference: <testLibraryFragment>::@class::C::@constructor::_
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 69
              nameEnd: 71
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart
      classes
        class C @25
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @27
              element: <not-implemented>
            U @30
              element: <not-implemented>
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 45
              redirectedConstructor: ConstructorMember
                base: package:test/foo.dart::<fragment>::@class::D::@constructor::named
                substitution: {T: U, U: T}
            _ @70
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 68
              periodOffset: 69
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::named#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_named_prefixed() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D extends C {
  D.named() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart' as foo;
class C {
  factory C() = foo.D.named;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      libraryImports
        package:test/foo.dart as foo @21
          enclosingElement3: <testLibraryFragment>
      libraryImportPrefixes
        foo @21
          reference: <testLibraryFragment>::@prefix::foo
          enclosingElement3: <testLibraryFragment>
      classes
        class C @32
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            factory @46
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::named
            _ @69
              reference: <testLibraryFragment>::@class::C::@constructor::_
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 68
              nameEnd: 70
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart as foo @21
      prefixes
        <testLibraryFragment>::@prefix2::foo
          fragments: @21
      classes
        class C @32
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 46
              redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::named
            _ @69
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 67
              periodOffset: 68
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::named#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_named_prefixed_generic() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D<T, U> extends C<U, T> {
  D.named() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart' as foo;
class C<T, U> {
  factory C() = foo.D<U, T>.named;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      libraryImports
        package:test/foo.dart as foo @21
          enclosingElement3: <testLibraryFragment>
      libraryImportPrefixes
        foo @21
          reference: <testLibraryFragment>::@prefix::foo
          enclosingElement3: <testLibraryFragment>
      classes
        class C @32
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @34
              defaultType: dynamic
            covariant U @37
              defaultType: dynamic
          constructors
            factory @52
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              redirectedConstructor: ConstructorMember
                base: package:test/foo.dart::<fragment>::@class::D::@constructor::named
                substitution: {T: U, U: T}
            _ @81
              reference: <testLibraryFragment>::@class::C::@constructor::_
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 80
              nameEnd: 82
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart as foo @21
      prefixes
        <testLibraryFragment>::@prefix2::foo
          fragments: @21
      classes
        class C @32
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @34
              element: <not-implemented>
            U @37
              element: <not-implemented>
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 52
              redirectedConstructor: ConstructorMember
                base: package:test/foo.dart::<fragment>::@class::D::@constructor::named
                substitution: {T: U, U: T}
            _ @81
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 79
              periodOffset: 80
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::named#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_named_unresolved_class() async {
    var library = await buildLibrary('''
class C<E> {
  factory C() = D.named<E>;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant E @8
              defaultType: dynamic
          constructors
            factory @23
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            E @8
              element: <not-implemented>
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 23
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        E
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_redirected_factory_named_unresolved_constructor() async {
    var library = await buildLibrary('''
class D {}
class C<E> {
  factory C() = D.named<E>;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class D @6
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
        class C @17
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant E @19
              defaultType: dynamic
          constructors
            factory @34
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class D @6
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class C @17
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            E @19
              element: <not-implemented>
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 34
  classes
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        E
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_redirected_factory_unnamed() async {
    var library = await buildLibrary('''
class C {
  factory C() = D;
  C._();
}
class D extends C {
  D() : super._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            factory @20
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              redirectedConstructor: <testLibraryFragment>::@class::D::@constructor::new
            _ @33
              reference: <testLibraryFragment>::@class::C::@constructor::_
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 32
              nameEnd: 34
        class D @46
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          supertype: C
          constructors
            @62
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
              superConstructor: <testLibraryFragment>::@class::C::@constructor::_
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 20
              redirectedConstructor: <testLibraryFragment>::@class::D::@constructor::new
            _ @33
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 31
              periodOffset: 32
        class D @46
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
              typeNameOffset: 62
              superConstructor: <testLibraryFragment>::@class::C::@constructor::_
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      supertype: C
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
          superConstructor: <testLibraryFragment>::@class::C::@constructor::_#element
''');
  }

  test_class_constructor_redirected_factory_unnamed_generic() async {
    var library = await buildLibrary('''
class C<T, U> {
  factory C() = D<U, T>;
  C._();
}
class D<T, U> extends C<U, T> {
  D() : super._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              defaultType: dynamic
            covariant U @11
              defaultType: dynamic
          constructors
            factory @26
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              redirectedConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::D::@constructor::new
                substitution: {T: U, U: T}
            _ @45
              reference: <testLibraryFragment>::@class::C::@constructor::_
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 44
              nameEnd: 46
        class D @58
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @60
              defaultType: dynamic
            covariant U @63
              defaultType: dynamic
          supertype: C<U, T>
          constructors
            @86
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
              superConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::C::@constructor::_
                substitution: {T: U, U: T}
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
            U @11
              element: <not-implemented>
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 26
              redirectedConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::D::@constructor::new
                substitution: {T: U, U: T}
            _ @45
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 43
              periodOffset: 44
        class D @58
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          typeParameters
            T @60
              element: <not-implemented>
            U @63
              element: <not-implemented>
          constructors
            new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
              typeNameOffset: 86
              superConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::C::@constructor::_
                substitution: {T: U, U: T}
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      typeParameters
        T
        U
      supertype: C<U, T>
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
          superConstructor: <testLibraryFragment>::@class::C::@constructor::_#element
''');
  }

  test_class_constructor_redirected_factory_unnamed_generic_viaTypeAlias() async {
    var library = await buildLibrary('''
typedef A<T, U> = C<T, U>;
class B<T, U> {
  factory B() = A<U, T>;
  B_();
}
class C<T, U> extends B<U, T> {
  C() : super._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class B @33
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @35
              defaultType: dynamic
            covariant U @38
              defaultType: dynamic
          constructors
            factory @53
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              redirectedConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::C::@constructor::new
                substitution: {T: U, U: T}
          methods
            abstract B_ @70
              reference: <testLibraryFragment>::@class::B::@method::B_
              enclosingElement3: <testLibraryFragment>::@class::B
              returnType: dynamic
        class C @84
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @86
              defaultType: dynamic
            covariant U @89
              defaultType: dynamic
          supertype: B<U, T>
          constructors
            @112
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
      typeAliases
        A @8
          reference: <testLibraryFragment>::@typeAlias::A
          typeParameters
            covariant T @10
              defaultType: dynamic
            covariant U @13
              defaultType: dynamic
          aliasedType: C<T, U>
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class B @33
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          typeParameters
            T @35
              element: <not-implemented>
            U @38
              element: <not-implemented>
          constructors
            factory new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 53
              redirectedConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::C::@constructor::new
                substitution: {T: U, U: T}
          methods
            B_ @70
              reference: <testLibraryFragment>::@class::B::@method::B_
              element: <testLibraryFragment>::@class::B::@method::B_#element
        class C @84
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @86
              element: <not-implemented>
            U @89
              element: <not-implemented>
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 112
      typeAliases
        A @8
          reference: <testLibraryFragment>::@typeAlias::A
          element: <testLibrary>::@typeAlias::A
          typeParameters
            T @10
              element: <not-implemented>
            U @13
              element: <not-implemented>
  classes
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      typeParameters
        T
        U
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new#element
      methods
        abstract B_
          reference: <testLibrary>::@class::B::@method::B_
          firstFragment: <testLibraryFragment>::@class::B::@method::B_
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      supertype: B<U, T>
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
  typeAliases
    A
      firstFragment: <testLibraryFragment>::@typeAlias::A
      typeParameters
        T
        U
      aliasedType: C<T, U>
''');
  }

  test_class_constructor_redirected_factory_unnamed_imported() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D extends C {
  D() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart';
class C {
  factory C() = D;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      libraryImports
        package:test/foo.dart
          enclosingElement3: <testLibraryFragment>
      classes
        class C @25
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            factory @39
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::new
            _ @52
              reference: <testLibraryFragment>::@class::C::@constructor::_
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 51
              nameEnd: 53
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart
      classes
        class C @25
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 39
              redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::new
            _ @52
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 50
              periodOffset: 51
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::new#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_unnamed_imported_generic() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D<T, U> extends C<U, T> {
  D() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart';
class C<T, U> {
  factory C() = D<U, T>;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      libraryImports
        package:test/foo.dart
          enclosingElement3: <testLibraryFragment>
      classes
        class C @25
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @27
              defaultType: dynamic
            covariant U @30
              defaultType: dynamic
          constructors
            factory @45
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              redirectedConstructor: ConstructorMember
                base: package:test/foo.dart::<fragment>::@class::D::@constructor::new
                substitution: {T: U, U: T}
            _ @64
              reference: <testLibraryFragment>::@class::C::@constructor::_
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 63
              nameEnd: 65
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart
      classes
        class C @25
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @27
              element: <not-implemented>
            U @30
              element: <not-implemented>
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 45
              redirectedConstructor: ConstructorMember
                base: package:test/foo.dart::<fragment>::@class::D::@constructor::new
                substitution: {T: U, U: T}
            _ @64
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 62
              periodOffset: 63
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::new#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_unnamed_imported_viaTypeAlias() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
typedef A = B;
class B extends C {
  B() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart';
class C {
  factory C() = A;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      libraryImports
        package:test/foo.dart
          enclosingElement3: <testLibraryFragment>
      classes
        class C @25
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            factory @39
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              redirectedConstructor: package:test/foo.dart::<fragment>::@class::B::@constructor::new
            _ @52
              reference: <testLibraryFragment>::@class::C::@constructor::_
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 51
              nameEnd: 53
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart
      classes
        class C @25
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 39
              redirectedConstructor: package:test/foo.dart::<fragment>::@class::B::@constructor::new
            _ @52
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 50
              periodOffset: 51
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: package:test/foo.dart::<fragment>::@class::B::@constructor::new#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_unnamed_prefixed() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D extends C {
  D() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart' as foo;
class C {
  factory C() = foo.D;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      libraryImports
        package:test/foo.dart as foo @21
          enclosingElement3: <testLibraryFragment>
      libraryImportPrefixes
        foo @21
          reference: <testLibraryFragment>::@prefix::foo
          enclosingElement3: <testLibraryFragment>
      classes
        class C @32
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            factory @46
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::new
            _ @63
              reference: <testLibraryFragment>::@class::C::@constructor::_
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 62
              nameEnd: 64
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart as foo @21
      prefixes
        <testLibraryFragment>::@prefix2::foo
          fragments: @21
      classes
        class C @32
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 46
              redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::new
            _ @63
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 61
              periodOffset: 62
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::new#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_unnamed_prefixed_generic() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
class D<T, U> extends C<U, T> {
  D() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart' as foo;
class C<T, U> {
  factory C() = foo.D<U, T>;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      libraryImports
        package:test/foo.dart as foo @21
          enclosingElement3: <testLibraryFragment>
      libraryImportPrefixes
        foo @21
          reference: <testLibraryFragment>::@prefix::foo
          enclosingElement3: <testLibraryFragment>
      classes
        class C @32
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @34
              defaultType: dynamic
            covariant U @37
              defaultType: dynamic
          constructors
            factory @52
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              redirectedConstructor: ConstructorMember
                base: package:test/foo.dart::<fragment>::@class::D::@constructor::new
                substitution: {T: U, U: T}
            _ @75
              reference: <testLibraryFragment>::@class::C::@constructor::_
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 74
              nameEnd: 76
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart as foo @21
      prefixes
        <testLibraryFragment>::@prefix2::foo
          fragments: @21
      classes
        class C @32
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @34
              element: <not-implemented>
            U @37
              element: <not-implemented>
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 52
              redirectedConstructor: ConstructorMember
                base: package:test/foo.dart::<fragment>::@class::D::@constructor::new
                substitution: {T: U, U: T}
            _ @75
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 73
              periodOffset: 74
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: package:test/foo.dart::<fragment>::@class::D::@constructor::new#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_unnamed_prefixed_viaTypeAlias() async {
    newFile('$testPackageLibPath/foo.dart', '''
import 'test.dart';
typedef A = B;
class B extends C {
  B() : super._();
}
''');
    var library = await buildLibrary('''
import 'foo.dart' as foo;
class C {
  factory C() = foo.A;
  C._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      libraryImports
        package:test/foo.dart as foo @21
          enclosingElement3: <testLibraryFragment>
      libraryImportPrefixes
        foo @21
          reference: <testLibraryFragment>::@prefix::foo
          enclosingElement3: <testLibraryFragment>
      classes
        class C @32
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            factory @46
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              redirectedConstructor: package:test/foo.dart::<fragment>::@class::B::@constructor::new
            _ @63
              reference: <testLibraryFragment>::@class::C::@constructor::_
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 62
              nameEnd: 64
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/foo.dart as foo @21
      prefixes
        <testLibraryFragment>::@prefix2::foo
          fragments: @21
      classes
        class C @32
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 46
              redirectedConstructor: package:test/foo.dart::<fragment>::@class::B::@constructor::new
            _ @63
              reference: <testLibraryFragment>::@class::C::@constructor::_
              element: <testLibraryFragment>::@class::C::@constructor::_#element
              typeName: C
              typeNameOffset: 61
              periodOffset: 62
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: package:test/foo.dart::<fragment>::@class::B::@constructor::new#element
        _
          firstFragment: <testLibraryFragment>::@class::C::@constructor::_
''');
  }

  test_class_constructor_redirected_factory_unnamed_unresolved() async {
    var library = await buildLibrary('''
class C<E> {
  factory C() = D<E>;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant E @8
              defaultType: dynamic
          constructors
            factory @23
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            E @8
              element: <not-implemented>
          constructors
            factory new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 23
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        E
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_redirected_factory_unnamed_viaTypeAlias() async {
    var library = await buildLibrary('''
typedef A = C;
class B {
  factory B() = A;
  B._();
}
class C extends B {
  C() : super._();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class B @21
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          constructors
            factory @35
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new
            _ @48
              reference: <testLibraryFragment>::@class::B::@constructor::_
              enclosingElement3: <testLibraryFragment>::@class::B
              periodOffset: 47
              nameEnd: 49
        class C @61
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: B
          constructors
            @77
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              superConstructor: <testLibraryFragment>::@class::B::@constructor::_
      typeAliases
        A @8
          reference: <testLibraryFragment>::@typeAlias::A
          aliasedType: C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class B @21
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            factory new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 35
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new
            _ @48
              reference: <testLibraryFragment>::@class::B::@constructor::_
              element: <testLibraryFragment>::@class::B::@constructor::_#element
              typeName: B
              typeNameOffset: 46
              periodOffset: 47
        class C @61
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 77
              superConstructor: <testLibraryFragment>::@class::B::@constructor::_
      typeAliases
        A @8
          reference: <testLibraryFragment>::@typeAlias::A
          element: <testLibrary>::@typeAlias::A
  classes
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      constructors
        factory new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new#element
        _
          firstFragment: <testLibraryFragment>::@class::B::@constructor::_
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: B
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::B::@constructor::_#element
  typeAliases
    A
      firstFragment: <testLibraryFragment>::@typeAlias::A
      aliasedType: C
''');
  }

  test_class_constructor_redirected_thisInvocation_named() async {
    var library = await buildLibrary('''
class C {
  const C.named();
  const C() : this.named();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            const named @20
              reference: <testLibraryFragment>::@class::C::@constructor::named
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 19
              nameEnd: 25
            const @37
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              constantInitializers
                RedirectingConstructorInvocation
                  thisKeyword: this @43
                  period: . @47
                  constructorName: SimpleIdentifier
                    token: named @48
                    staticElement: <testLibraryFragment>::@class::C::@constructor::named
                    element: <testLibraryFragment>::@class::C::@constructor::named#element
                    staticType: null
                  argumentList: ArgumentList
                    leftParenthesis: ( @53
                    rightParenthesis: ) @54
                  staticElement: <testLibraryFragment>::@class::C::@constructor::named
                  element: <testLibraryFragment>::@class::C::@constructor::named#element
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const named @20
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 18
              periodOffset: 19
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 37
              constantInitializers
                RedirectingConstructorInvocation
                  thisKeyword: this @43
                  period: . @47
                  constructorName: SimpleIdentifier
                    token: named @48
                    staticElement: <testLibraryFragment>::@class::C::@constructor::named
                    element: <testLibraryFragment>::@class::C::@constructor::named#element
                    staticType: null
                  argumentList: ArgumentList
                    leftParenthesis: ( @53
                    rightParenthesis: ) @54
                  staticElement: <testLibraryFragment>::@class::C::@constructor::named
                  element: <testLibraryFragment>::@class::C::@constructor::named#element
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        const named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named#element
''');
  }

  test_class_constructor_redirected_thisInvocation_named_generic() async {
    var library = await buildLibrary('''
class C<T> {
  const C.named();
  const C() : this.named();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              defaultType: dynamic
          constructors
            const named @23
              reference: <testLibraryFragment>::@class::C::@constructor::named
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 22
              nameEnd: 28
            const @40
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              constantInitializers
                RedirectingConstructorInvocation
                  thisKeyword: this @46
                  period: . @50
                  constructorName: SimpleIdentifier
                    token: named @51
                    staticElement: <testLibraryFragment>::@class::C::@constructor::named
                    element: <testLibraryFragment>::@class::C::@constructor::named#element
                    staticType: null
                  argumentList: ArgumentList
                    leftParenthesis: ( @56
                    rightParenthesis: ) @57
                  staticElement: <testLibraryFragment>::@class::C::@constructor::named
                  element: <testLibraryFragment>::@class::C::@constructor::named#element
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            const named @23
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 21
              periodOffset: 22
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 40
              constantInitializers
                RedirectingConstructorInvocation
                  thisKeyword: this @46
                  period: . @50
                  constructorName: SimpleIdentifier
                    token: named @51
                    staticElement: <testLibraryFragment>::@class::C::@constructor::named
                    element: <testLibraryFragment>::@class::C::@constructor::named#element
                    staticType: null
                  argumentList: ArgumentList
                    leftParenthesis: ( @56
                    rightParenthesis: ) @57
                  staticElement: <testLibraryFragment>::@class::C::@constructor::named
                  element: <testLibraryFragment>::@class::C::@constructor::named#element
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      constructors
        const named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named#element
''');
  }

  test_class_constructor_redirected_thisInvocation_named_notConst() async {
    var library = await buildLibrary('''
class C {
  C.named();
  C() : this.named();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            named @14
              reference: <testLibraryFragment>::@class::C::@constructor::named
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 13
              nameEnd: 19
            @25
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            named @14
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 12
              periodOffset: 13
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 25
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::named#element
''');
  }

  test_class_constructor_redirected_thisInvocation_unnamed() async {
    var library = await buildLibrary('''
class C {
  const C();
  const C.named() : this();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            const @18
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
            const named @33
              reference: <testLibraryFragment>::@class::C::@constructor::named
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 32
              nameEnd: 38
              constantInitializers
                RedirectingConstructorInvocation
                  thisKeyword: this @43
                  argumentList: ArgumentList
                    leftParenthesis: ( @47
                    rightParenthesis: ) @48
                  staticElement: <testLibraryFragment>::@class::C::@constructor::new
                  element: <testLibraryFragment>::@class::C::@constructor::new#element
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 18
            const named @33
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 31
              periodOffset: 32
              constantInitializers
                RedirectingConstructorInvocation
                  thisKeyword: this @43
                  argumentList: ArgumentList
                    leftParenthesis: ( @47
                    rightParenthesis: ) @48
                  staticElement: <testLibraryFragment>::@class::C::@constructor::new
                  element: <testLibraryFragment>::@class::C::@constructor::new#element
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
        const named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new#element
''');
  }

  test_class_constructor_redirected_thisInvocation_unnamed_generic() async {
    var library = await buildLibrary('''
class C<T> {
  const C();
  const C.named() : this();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              defaultType: dynamic
          constructors
            const @21
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
            const named @36
              reference: <testLibraryFragment>::@class::C::@constructor::named
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 35
              nameEnd: 41
              constantInitializers
                RedirectingConstructorInvocation
                  thisKeyword: this @46
                  argumentList: ArgumentList
                    leftParenthesis: ( @50
                    rightParenthesis: ) @51
                  staticElement: <testLibraryFragment>::@class::C::@constructor::new
                  element: <testLibraryFragment>::@class::C::@constructor::new#element
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 21
            const named @36
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 34
              periodOffset: 35
              constantInitializers
                RedirectingConstructorInvocation
                  thisKeyword: this @46
                  argumentList: ArgumentList
                    leftParenthesis: ( @50
                    rightParenthesis: ) @51
                  staticElement: <testLibraryFragment>::@class::C::@constructor::new
                  element: <testLibraryFragment>::@class::C::@constructor::new#element
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
        const named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new#element
''');
  }

  test_class_constructor_redirected_thisInvocation_unnamed_notConst() async {
    var library = await buildLibrary('''
class C {
  C();
  C.named() : this();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            @12
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
            named @21
              reference: <testLibraryFragment>::@class::C::@constructor::named
              enclosingElement3: <testLibraryFragment>::@class::C
              periodOffset: 20
              nameEnd: 26
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 12
            named @21
              reference: <testLibraryFragment>::@class::C::@constructor::named
              element: <testLibraryFragment>::@class::C::@constructor::named#element
              typeName: C
              typeNameOffset: 19
              periodOffset: 20
              redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
        named
          firstFragment: <testLibraryFragment>::@class::C::@constructor::named
          redirectedConstructor: <testLibraryFragment>::@class::C::@constructor::new#element
''');
  }

  test_class_constructor_superConstructor_generic_named() async {
    var library = await buildLibrary('''
class A<T> {
  A.named(T a);
}
class B extends A<int> {
  B() : super.named(0);
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              defaultType: dynamic
          constructors
            named @17
              reference: <testLibraryFragment>::@class::A::@constructor::named
              enclosingElement3: <testLibraryFragment>::@class::A
              periodOffset: 16
              nameEnd: 22
              parameters
                requiredPositional a @25
                  type: T
        class B @37
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A<int>
          constructors
            @58
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              superConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::A::@constructor::named
                substitution: {T: int}
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            named @17
              reference: <testLibraryFragment>::@class::A::@constructor::named
              element: <testLibraryFragment>::@class::A::@constructor::named#element
              typeName: A
              typeNameOffset: 15
              periodOffset: 16
              formalParameters
                a @25
                  element: <testLibraryFragment>::@class::A::@constructor::named::@parameter::a#element
        class B @37
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 58
              superConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::A::@constructor::named
                substitution: {T: int}
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
      constructors
        named
          firstFragment: <testLibraryFragment>::@class::A::@constructor::named
          formalParameters
            requiredPositional a
              type: T
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A<int>
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::named#element
''');
  }

  test_class_constructor_superConstructor_notGeneric_named() async {
    var library = await buildLibrary('''
class A {
  A.named();
}
class B extends A {
  B() : super.named();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            named @14
              reference: <testLibraryFragment>::@class::A::@constructor::named
              enclosingElement3: <testLibraryFragment>::@class::A
              periodOffset: 13
              nameEnd: 19
        class B @31
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            @47
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::named
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            named @14
              reference: <testLibraryFragment>::@class::A::@constructor::named
              element: <testLibraryFragment>::@class::A::@constructor::named#element
              typeName: A
              typeNameOffset: 12
              periodOffset: 13
        class B @31
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 47
              superConstructor: <testLibraryFragment>::@class::A::@constructor::named
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        named
          firstFragment: <testLibraryFragment>::@class::A::@constructor::named
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::named#element
''');
  }

  test_class_constructor_superConstructor_notGeneric_unnamed_explicit() async {
    var library = await buildLibrary('''
class A {}
class B extends A {
  B() : super();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
        class B @17
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            @33
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @17
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 33
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_superConstructor_notGeneric_unnamed_implicit() async {
    var library = await buildLibrary('''
class A {}
class B extends A {
  B();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
        class B @17
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            @33
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @17
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 33
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_superConstructor_notGeneric_unnamed_implicit2() async {
    var library = await buildLibrary('''
class A {}
class B extends A {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
        class B @17
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @17
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_constructor_unnamed_implicit() async {
    var library = await buildLibrary('class C {}');
    configuration.withDisplayName = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              displayName: C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructor_withCycles_const() async {
    var library = await buildLibrary('''
class C {
  final x;
  const C() : x = const D();
}
class D {
  final x;
  const D() : x = const C();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            final x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            const @29
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              constantInitializers
                ConstructorFieldInitializer
                  fieldName: SimpleIdentifier
                    token: x @35
                    staticElement: <testLibraryFragment>::@class::C::@field::x
                    element: <testLibraryFragment>::@class::C::@field::x#element
                    staticType: null
                  equals: = @37
                  expression: InstanceCreationExpression
                    keyword: const @39
                    constructorName: ConstructorName
                      type: NamedType
                        name: D @45
                        element: <testLibraryFragment>::@class::D
                        element2: <testLibrary>::@class::D
                        type: D
                      staticElement: <testLibraryFragment>::@class::D::@constructor::new
                      element: <testLibraryFragment>::@class::D::@constructor::new#element
                    argumentList: ArgumentList
                      leftParenthesis: ( @46
                      rightParenthesis: ) @47
                    staticType: D
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
        class D @58
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          fields
            final x @70
              reference: <testLibraryFragment>::@class::D::@field::x
              enclosingElement3: <testLibraryFragment>::@class::D
              type: dynamic
          constructors
            const @81
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
              constantInitializers
                ConstructorFieldInitializer
                  fieldName: SimpleIdentifier
                    token: x @87
                    staticElement: <testLibraryFragment>::@class::D::@field::x
                    element: <testLibraryFragment>::@class::D::@field::x#element
                    staticType: null
                  equals: = @89
                  expression: InstanceCreationExpression
                    keyword: const @91
                    constructorName: ConstructorName
                      type: NamedType
                        name: C @97
                        element: <testLibraryFragment>::@class::C
                        element2: <testLibrary>::@class::C
                        type: C
                      staticElement: <testLibraryFragment>::@class::C::@constructor::new
                      element: <testLibraryFragment>::@class::C::@constructor::new#element
                    argumentList: ArgumentList
                      leftParenthesis: ( @98
                      rightParenthesis: ) @99
                    staticType: C
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::D::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::D
              returnType: dynamic
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 29
              constantInitializers
                ConstructorFieldInitializer
                  fieldName: SimpleIdentifier
                    token: x @35
                    staticElement: <testLibraryFragment>::@class::C::@field::x
                    element: <testLibraryFragment>::@class::C::@field::x#element
                    staticType: null
                  equals: = @37
                  expression: InstanceCreationExpression
                    keyword: const @39
                    constructorName: ConstructorName
                      type: NamedType
                        name: D @45
                        element: <testLibraryFragment>::@class::D
                        element2: <testLibrary>::@class::D
                        type: D
                      staticElement: <testLibraryFragment>::@class::D::@constructor::new
                      element: <testLibraryFragment>::@class::D::@constructor::new#element
                    argumentList: ArgumentList
                      leftParenthesis: ( @46
                      rightParenthesis: ) @47
                    staticType: D
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
        class D @58
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          fields
            x @70
              reference: <testLibraryFragment>::@class::D::@field::x
              element: <testLibraryFragment>::@class::D::@field::x#element
              getter2: <testLibraryFragment>::@class::D::@getter::x
          constructors
            const new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
              typeNameOffset: 81
              constantInitializers
                ConstructorFieldInitializer
                  fieldName: SimpleIdentifier
                    token: x @87
                    staticElement: <testLibraryFragment>::@class::D::@field::x
                    element: <testLibraryFragment>::@class::D::@field::x#element
                    staticType: null
                  equals: = @89
                  expression: InstanceCreationExpression
                    keyword: const @91
                    constructorName: ConstructorName
                      type: NamedType
                        name: C @97
                        element: <testLibraryFragment>::@class::C
                        element2: <testLibrary>::@class::C
                        type: C
                      staticElement: <testLibraryFragment>::@class::C::@constructor::new
                      element: <testLibraryFragment>::@class::C::@constructor::new#element
                    argumentList: ArgumentList
                      leftParenthesis: ( @98
                      rightParenthesis: ) @99
                    staticType: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::D::@getter::x
              element: <testLibraryFragment>::@class::D::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      fields
        final x
          firstFragment: <testLibraryFragment>::@class::D::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::D::@getter::x#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::D::@getter::x
''');
  }

  test_class_constructor_withCycles_nonConst() async {
    var library = await buildLibrary('''
class C {
  final x;
  C() : x = new D();
}
class D {
  final x;
  D() : x = new C();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            final x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            @23
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
        class D @50
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          fields
            final x @62
              reference: <testLibraryFragment>::@class::D::@field::x
              enclosingElement3: <testLibraryFragment>::@class::D
              type: dynamic
          constructors
            @67
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::D::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::D
              returnType: dynamic
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 23
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
        class D @50
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          fields
            x @62
              reference: <testLibraryFragment>::@class::D::@field::x
              element: <testLibraryFragment>::@class::D::@field::x#element
              getter2: <testLibraryFragment>::@class::D::@getter::x
          constructors
            new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
              typeNameOffset: 67
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::D::@getter::x
              element: <testLibraryFragment>::@class::D::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      fields
        final x
          firstFragment: <testLibraryFragment>::@class::D::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::D::@getter::x#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::D::@getter::x
''');
  }

  test_class_constructors_named() async {
    var library = await buildLibrary('''
class C {
  C.foo();
}
''');
    configuration.withDisplayName = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            foo @14
              reference: <testLibraryFragment>::@class::C::@constructor::foo
              enclosingElement3: <testLibraryFragment>::@class::C
              displayName: C.foo
              periodOffset: 13
              nameEnd: 17
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            foo @14
              reference: <testLibraryFragment>::@class::C::@constructor::foo
              element: <testLibraryFragment>::@class::C::@constructor::foo#element
              typeName: C
              typeNameOffset: 12
              periodOffset: 13
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        foo
          firstFragment: <testLibraryFragment>::@class::C::@constructor::foo
''');
  }

  test_class_constructors_unnamed() async {
    var library = await buildLibrary('''
class C {
  C();
}
''');
    configuration.withDisplayName = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            @12
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              displayName: C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 12
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_constructors_unnamed_new() async {
    var library = await buildLibrary('''
class C {
  C.new();
}
''');
    configuration.withDisplayName = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            @14
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              displayName: C
              periodOffset: 13
              nameEnd: 17
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            new @14
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 12
              periodOffset: 13
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_documented() async {
    var library = await buildLibrary('''
/**
 * Docs
 */
class C {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @22
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          documentationComment: /**\n * Docs\n */
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @22
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /**\n * Docs\n */
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_documented_mix() async {
    var library = await buildLibrary('''
/**
 * aaa
 */
/**
 * bbb
 */
class A {}

/**
 * aaa
 */
/// bbb
/// ccc
class B {}

/// aaa
/// bbb
/**
 * ccc
 */
class C {}

/// aaa
/// bbb
/**
 * ccc
 */
/// ddd
class D {}

/**
 * aaa
 */
// bbb
class E {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @36
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          documentationComment: /**\n * bbb\n */
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
        class B @79
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          documentationComment: /// bbb\n/// ccc
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
        class C @122
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          documentationComment: /**\n * ccc\n */
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
        class D @173
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          documentationComment: /// ddd
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
        class E @207
          reference: <testLibraryFragment>::@class::E
          enclosingElement3: <testLibraryFragment>
          documentationComment: /**\n * aaa\n */
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::E::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::E
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @36
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @79
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
        class C @122
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @173
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @207
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      documentationComment: /**\n * bbb\n */
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      documentationComment: /// bbb\n/// ccc
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /**\n * ccc\n */
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      documentationComment: /// ddd
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      documentationComment: /**\n * aaa\n */
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
''');
  }

  test_class_documented_tripleSlash() async {
    var library = await buildLibrary('''
/// first
/// second
/// third
class C {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @37
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          documentationComment: /// first\n/// second\n/// third
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @37
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /// first\n/// second\n/// third
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_documented_with_references() async {
    var library = await buildLibrary('''
/**
 * Docs referring to [D] and [E]
 */
class C {}

class D {}
class E {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @47
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          documentationComment: /**\n * Docs referring to [D] and [E]\n */
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
        class D @59
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
        class E @70
          reference: <testLibraryFragment>::@class::E
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::E::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::E
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @47
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @59
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @70
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /**\n * Docs referring to [D] and [E]\n */
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
''');
  }

  test_class_documented_with_windows_line_endings() async {
    var library = await buildLibrary('/**\r\n * Docs\r\n */\r\nclass C {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @25
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          documentationComment: /**\n * Docs\n */
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @25
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /**\n * Docs\n */
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_documented_withLeadingNotDocumentation() async {
    var library = await buildLibrary('''
// Extra comment so doc comment offset != 0
/**
 * Docs
 */
class C {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @66
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          documentationComment: /**\n * Docs\n */
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @66
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /**\n * Docs\n */
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_documented_withMetadata() async {
    var library = await buildLibrary('''
/// Comment 1
/// Comment 2
@Annotation()
class BeforeMeta {}

/// Comment 1
/// Comment 2
@Annotation.named()
class BeforeMetaNamed {}

@Annotation()
/// Comment 1
/// Comment 2
class AfterMeta {}

/// Comment 1
@Annotation()
/// Comment 2
class AroundMeta {}

/// Doc comment.
@Annotation()
// Not doc comment.
class DocBeforeMetaNotDocAfter {}

class Annotation {
  const Annotation();
  const Annotation.named();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class BeforeMeta @48
          reference: <testLibraryFragment>::@class::BeforeMeta
          enclosingElement3: <testLibraryFragment>
          documentationComment: /// Comment 1\n/// Comment 2
          metadata
            Annotation
              atSign: @ @28
              name: SimpleIdentifier
                token: Annotation @29
                staticElement: <testLibraryFragment>::@class::Annotation
                element: <testLibrary>::@class::Annotation
                staticType: null
              arguments: ArgumentList
                leftParenthesis: ( @39
                rightParenthesis: ) @40
              element: <testLibraryFragment>::@class::Annotation::@constructor::new
              element2: <testLibraryFragment>::@class::Annotation::@constructor::new#element
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::BeforeMeta::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::BeforeMeta
        class BeforeMetaNamed @117
          reference: <testLibraryFragment>::@class::BeforeMetaNamed
          enclosingElement3: <testLibraryFragment>
          documentationComment: /// Comment 1\n/// Comment 2
          metadata
            Annotation
              atSign: @ @91
              name: PrefixedIdentifier
                prefix: SimpleIdentifier
                  token: Annotation @92
                  staticElement: <testLibraryFragment>::@class::Annotation
                  element: <testLibrary>::@class::Annotation
                  staticType: null
                period: . @102
                identifier: SimpleIdentifier
                  token: named @103
                  staticElement: <testLibraryFragment>::@class::Annotation::@constructor::named
                  element: <testLibraryFragment>::@class::Annotation::@constructor::named#element
                  staticType: null
                staticElement: <testLibraryFragment>::@class::Annotation::@constructor::named
                element: <testLibraryFragment>::@class::Annotation::@constructor::named#element
                staticType: null
              arguments: ArgumentList
                leftParenthesis: ( @108
                rightParenthesis: ) @109
              element: <testLibraryFragment>::@class::Annotation::@constructor::named
              element2: <testLibraryFragment>::@class::Annotation::@constructor::named#element
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::BeforeMetaNamed::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::BeforeMetaNamed
        class AfterMeta @185
          reference: <testLibraryFragment>::@class::AfterMeta
          enclosingElement3: <testLibraryFragment>
          documentationComment: /// Comment 1\n/// Comment 2
          metadata
            Annotation
              atSign: @ @137
              name: SimpleIdentifier
                token: Annotation @138
                staticElement: <testLibraryFragment>::@class::Annotation
                element: <testLibrary>::@class::Annotation
                staticType: null
              arguments: ArgumentList
                leftParenthesis: ( @148
                rightParenthesis: ) @149
              element: <testLibraryFragment>::@class::Annotation::@constructor::new
              element2: <testLibraryFragment>::@class::Annotation::@constructor::new#element
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::AfterMeta::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::AfterMeta
        class AroundMeta @247
          reference: <testLibraryFragment>::@class::AroundMeta
          enclosingElement3: <testLibraryFragment>
          documentationComment: /// Comment 2
          metadata
            Annotation
              atSign: @ @213
              name: SimpleIdentifier
                token: Annotation @214
                staticElement: <testLibraryFragment>::@class::Annotation
                element: <testLibrary>::@class::Annotation
                staticType: null
              arguments: ArgumentList
                leftParenthesis: ( @224
                rightParenthesis: ) @225
              element: <testLibraryFragment>::@class::Annotation::@constructor::new
              element2: <testLibraryFragment>::@class::Annotation::@constructor::new#element
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::AroundMeta::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::AroundMeta
        class DocBeforeMetaNotDocAfter @319
          reference: <testLibraryFragment>::@class::DocBeforeMetaNotDocAfter
          enclosingElement3: <testLibraryFragment>
          documentationComment: /// Doc comment.
          metadata
            Annotation
              atSign: @ @279
              name: SimpleIdentifier
                token: Annotation @280
                staticElement: <testLibraryFragment>::@class::Annotation
                element: <testLibrary>::@class::Annotation
                staticType: null
              arguments: ArgumentList
                leftParenthesis: ( @290
                rightParenthesis: ) @291
              element: <testLibraryFragment>::@class::Annotation::@constructor::new
              element2: <testLibraryFragment>::@class::Annotation::@constructor::new#element
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::DocBeforeMetaNotDocAfter::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::DocBeforeMetaNotDocAfter
        class Annotation @354
          reference: <testLibraryFragment>::@class::Annotation
          enclosingElement3: <testLibraryFragment>
          constructors
            const @375
              reference: <testLibraryFragment>::@class::Annotation::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::Annotation
            const named @408
              reference: <testLibraryFragment>::@class::Annotation::@constructor::named
              enclosingElement3: <testLibraryFragment>::@class::Annotation
              periodOffset: 407
              nameEnd: 413
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class BeforeMeta @48
          reference: <testLibraryFragment>::@class::BeforeMeta
          element: <testLibrary>::@class::BeforeMeta
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::BeforeMeta::@constructor::new
              element: <testLibraryFragment>::@class::BeforeMeta::@constructor::new#element
              typeName: BeforeMeta
        class BeforeMetaNamed @117
          reference: <testLibraryFragment>::@class::BeforeMetaNamed
          element: <testLibrary>::@class::BeforeMetaNamed
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::BeforeMetaNamed::@constructor::new
              element: <testLibraryFragment>::@class::BeforeMetaNamed::@constructor::new#element
              typeName: BeforeMetaNamed
        class AfterMeta @185
          reference: <testLibraryFragment>::@class::AfterMeta
          element: <testLibrary>::@class::AfterMeta
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::AfterMeta::@constructor::new
              element: <testLibraryFragment>::@class::AfterMeta::@constructor::new#element
              typeName: AfterMeta
        class AroundMeta @247
          reference: <testLibraryFragment>::@class::AroundMeta
          element: <testLibrary>::@class::AroundMeta
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::AroundMeta::@constructor::new
              element: <testLibraryFragment>::@class::AroundMeta::@constructor::new#element
              typeName: AroundMeta
        class DocBeforeMetaNotDocAfter @319
          reference: <testLibraryFragment>::@class::DocBeforeMetaNotDocAfter
          element: <testLibrary>::@class::DocBeforeMetaNotDocAfter
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::DocBeforeMetaNotDocAfter::@constructor::new
              element: <testLibraryFragment>::@class::DocBeforeMetaNotDocAfter::@constructor::new#element
              typeName: DocBeforeMetaNotDocAfter
        class Annotation @354
          reference: <testLibraryFragment>::@class::Annotation
          element: <testLibrary>::@class::Annotation
          constructors
            const new
              reference: <testLibraryFragment>::@class::Annotation::@constructor::new
              element: <testLibraryFragment>::@class::Annotation::@constructor::new#element
              typeName: Annotation
              typeNameOffset: 375
            const named @408
              reference: <testLibraryFragment>::@class::Annotation::@constructor::named
              element: <testLibraryFragment>::@class::Annotation::@constructor::named#element
              typeName: Annotation
              typeNameOffset: 397
              periodOffset: 407
  classes
    class BeforeMeta
      reference: <testLibrary>::@class::BeforeMeta
      firstFragment: <testLibraryFragment>::@class::BeforeMeta
      documentationComment: /// Comment 1\n/// Comment 2
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::BeforeMeta::@constructor::new
    class BeforeMetaNamed
      reference: <testLibrary>::@class::BeforeMetaNamed
      firstFragment: <testLibraryFragment>::@class::BeforeMetaNamed
      documentationComment: /// Comment 1\n/// Comment 2
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::BeforeMetaNamed::@constructor::new
    class AfterMeta
      reference: <testLibrary>::@class::AfterMeta
      firstFragment: <testLibraryFragment>::@class::AfterMeta
      documentationComment: /// Comment 1\n/// Comment 2
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::AfterMeta::@constructor::new
    class AroundMeta
      reference: <testLibrary>::@class::AroundMeta
      firstFragment: <testLibraryFragment>::@class::AroundMeta
      documentationComment: /// Comment 2
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::AroundMeta::@constructor::new
    class DocBeforeMetaNotDocAfter
      reference: <testLibrary>::@class::DocBeforeMetaNotDocAfter
      firstFragment: <testLibraryFragment>::@class::DocBeforeMetaNotDocAfter
      documentationComment: /// Doc comment.
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::DocBeforeMetaNotDocAfter::@constructor::new
    class Annotation
      reference: <testLibrary>::@class::Annotation
      firstFragment: <testLibraryFragment>::@class::Annotation
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::Annotation::@constructor::new
        const named
          firstFragment: <testLibraryFragment>::@class::Annotation::@constructor::named
''');
  }

  test_class_field_abstract() async {
    var library = await buildLibrary('''
abstract class C {
  abstract int i;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        abstract class C @15
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            abstract i @34
              reference: <testLibraryFragment>::@class::C::@field::i
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic abstract get i @-1
              reference: <testLibraryFragment>::@class::C::@getter::i
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic abstract set i= @-1
              reference: <testLibraryFragment>::@class::C::@setter::i
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _i @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @15
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            i @34
              reference: <testLibraryFragment>::@class::C::@field::i
              element: <testLibraryFragment>::@class::C::@field::i#element
              getter2: <testLibraryFragment>::@class::C::@getter::i
              setter2: <testLibraryFragment>::@class::C::@setter::i
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get i
              reference: <testLibraryFragment>::@class::C::@getter::i
              element: <testLibraryFragment>::@class::C::@getter::i#element
          setters
            synthetic set i
              reference: <testLibraryFragment>::@class::C::@setter::i
              element: <testLibraryFragment>::@class::C::@setter::i#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::i::@parameter::_i#element
  classes
    abstract class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        abstract i
          firstFragment: <testLibraryFragment>::@class::C::@field::i
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::i#element
          setter: <testLibraryFragment>::@class::C::@setter::i#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic abstract get i
          firstFragment: <testLibraryFragment>::@class::C::@getter::i
      setters
        synthetic abstract set i
          firstFragment: <testLibraryFragment>::@class::C::@setter::i
          formalParameters
            requiredPositional _i
              type: int
''');
  }

  test_class_field_const() async {
    var library = await buildLibrary('class C { static const int i = 0; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            static const i @27
              reference: <testLibraryFragment>::@class::C::@field::i
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              shouldUseTypeForInitializerInference: true
              constantInitializer
                IntegerLiteral
                  literal: 0 @31
                  staticType: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic static get i @-1
              reference: <testLibraryFragment>::@class::C::@getter::i
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer i @27
              reference: <testLibraryFragment>::@class::C::@field::i
              element: <testLibraryFragment>::@class::C::@field::i#element
              getter2: <testLibraryFragment>::@class::C::@getter::i
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get i
              reference: <testLibraryFragment>::@class::C::@getter::i
              element: <testLibraryFragment>::@class::C::@getter::i#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        static const hasInitializer i
          firstFragment: <testLibraryFragment>::@class::C::@field::i
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::i#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic static get i
          firstFragment: <testLibraryFragment>::@class::C::@getter::i
''');
  }

  test_class_field_const_late() async {
    var library =
        await buildLibrary('class C { static late const int i = 0; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            static late const i @32
              reference: <testLibraryFragment>::@class::C::@field::i
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              shouldUseTypeForInitializerInference: true
              constantInitializer
                IntegerLiteral
                  literal: 0 @36
                  staticType: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic static get i @-1
              reference: <testLibraryFragment>::@class::C::@getter::i
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer i @32
              reference: <testLibraryFragment>::@class::C::@field::i
              element: <testLibraryFragment>::@class::C::@field::i#element
              getter2: <testLibraryFragment>::@class::C::@getter::i
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get i
              reference: <testLibraryFragment>::@class::C::@getter::i
              element: <testLibraryFragment>::@class::C::@getter::i#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        static late const hasInitializer i
          firstFragment: <testLibraryFragment>::@class::C::@field::i
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::i#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic static get i
          firstFragment: <testLibraryFragment>::@class::C::@getter::i
''');
  }

  test_class_field_covariant() async {
    var library = await buildLibrary('''
class C {
  covariant int x;
}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            covariant x @26
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional covariant _x @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @26
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        covariant x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional covariant _x
              type: int
''');
  }

  test_class_field_documented() async {
    var library = await buildLibrary('''
class C {
  /**
   * Docs
   */
  var x;
}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @38
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              documentationComment: /**\n   * Docs\n   */
              type: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: dynamic
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @38
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
''');
  }

  test_class_field_duplicate_getter() async {
    var library = await buildLibrary('''
class C {
  int foo = 0;
  int get foo => 0;
}
''');
    configuration
      ..withAugmentedWithoutAugmentation = true
      ..withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            foo @16
              reference: <testLibraryFragment>::@class::C::@field::foo::@def::0
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_0
              getter: getter_0
              setter: setter_0
            synthetic foo @-1
              reference: <testLibraryFragment>::@class::C::@field::foo::@def::1
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              id: field_1
              getter: getter_1
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get foo @-1
              reference: <testLibraryFragment>::@class::C::@getter::foo::@def::0
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
              id: getter_0
              variable: field_0
            synthetic set foo= @-1
              reference: <testLibraryFragment>::@class::C::@setter::foo
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _foo @-1
                  type: int
              returnType: void
              id: setter_0
              variable: field_0
            get foo @35
              reference: <testLibraryFragment>::@class::C::@getter::foo::@def::1
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
              id: getter_1
              variable: field_1
          augmented
            fields
              <testLibraryFragment>::@class::C::@field::foo::@def::0
              <testLibraryFragment>::@class::C::@field::foo::@def::1
            constructors
              <testLibraryFragment>::@class::C::@constructor::new
            accessors
              <testLibraryFragment>::@class::C::@getter::foo::@def::0
              <testLibraryFragment>::@class::C::@getter::foo::@def::1
              <testLibraryFragment>::@class::C::@setter::foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer foo @16
              reference: <testLibraryFragment>::@class::C::@field::foo::@def::0
              element: <testLibraryFragment>::@class::C::@field::foo::@def::0#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo::@def::0
              setter2: <testLibraryFragment>::@class::C::@setter::foo
            synthetic foo
              reference: <testLibraryFragment>::@class::C::@field::foo::@def::1
              element: <testLibraryFragment>::@class::C::@field::foo::@def::1#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo::@def::1
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::C::@getter::foo::@def::0
              element: <testLibraryFragment>::@class::C::@getter::foo::@def::0#element
            get foo @35
              reference: <testLibraryFragment>::@class::C::@getter::foo::@def::1
              element: <testLibraryFragment>::@class::C::@getter::foo::@def::1#element
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::C::@setter::foo
              element: <testLibraryFragment>::@class::C::@setter::foo#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::foo::@parameter::_foo#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo::@def::0
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::foo::@def::0#element
          setter: <testLibraryFragment>::@class::C::@setter::foo#element
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo::@def::1
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::foo::@def::1#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo::@def::0
        get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo::@def::1
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::C::@setter::foo
          formalParameters
            requiredPositional _foo
              type: int
''');
  }

  test_class_field_duplicate_setter() async {
    var library = await buildLibrary('''
class C {
  int foo = 0;
  set foo(int _) {}
}
''');
    configuration
      ..withAugmentedWithoutAugmentation = true
      ..withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            foo @16
              reference: <testLibraryFragment>::@class::C::@field::foo::@def::0
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              shouldUseTypeForInitializerInference: true
              id: field_0
              getter: getter_0
              setter: setter_0
            synthetic foo @-1
              reference: <testLibraryFragment>::@class::C::@field::foo::@def::1
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              id: field_1
              setter: setter_1
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get foo @-1
              reference: <testLibraryFragment>::@class::C::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
              id: getter_0
              variable: field_0
            synthetic set foo= @-1
              reference: <testLibraryFragment>::@class::C::@setter::foo::@def::0
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _foo @-1
                  type: int
              returnType: void
              id: setter_0
              variable: field_0
            set foo= @31
              reference: <testLibraryFragment>::@class::C::@setter::foo::@def::1
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _ @39
                  type: int
              returnType: void
              id: setter_1
              variable: field_1
          augmented
            fields
              <testLibraryFragment>::@class::C::@field::foo::@def::0
              <testLibraryFragment>::@class::C::@field::foo::@def::1
            constructors
              <testLibraryFragment>::@class::C::@constructor::new
            accessors
              <testLibraryFragment>::@class::C::@getter::foo
              <testLibraryFragment>::@class::C::@setter::foo::@def::0
              <testLibraryFragment>::@class::C::@setter::foo::@def::1
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer foo @16
              reference: <testLibraryFragment>::@class::C::@field::foo::@def::0
              element: <testLibraryFragment>::@class::C::@field::foo::@def::0#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo
              setter2: <testLibraryFragment>::@class::C::@setter::foo::@def::0
            synthetic foo
              reference: <testLibraryFragment>::@class::C::@field::foo::@def::1
              element: <testLibraryFragment>::@class::C::@field::foo::@def::1#element
              setter2: <testLibraryFragment>::@class::C::@setter::foo::@def::1
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::C::@getter::foo
              element: <testLibraryFragment>::@class::C::@getter::foo#element
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::C::@setter::foo::@def::0
              element: <testLibraryFragment>::@class::C::@setter::foo::@def::0#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::foo::@def::0::@parameter::_foo#element
            set foo @31
              reference: <testLibraryFragment>::@class::C::@setter::foo::@def::1
              element: <testLibraryFragment>::@class::C::@setter::foo::@def::1#element
              formalParameters
                _ @39
                  element: <testLibraryFragment>::@class::C::@setter::foo::@def::1::@parameter::_#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo::@def::0
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::foo#element
          setter: <testLibraryFragment>::@class::C::@setter::foo::@def::0#element
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo::@def::1
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::foo::@def::1#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::C::@setter::foo::@def::0
          formalParameters
            requiredPositional _foo
              type: int
        set foo
          firstFragment: <testLibraryFragment>::@class::C::@setter::foo::@def::1
          formalParameters
            requiredPositional _
              type: int
''');
  }

  test_class_field_external() async {
    var library = await buildLibrary('''
abstract class C {
  external int i;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        abstract class C @15
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            external i @34
              reference: <testLibraryFragment>::@class::C::@field::i
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get i @-1
              reference: <testLibraryFragment>::@class::C::@getter::i
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic set i= @-1
              reference: <testLibraryFragment>::@class::C::@setter::i
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _i @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @15
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            i @34
              reference: <testLibraryFragment>::@class::C::@field::i
              element: <testLibraryFragment>::@class::C::@field::i#element
              getter2: <testLibraryFragment>::@class::C::@getter::i
              setter2: <testLibraryFragment>::@class::C::@setter::i
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get i
              reference: <testLibraryFragment>::@class::C::@getter::i
              element: <testLibraryFragment>::@class::C::@getter::i#element
          setters
            synthetic set i
              reference: <testLibraryFragment>::@class::C::@setter::i
              element: <testLibraryFragment>::@class::C::@setter::i#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::i::@parameter::_i#element
  classes
    abstract class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        external i
          firstFragment: <testLibraryFragment>::@class::C::@field::i
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::i#element
          setter: <testLibraryFragment>::@class::C::@setter::i#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get i
          firstFragment: <testLibraryFragment>::@class::C::@getter::i
      setters
        synthetic set i
          firstFragment: <testLibraryFragment>::@class::C::@setter::i
          formalParameters
            requiredPositional _i
              type: int
''');
  }

  test_class_field_final_hasInitializer_hasConstConstructor() async {
    var library = await buildLibrary('''
class C {
  final x = 42;
  const C();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            final x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              shouldUseTypeForInitializerInference: false
              constantInitializer
                IntegerLiteral
                  literal: 42 @22
                  staticType: int
          constructors
            const @34
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 34
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final hasInitializer x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
''');
  }

  test_class_field_final_hasInitializer_hasConstConstructor_genericFunctionType() async {
    var library = await buildLibrary('''
class A<T> {
  const A();
}
class B {
  final f = const A<int Function(double a)>();
  const B();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              defaultType: dynamic
          constructors
            const @21
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
        class B @34
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          fields
            final f @46
              reference: <testLibraryFragment>::@class::B::@field::f
              enclosingElement3: <testLibraryFragment>::@class::B
              type: A<int Function(double)>
              shouldUseTypeForInitializerInference: false
              constantInitializer
                InstanceCreationExpression
                  keyword: const @50
                  constructorName: ConstructorName
                    type: NamedType
                      name: A @56
                      typeArguments: TypeArgumentList
                        leftBracket: < @57
                        arguments
                          GenericFunctionType
                            returnType: NamedType
                              name: int @58
                              element: dart:core::<fragment>::@class::int
                              element2: dart:core::@class::int
                              type: int
                            functionKeyword: Function @62
                            parameters: FormalParameterList
                              leftParenthesis: ( @70
                              parameter: SimpleFormalParameter
                                type: NamedType
                                  name: double @71
                                  element: dart:core::<fragment>::@class::double
                                  element2: dart:core::@class::double
                                  type: double
                                name: a @78
                                declaredElement: a@78
                                  type: double
                              rightParenthesis: ) @79
                            declaredElement: GenericFunctionTypeElement
                              parameters
                                a
                                  kind: required positional
                                  type: double
                              returnType: int
                              type: int Function(double)
                            type: int Function(double)
                        rightBracket: > @80
                      element: <testLibraryFragment>::@class::A
                      element2: <testLibrary>::@class::A
                      type: A<int Function(double)>
                    staticElement: ConstructorMember
                      base: <testLibraryFragment>::@class::A::@constructor::new
                      substitution: {T: int Function(double)}
                    element: <testLibraryFragment>::@class::A::@constructor::new#element
                  argumentList: ArgumentList
                    leftParenthesis: ( @81
                    rightParenthesis: ) @82
                  staticType: A<int Function(double)>
          constructors
            const @93
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
          accessors
            synthetic get f @-1
              reference: <testLibraryFragment>::@class::B::@getter::f
              enclosingElement3: <testLibraryFragment>::@class::B
              returnType: A<int Function(double)>
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 21
        class B @34
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          fields
            hasInitializer f @46
              reference: <testLibraryFragment>::@class::B::@field::f
              element: <testLibraryFragment>::@class::B::@field::f#element
              getter2: <testLibraryFragment>::@class::B::@getter::f
          constructors
            const new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 93
          getters
            synthetic get f
              reference: <testLibraryFragment>::@class::B::@getter::f
              element: <testLibraryFragment>::@class::B::@getter::f#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      fields
        final hasInitializer f
          firstFragment: <testLibraryFragment>::@class::B::@field::f
          type: A<int Function(double)>
          getter: <testLibraryFragment>::@class::B::@getter::f#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
      getters
        synthetic get f
          firstFragment: <testLibraryFragment>::@class::B::@getter::f
''');
  }

  test_class_field_final_hasInitializer_noConstConstructor() async {
    var library = await buildLibrary('''
class C {
  final x = 42;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            final x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              shouldUseTypeForInitializerInference: false
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final hasInitializer x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
''');
  }

  test_class_field_final_withSetter() async {
    var library = await buildLibrary(r'''
class A {
  final int foo;
  A(this.foo);
  set foo(int newValue) {}
}
''');
    configuration.withPropertyLinking = true;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final foo @22
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              id: field_0
              getter: getter_0
              setter: setter_0
          constructors
            @29
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional final hasImplicitType this.foo @36
                  type: int
                  field: <testLibraryFragment>::@class::A::@field::foo
          accessors
            synthetic get foo @-1
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
              id: getter_0
              variable: field_0
            set foo= @48
              reference: <testLibraryFragment>::@class::A::@setter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional newValue @56
                  type: int
              returnType: void
              id: setter_0
              variable: field_0
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            foo @22
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
              setter2: <testLibraryFragment>::@class::A::@setter::foo
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 29
              formalParameters
                this.foo @36
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::foo#element
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
          setters
            set foo @48
              reference: <testLibraryFragment>::@class::A::@setter::foo
              element: <testLibraryFragment>::@class::A::@setter::foo#element
              formalParameters
                newValue @56
                  element: <testLibraryFragment>::@class::A::@setter::foo::@parameter::newValue#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
          setter: <testLibraryFragment>::@class::A::@setter::foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType foo
              type: int
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
      setters
        set foo
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo
          formalParameters
            requiredPositional newValue
              type: int
''');
  }

  test_class_field_formal_param_inferred_type_implicit() async {
    var library = await buildLibrary('class C extends D { var v; C(this.v); }'
        ' abstract class D { int get v; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: D
          fields
            v @24
              reference: <testLibraryFragment>::@class::C::@field::v
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            @27
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional final hasImplicitType this.v @34
                  type: int
                  field: <testLibraryFragment>::@class::C::@field::v
              superConstructor: <testLibraryFragment>::@class::D::@constructor::new
          accessors
            synthetic get v @-1
              reference: <testLibraryFragment>::@class::C::@getter::v
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic set v= @-1
              reference: <testLibraryFragment>::@class::C::@setter::v
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _v @-1
                  type: int
              returnType: void
        abstract class D @55
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic v @-1
              reference: <testLibraryFragment>::@class::D::@field::v
              enclosingElement3: <testLibraryFragment>::@class::D
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
          accessors
            abstract get v @67
              reference: <testLibraryFragment>::@class::D::@getter::v
              enclosingElement3: <testLibraryFragment>::@class::D
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            v @24
              reference: <testLibraryFragment>::@class::C::@field::v
              element: <testLibraryFragment>::@class::C::@field::v#element
              getter2: <testLibraryFragment>::@class::C::@getter::v
              setter2: <testLibraryFragment>::@class::C::@setter::v
          constructors
            new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 27
              formalParameters
                this.v @34
                  element: <testLibraryFragment>::@class::C::@constructor::new::@parameter::v#element
              superConstructor: <testLibraryFragment>::@class::D::@constructor::new
          getters
            synthetic get v
              reference: <testLibraryFragment>::@class::C::@getter::v
              element: <testLibraryFragment>::@class::C::@getter::v#element
          setters
            synthetic set v
              reference: <testLibraryFragment>::@class::C::@setter::v
              element: <testLibraryFragment>::@class::C::@setter::v#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::v::@parameter::_v#element
        class D @55
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          fields
            synthetic v
              reference: <testLibraryFragment>::@class::D::@field::v
              element: <testLibraryFragment>::@class::D::@field::v#element
              getter2: <testLibraryFragment>::@class::D::@getter::v
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
          getters
            get v @67
              reference: <testLibraryFragment>::@class::D::@getter::v
              element: <testLibraryFragment>::@class::D::@getter::v#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D
      fields
        v
          firstFragment: <testLibraryFragment>::@class::C::@field::v
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::v#element
          setter: <testLibraryFragment>::@class::C::@setter::v#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType v
              type: int
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
      getters
        synthetic get v
          firstFragment: <testLibraryFragment>::@class::C::@getter::v
      setters
        synthetic set v
          firstFragment: <testLibraryFragment>::@class::C::@setter::v
          formalParameters
            requiredPositional _v
              type: int
    abstract class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      fields
        synthetic v
          firstFragment: <testLibraryFragment>::@class::D::@field::v
          type: int
          getter: <testLibraryFragment>::@class::D::@getter::v#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
      getters
        abstract get v
          firstFragment: <testLibraryFragment>::@class::D::@getter::v
''');
  }

  test_class_field_implicit_type() async {
    var library = await buildLibrary('class C { var x; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: dynamic
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
''');
  }

  test_class_field_implicit_type_late() async {
    var library = await buildLibrary('class C { late var x; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            late x @19
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: dynamic
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            x @19
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        late x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: dynamic
''');
  }

  test_class_field_inferred_type_nonStatic_explicit_initialized() async {
    var library = await buildLibrary('class C { num v = 0; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            v @14
              reference: <testLibraryFragment>::@class::C::@field::v
              enclosingElement3: <testLibraryFragment>::@class::C
              type: num
              shouldUseTypeForInitializerInference: true
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get v @-1
              reference: <testLibraryFragment>::@class::C::@getter::v
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: num
            synthetic set v= @-1
              reference: <testLibraryFragment>::@class::C::@setter::v
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _v @-1
                  type: num
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer v @14
              reference: <testLibraryFragment>::@class::C::@field::v
              element: <testLibraryFragment>::@class::C::@field::v#element
              getter2: <testLibraryFragment>::@class::C::@getter::v
              setter2: <testLibraryFragment>::@class::C::@setter::v
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get v
              reference: <testLibraryFragment>::@class::C::@getter::v
              element: <testLibraryFragment>::@class::C::@getter::v#element
          setters
            synthetic set v
              reference: <testLibraryFragment>::@class::C::@setter::v
              element: <testLibraryFragment>::@class::C::@setter::v#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::v::@parameter::_v#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        hasInitializer v
          firstFragment: <testLibraryFragment>::@class::C::@field::v
          type: num
          getter: <testLibraryFragment>::@class::C::@getter::v#element
          setter: <testLibraryFragment>::@class::C::@setter::v#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get v
          firstFragment: <testLibraryFragment>::@class::C::@getter::v
      setters
        synthetic set v
          firstFragment: <testLibraryFragment>::@class::C::@setter::v
          formalParameters
            requiredPositional _v
              type: num
''');
  }

  test_class_field_inferred_type_nonStatic_implicit_initialized() async {
    var library = await buildLibrary('class C { var v = 0; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            v @14
              reference: <testLibraryFragment>::@class::C::@field::v
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              shouldUseTypeForInitializerInference: false
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get v @-1
              reference: <testLibraryFragment>::@class::C::@getter::v
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic set v= @-1
              reference: <testLibraryFragment>::@class::C::@setter::v
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _v @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer v @14
              reference: <testLibraryFragment>::@class::C::@field::v
              element: <testLibraryFragment>::@class::C::@field::v#element
              getter2: <testLibraryFragment>::@class::C::@getter::v
              setter2: <testLibraryFragment>::@class::C::@setter::v
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get v
              reference: <testLibraryFragment>::@class::C::@getter::v
              element: <testLibraryFragment>::@class::C::@getter::v#element
          setters
            synthetic set v
              reference: <testLibraryFragment>::@class::C::@setter::v
              element: <testLibraryFragment>::@class::C::@setter::v#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::v::@parameter::_v#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        hasInitializer v
          firstFragment: <testLibraryFragment>::@class::C::@field::v
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::v#element
          setter: <testLibraryFragment>::@class::C::@setter::v#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get v
          firstFragment: <testLibraryFragment>::@class::C::@getter::v
      setters
        synthetic set v
          firstFragment: <testLibraryFragment>::@class::C::@setter::v
          formalParameters
            requiredPositional _v
              type: int
''');
  }

  test_class_field_inferred_type_nonStatic_implicit_uninitialized() async {
    var library = await buildLibrary(
        'class C extends D { var v; } abstract class D { int get v; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: D
          fields
            v @24
              reference: <testLibraryFragment>::@class::C::@field::v
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              superConstructor: <testLibraryFragment>::@class::D::@constructor::new
          accessors
            synthetic get v @-1
              reference: <testLibraryFragment>::@class::C::@getter::v
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic set v= @-1
              reference: <testLibraryFragment>::@class::C::@setter::v
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _v @-1
                  type: int
              returnType: void
        abstract class D @44
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic v @-1
              reference: <testLibraryFragment>::@class::D::@field::v
              enclosingElement3: <testLibraryFragment>::@class::D
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
          accessors
            abstract get v @56
              reference: <testLibraryFragment>::@class::D::@getter::v
              enclosingElement3: <testLibraryFragment>::@class::D
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            v @24
              reference: <testLibraryFragment>::@class::C::@field::v
              element: <testLibraryFragment>::@class::C::@field::v#element
              getter2: <testLibraryFragment>::@class::C::@getter::v
              setter2: <testLibraryFragment>::@class::C::@setter::v
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              superConstructor: <testLibraryFragment>::@class::D::@constructor::new
          getters
            synthetic get v
              reference: <testLibraryFragment>::@class::C::@getter::v
              element: <testLibraryFragment>::@class::C::@getter::v#element
          setters
            synthetic set v
              reference: <testLibraryFragment>::@class::C::@setter::v
              element: <testLibraryFragment>::@class::C::@setter::v#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::v::@parameter::_v#element
        class D @44
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          fields
            synthetic v
              reference: <testLibraryFragment>::@class::D::@field::v
              element: <testLibraryFragment>::@class::D::@field::v#element
              getter2: <testLibraryFragment>::@class::D::@getter::v
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
          getters
            get v @56
              reference: <testLibraryFragment>::@class::D::@getter::v
              element: <testLibraryFragment>::@class::D::@getter::v#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D
      fields
        v
          firstFragment: <testLibraryFragment>::@class::C::@field::v
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::v#element
          setter: <testLibraryFragment>::@class::C::@setter::v#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
      getters
        synthetic get v
          firstFragment: <testLibraryFragment>::@class::C::@getter::v
      setters
        synthetic set v
          firstFragment: <testLibraryFragment>::@class::C::@setter::v
          formalParameters
            requiredPositional _v
              type: int
    abstract class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      fields
        synthetic v
          firstFragment: <testLibraryFragment>::@class::D::@field::v
          type: int
          getter: <testLibraryFragment>::@class::D::@getter::v#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
      getters
        abstract get v
          firstFragment: <testLibraryFragment>::@class::D::@getter::v
''');
  }

  test_class_field_inferred_type_nonStatic_inherited_resolveInitializer() async {
    var library = await buildLibrary(r'''
const a = 0;
abstract class A {
  const A();
  List<int> get f;
}
class B extends A {
  const B();
  final f = [a];
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        abstract class A @28
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic f @-1
              reference: <testLibraryFragment>::@class::A::@field::f
              enclosingElement3: <testLibraryFragment>::@class::A
              type: List<int>
          constructors
            const @40
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            abstract get f @61
              reference: <testLibraryFragment>::@class::A::@getter::f
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: List<int>
        class B @72
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          fields
            final f @107
              reference: <testLibraryFragment>::@class::B::@field::f
              enclosingElement3: <testLibraryFragment>::@class::B
              type: List<int>
              shouldUseTypeForInitializerInference: true
              constantInitializer
                ListLiteral
                  leftBracket: [ @111
                  elements
                    SimpleIdentifier
                      token: a @112
                      staticElement: <testLibraryFragment>::@getter::a
                      element: <testLibraryFragment>::@getter::a#element
                      staticType: int
                  rightBracket: ] @113
                  staticType: List<int>
          constructors
            const @94
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
          accessors
            synthetic get f @-1
              reference: <testLibraryFragment>::@class::B::@getter::f
              enclosingElement3: <testLibraryFragment>::@class::B
              returnType: List<int>
      topLevelVariables
        static const a @6
          reference: <testLibraryFragment>::@topLevelVariable::a
          enclosingElement3: <testLibraryFragment>
          type: int
          shouldUseTypeForInitializerInference: false
          constantInitializer
            IntegerLiteral
              literal: 0 @10
              staticType: int
      accessors
        synthetic static get a @-1
          reference: <testLibraryFragment>::@getter::a
          enclosingElement3: <testLibraryFragment>
          returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @28
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic f
              reference: <testLibraryFragment>::@class::A::@field::f
              element: <testLibraryFragment>::@class::A::@field::f#element
              getter2: <testLibraryFragment>::@class::A::@getter::f
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 40
          getters
            get f @61
              reference: <testLibraryFragment>::@class::A::@getter::f
              element: <testLibraryFragment>::@class::A::@getter::f#element
        class B @72
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          fields
            hasInitializer f @107
              reference: <testLibraryFragment>::@class::B::@field::f
              element: <testLibraryFragment>::@class::B::@field::f#element
              getter2: <testLibraryFragment>::@class::B::@getter::f
          constructors
            const new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 94
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
          getters
            synthetic get f
              reference: <testLibraryFragment>::@class::B::@getter::f
              element: <testLibraryFragment>::@class::B::@getter::f#element
      topLevelVariables
        hasInitializer a @6
          reference: <testLibraryFragment>::@topLevelVariable::a
          element: <testLibrary>::@topLevelVariable::a
          getter2: <testLibraryFragment>::@getter::a
      getters
        synthetic get a
          reference: <testLibraryFragment>::@getter::a
          element: <testLibraryFragment>::@getter::a#element
  classes
    abstract class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic f
          firstFragment: <testLibraryFragment>::@class::A::@field::f
          type: List<int>
          getter: <testLibraryFragment>::@class::A::@getter::f#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        abstract get f
          firstFragment: <testLibraryFragment>::@class::A::@getter::f
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      fields
        final hasInitializer f
          firstFragment: <testLibraryFragment>::@class::B::@field::f
          type: List<int>
          getter: <testLibraryFragment>::@class::B::@getter::f#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
      getters
        synthetic get f
          firstFragment: <testLibraryFragment>::@class::B::@getter::f
  topLevelVariables
    const hasInitializer a
      reference: <testLibrary>::@topLevelVariable::a
      firstFragment: <testLibraryFragment>::@topLevelVariable::a
      type: int
      getter: <testLibraryFragment>::@getter::a#element
  getters
    synthetic static get a
      firstFragment: <testLibraryFragment>::@getter::a
''');
  }

  test_class_field_inferred_type_static_implicit_initialized() async {
    var library = await buildLibrary('class C { static var v = 0; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            static v @21
              reference: <testLibraryFragment>::@class::C::@field::v
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              shouldUseTypeForInitializerInference: false
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic static get v @-1
              reference: <testLibraryFragment>::@class::C::@getter::v
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic static set v= @-1
              reference: <testLibraryFragment>::@class::C::@setter::v
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _v @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer v @21
              reference: <testLibraryFragment>::@class::C::@field::v
              element: <testLibraryFragment>::@class::C::@field::v#element
              getter2: <testLibraryFragment>::@class::C::@getter::v
              setter2: <testLibraryFragment>::@class::C::@setter::v
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get v
              reference: <testLibraryFragment>::@class::C::@getter::v
              element: <testLibraryFragment>::@class::C::@getter::v#element
          setters
            synthetic set v
              reference: <testLibraryFragment>::@class::C::@setter::v
              element: <testLibraryFragment>::@class::C::@setter::v#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::v::@parameter::_v#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        static hasInitializer v
          firstFragment: <testLibraryFragment>::@class::C::@field::v
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::v#element
          setter: <testLibraryFragment>::@class::C::@setter::v#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic static get v
          firstFragment: <testLibraryFragment>::@class::C::@getter::v
      setters
        synthetic static set v
          firstFragment: <testLibraryFragment>::@class::C::@setter::v
          formalParameters
            requiredPositional _v
              type: int
''');
  }

  test_class_field_inheritedContextType_double() async {
    var library = await buildLibrary('''
abstract class A {
  const A();
  double get foo;
}
class B extends A {
  const B();
  final foo = 2;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        abstract class A @15
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic foo @-1
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: double
          constructors
            const @27
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            abstract get foo @45
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: double
        class B @58
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          fields
            final foo @93
              reference: <testLibraryFragment>::@class::B::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::B
              type: double
              shouldUseTypeForInitializerInference: true
              constantInitializer
                IntegerLiteral
                  literal: 2 @99
                  staticType: double
          constructors
            const @80
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
          accessors
            synthetic get foo @-1
              reference: <testLibraryFragment>::@class::B::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::B
              returnType: double
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @15
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
          constructors
            const new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 27
          getters
            get foo @45
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
        class B @58
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          fields
            hasInitializer foo @93
              reference: <testLibraryFragment>::@class::B::@field::foo
              element: <testLibraryFragment>::@class::B::@field::foo#element
              getter2: <testLibraryFragment>::@class::B::@getter::foo
          constructors
            const new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              typeNameOffset: 80
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::B::@getter::foo
              element: <testLibraryFragment>::@class::B::@getter::foo#element
  classes
    abstract class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: double
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        abstract get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      fields
        final hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::B::@field::foo
          type: double
          getter: <testLibraryFragment>::@class::B::@getter::foo#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::B::@getter::foo
''');
  }

  test_class_field_isPromotable_abstractGetter() async {
    var library = await buildLibrary(r'''
abstract class A {
  int? get _foo;
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        abstract class A @15
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic promotable _foo @-1
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            abstract get _foo @30
              reference: <testLibraryFragment>::@class::A::@getter::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int?
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @15
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic _foo
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            get _foo @30
              reference: <testLibraryFragment>::@class::A::@getter::_foo
              element: <testLibraryFragment>::@class::A::@getter::_foo#element
  classes
    abstract class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        abstract get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
''');
  }

  test_class_field_isPromotable_hasGetter() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  int? get _foo => 0;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
  fieldNameNonPromotabilityInfo
    _foo
      conflictingGetters
        <testLibraryFragment>::@class::B::@getter::_foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      fields
        synthetic _foo
          firstFragment: <testLibraryFragment>::@class::B::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::B::@getter::_foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
      getters
        get _foo
          firstFragment: <testLibraryFragment>::@class::B::@getter::_foo
  fieldNameNonPromotabilityInfo
    _foo
      conflictingGetters
        <testLibraryFragment>::@class::B::@getter::_foo
''');
  }

  test_class_field_isPromotable_hasGetter_abstract() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

abstract class B {
  int? get _foo;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final promotable _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
    abstract class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      fields
        synthetic promotable _foo
          firstFragment: <testLibraryFragment>::@class::B::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::B::@getter::_foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
      getters
        abstract get _foo
          firstFragment: <testLibraryFragment>::@class::B::@getter::_foo
''');
  }

  test_class_field_isPromotable_hasGetter_inPart() async {
    newFile('$testPackageLibPath/a.dart', r'''
part of 'test.dart';
class B {
  int? get _foo => 0;
}
''');

    var library = await buildLibrary(r'''
part 'a.dart';
class A {
  final int? _foo;
  A(this._foo);
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final _foo @38
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
  fieldNameNonPromotabilityInfo
    _foo
      conflictingGetters
        <testLibrary>::@fragment::package:test/a.dart::@class::B::@getter::_foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class A @21
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @38
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibrary>::@fragment::package:test/a.dart::@class::B
      fields
        synthetic _foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@class::B::@field::_foo
          type: int?
          getter: <testLibrary>::@fragment::package:test/a.dart::@class::B::@getter::_foo#element
      constructors
        synthetic new
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@class::B::@constructor::new
      getters
        get _foo
          firstFragment: <testLibrary>::@fragment::package:test/a.dart::@class::B::@getter::_foo
  fieldNameNonPromotabilityInfo
    _foo
      conflictingGetters
        <testLibrary>::@fragment::package:test/a.dart::@class::B::@getter::_foo
''');
  }

  test_class_field_isPromotable_hasGetter_static() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  static int? get _foo => 0;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final promotable _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      fields
        synthetic static _foo
          firstFragment: <testLibraryFragment>::@class::B::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::B::@getter::_foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
      getters
        static get _foo
          firstFragment: <testLibraryFragment>::@class::B::@getter::_foo
''');
  }

  test_class_field_isPromotable_hasNotFinalField() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  int? _foo;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
  fieldNameNonPromotabilityInfo
    _foo
      conflictingFields
        <testLibraryFragment>::@class::B::@field::_foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      fields
        _foo
          firstFragment: <testLibraryFragment>::@class::B::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::B::@getter::_foo#element
          setter: <testLibraryFragment>::@class::B::@setter::_foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::B::@getter::_foo
      setters
        synthetic set _foo
          firstFragment: <testLibraryFragment>::@class::B::@setter::_foo
          formalParameters
            requiredPositional __foo
              type: int?
  fieldNameNonPromotabilityInfo
    _foo
      conflictingFields
        <testLibraryFragment>::@class::B::@field::_foo
''');
  }

  test_class_field_isPromotable_hasNotFinalField_static() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  static int? _foo;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final promotable _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      fields
        static _foo
          firstFragment: <testLibraryFragment>::@class::B::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::B::@getter::_foo#element
          setter: <testLibraryFragment>::@class::B::@setter::_foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
      getters
        synthetic static get _foo
          firstFragment: <testLibraryFragment>::@class::B::@getter::_foo
      setters
        synthetic static set _foo
          firstFragment: <testLibraryFragment>::@class::B::@setter::_foo
          formalParameters
            requiredPositional __foo
              type: int?
''');
  }

  test_class_field_isPromotable_hasSetter() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  set _field(int? _) {}
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final promotable _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      fields
        synthetic _field
          firstFragment: <testLibraryFragment>::@class::B::@field::_field
          type: int?
          setter: <testLibraryFragment>::@class::B::@setter::_field#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
      setters
        set _field
          firstFragment: <testLibraryFragment>::@class::B::@setter::_field
          formalParameters
            requiredPositional _
              type: int?
''');
  }

  test_class_field_isPromotable_language217() async {
    var library = await buildLibrary(r'''
// @dart = 2.19
class A {
  final int? _foo;
  A(this._foo);
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @22
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final _foo @39
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @22
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @39
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_field() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  final int? _foo = 0;
}

/// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
class C implements B {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
  fieldNameNonPromotabilityInfo
    _foo
      conflictingNsmClasses
        <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      fields
        final hasInitializer _foo
          firstFragment: <testLibraryFragment>::@class::B::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::B::@getter::_foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::B::@getter::_foo
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
      interfaces
        B
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        noSuchMethod
          reference: <testLibrary>::@class::C::@method::noSuchMethod
          firstFragment: <testLibraryFragment>::@class::C::@method::noSuchMethod
          formalParameters
            requiredPositional invocation
              type: Invocation
  fieldNameNonPromotabilityInfo
    _foo
      conflictingNsmClasses
        <testLibraryFragment>::@class::C
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_field_implementedInMixin() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

mixin M {
  final int? _foo = 0;
}

class B {
  final int? _foo = 0;
}

/// `_foo` is implemented in [M].
class C with M implements B {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(
      classNames: {'A', 'B'},
      mixinNames: {'M'},
    );
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final promotable _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
        class B @90
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          fields
            final promotable _foo @107
              reference: <testLibraryFragment>::@class::B::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::B
              type: int?
              shouldUseTypeForInitializerInference: true
      mixins
        mixin M @54
          reference: <testLibraryFragment>::@mixin::M
          enclosingElement3: <testLibraryFragment>
          superclassConstraints
            Object
          fields
            final promotable _foo @71
              reference: <testLibraryFragment>::@mixin::M::@field::_foo
              enclosingElement3: <testLibraryFragment>::@mixin::M
              type: int?
              shouldUseTypeForInitializerInference: true
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
        class B @90
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          fields
            hasInitializer _foo @107
              reference: <testLibraryFragment>::@class::B::@field::_foo
              element: <testLibraryFragment>::@class::B::@field::_foo#element
              getter2: <testLibraryFragment>::@class::B::@getter::_foo
      mixins
        mixin M @54
          reference: <testLibraryFragment>::@mixin::M
          element: <testLibrary>::@mixin::M
          fields
            hasInitializer _foo @71
              reference: <testLibraryFragment>::@mixin::M::@field::_foo
              element: <testLibraryFragment>::@mixin::M::@field::_foo#element
              getter2: <testLibraryFragment>::@mixin::M::@getter::_foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      fields
        final promotable hasInitializer _foo
          firstFragment: <testLibraryFragment>::@class::B::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::B::@getter::_foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::B::@getter::_foo
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /// `_foo` is implemented in [M].
      supertype: Object
      mixins
        M
      interfaces
        B
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        noSuchMethod
          reference: <testLibrary>::@class::C::@method::noSuchMethod
          firstFragment: <testLibraryFragment>::@class::C::@method::noSuchMethod
          formalParameters
            requiredPositional invocation
              type: Invocation
  mixins
    mixin M
      reference: <testLibrary>::@mixin::M
      firstFragment: <testLibraryFragment>::@mixin::M
      superclassConstraints
        Object
      fields
        final promotable hasInitializer _foo
          firstFragment: <testLibraryFragment>::@mixin::M::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@mixin::M::@getter::_foo#element
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@mixin::M::@getter::_foo
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_field_implementedInSuperclass() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  final int? _foo = 0;
}

class C {
  final int? _foo = 0;
}

/// `_foo` is implemented in [B].
class D extends B implements C {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(
      classNames: {'A', 'B'},
    );
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final promotable _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
        class B @54
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          fields
            final promotable _foo @71
              reference: <testLibraryFragment>::@class::B::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::B
              type: int?
              shouldUseTypeForInitializerInference: true
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
        class B @54
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          fields
            hasInitializer _foo @71
              reference: <testLibraryFragment>::@class::B::@field::_foo
              element: <testLibraryFragment>::@class::B::@field::_foo#element
              getter2: <testLibraryFragment>::@class::B::@getter::_foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      fields
        final promotable hasInitializer _foo
          firstFragment: <testLibraryFragment>::@class::B::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::B::@getter::_foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::B::@getter::_foo
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final promotable hasInitializer _foo
          firstFragment: <testLibraryFragment>::@class::C::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::C::@getter::_foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::_foo
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      documentationComment: /// `_foo` is implemented in [B].
      supertype: B
      interfaces
        C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
          superConstructor: <testLibraryFragment>::@class::B::@constructor::new#element
      methods
        noSuchMethod
          reference: <testLibrary>::@class::D::@method::noSuchMethod
          firstFragment: <testLibraryFragment>::@class::D::@method::noSuchMethod
          formalParameters
            requiredPositional invocation
              type: Invocation
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_field_inClassTypeAlias() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  final int? _foo = 0;
}

mixin M {
  dynamic noSuchMethod(Invocation invocation) {}
}

/// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
class E = Object with M implements B;
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
  fieldNameNonPromotabilityInfo
    _foo
      conflictingNsmClasses
        <testLibraryFragment>::@class::E
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      fields
        final hasInitializer _foo
          firstFragment: <testLibraryFragment>::@class::B::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::B::@getter::_foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::B::@getter::_foo
    class alias E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      documentationComment: /// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
      supertype: Object
      mixins
        M
      interfaces
        B
      constructors
        synthetic const new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
  mixins
    mixin M
      reference: <testLibrary>::@mixin::M
      firstFragment: <testLibraryFragment>::@mixin::M
      superclassConstraints
        Object
      methods
        noSuchMethod
          reference: <testLibrary>::@mixin::M::@method::noSuchMethod
          firstFragment: <testLibraryFragment>::@mixin::M::@method::noSuchMethod
          formalParameters
            requiredPositional invocation
              type: Invocation
  fieldNameNonPromotabilityInfo
    _foo
      conflictingNsmClasses
        <testLibraryFragment>::@class::E
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_field_inEnum() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B {
  final int? _foo = 0;
}

/// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
enum E implements B {
  v;
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(
      classNames: {'A', 'B'},
    );
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
        class B @54
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          fields
            final _foo @71
              reference: <testLibraryFragment>::@class::B::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::B
              type: int?
              shouldUseTypeForInitializerInference: true
  fieldNameNonPromotabilityInfo
    _foo
      conflictingNsmClasses
        <testLibraryFragment>::@enum::E
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
        class B @54
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          fields
            hasInitializer _foo @71
              reference: <testLibraryFragment>::@class::B::@field::_foo
              element: <testLibraryFragment>::@class::B::@field::_foo#element
              getter2: <testLibraryFragment>::@class::B::@getter::_foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      fields
        final hasInitializer _foo
          firstFragment: <testLibraryFragment>::@class::B::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::B::@getter::_foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::B::@getter::_foo
  enums
    enum E
      reference: <testLibrary>::@enum::E
      firstFragment: <testLibraryFragment>::@enum::E
      documentationComment: /// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
      supertype: Enum
      interfaces
        B
      fields
        static const enumConstant hasInitializer v
          firstFragment: <testLibraryFragment>::@enum::E::@field::v
          type: E
          getter: <testLibraryFragment>::@enum::E::@getter::v#element
        synthetic static const values
          firstFragment: <testLibraryFragment>::@enum::E::@field::values
          type: List<E>
          getter: <testLibraryFragment>::@enum::E::@getter::values#element
      constructors
        synthetic const new
          firstFragment: <testLibraryFragment>::@enum::E::@constructor::new
      getters
        synthetic static get v
          firstFragment: <testLibraryFragment>::@enum::E::@getter::v
        synthetic static get values
          firstFragment: <testLibraryFragment>::@enum::E::@getter::values
      methods
        noSuchMethod
          reference: <testLibrary>::@enum::E::@method::noSuchMethod
          firstFragment: <testLibraryFragment>::@enum::E::@method::noSuchMethod
          formalParameters
            requiredPositional invocation
              type: Invocation
  fieldNameNonPromotabilityInfo
    _foo
      conflictingNsmClasses
        <testLibraryFragment>::@enum::E
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_getter() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

abstract class B {
  int? get _foo;
}

/// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
class C implements B {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
  fieldNameNonPromotabilityInfo
    _foo
      conflictingNsmClasses
        <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
    abstract class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      fields
        synthetic _foo
          firstFragment: <testLibraryFragment>::@class::B::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::B::@getter::_foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
      getters
        abstract get _foo
          firstFragment: <testLibraryFragment>::@class::B::@getter::_foo
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
      interfaces
        B
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        noSuchMethod
          reference: <testLibrary>::@class::C::@method::noSuchMethod
          firstFragment: <testLibraryFragment>::@class::C::@method::noSuchMethod
          formalParameters
            requiredPositional invocation
              type: Invocation
  fieldNameNonPromotabilityInfo
    _foo
      conflictingNsmClasses
        <testLibraryFragment>::@class::C
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_inDifferentLibrary() async {
    newFile('$testPackageLibPath/a.dart', r'''
class B {
  int? get _foo => 0;
}
''');

    var library = await buildLibrary(r'''
import 'a.dart';

class A {
  final int? _foo;
  A(this._foo);
}

/// Has a noSuchMethod thrower for B._field, but since private names in
/// different libraries are distinct, this has no effect on promotion of
/// C._field.
class C implements B {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(
      classNames: {'A'},
    );
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      libraryImports
        package:test/a.dart
          enclosingElement3: <testLibraryFragment>
      classes
        class A @24
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final promotable _foo @41
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/a.dart
      classes
        class A @24
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @41
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /// Has a noSuchMethod thrower for B._field, but since private names in\n/// different libraries are distinct, this has no effect on promotion of\n/// C._field.
      interfaces
        B
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        noSuchMethod
          reference: <testLibrary>::@class::C::@method::noSuchMethod
          firstFragment: <testLibraryFragment>::@class::C::@method::noSuchMethod
          formalParameters
            requiredPositional invocation
              type: Invocation
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_inheritedInterface() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

class B extends A {
  A(super.value);
}

/// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
class C implements B {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
  fieldNameNonPromotabilityInfo
    _foo
      conflictingNsmClasses
        <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
      methods
        abstract A
          reference: <testLibrary>::@class::B::@method::A
          firstFragment: <testLibraryFragment>::@class::B::@method::A
          formalParameters
            requiredPositional final hasImplicitType value
              type: dynamic
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
      interfaces
        B
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        noSuchMethod
          reference: <testLibrary>::@class::C::@method::noSuchMethod
          firstFragment: <testLibraryFragment>::@class::C::@method::noSuchMethod
          formalParameters
            requiredPositional invocation
              type: Invocation
  fieldNameNonPromotabilityInfo
    _foo
      conflictingNsmClasses
        <testLibraryFragment>::@class::C
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_mixedInterface() async {
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

mixin M {
  final int? _foo = 0;
}

class B with M {}

/// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
class C implements B {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(
      classNames: {'A'},
      mixinNames: {'M'},
    );
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
      mixins
        mixin M @54
          reference: <testLibraryFragment>::@mixin::M
          enclosingElement3: <testLibraryFragment>
          superclassConstraints
            Object
          fields
            final _foo @71
              reference: <testLibraryFragment>::@mixin::M::@field::_foo
              enclosingElement3: <testLibraryFragment>::@mixin::M
              type: int?
              shouldUseTypeForInitializerInference: true
  fieldNameNonPromotabilityInfo
    _foo
      conflictingNsmClasses
        <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
      mixins
        mixin M @54
          reference: <testLibraryFragment>::@mixin::M
          element: <testLibrary>::@mixin::M
          fields
            hasInitializer _foo @71
              reference: <testLibraryFragment>::@mixin::M::@field::_foo
              element: <testLibraryFragment>::@mixin::M::@field::_foo#element
              getter2: <testLibraryFragment>::@mixin::M::@getter::_foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: Object
      mixins
        M
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      documentationComment: /// Implicitly implements `_foo` as a getter that forwards to [noSuchMethod].
      interfaces
        B
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        noSuchMethod
          reference: <testLibrary>::@class::C::@method::noSuchMethod
          firstFragment: <testLibraryFragment>::@class::C::@method::noSuchMethod
          formalParameters
            requiredPositional invocation
              type: Invocation
  mixins
    mixin M
      reference: <testLibrary>::@mixin::M
      firstFragment: <testLibraryFragment>::@mixin::M
      superclassConstraints
        Object
      fields
        final hasInitializer _foo
          firstFragment: <testLibraryFragment>::@mixin::M::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@mixin::M::@getter::_foo#element
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@mixin::M::@getter::_foo
  fieldNameNonPromotabilityInfo
    _foo
      conflictingNsmClasses
        <testLibraryFragment>::@class::C
''');
  }

  test_class_field_isPromotable_noSuchMethodForwarder_unusedMixin() async {
    // Mixins are implicitly abstract so the presence of a mixin that inherits
    // a field into its interface, and doesn't implement it, doesn't mean that
    // a noSuchMethod forwarder created for it. So,  this does not block that
    // field from promoting.
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  A(this._foo);
}

mixin M implements A {
  dynamic noSuchMethod(Invocation invocation) {}
}
''');

    configuration.forPromotableFields(
      classNames: {'A'},
    );
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final promotable _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
  mixins
    mixin M
      reference: <testLibrary>::@mixin::M
      firstFragment: <testLibraryFragment>::@mixin::M
      superclassConstraints
        Object
      interfaces
        A
      methods
        noSuchMethod
          reference: <testLibrary>::@mixin::M::@method::noSuchMethod
          firstFragment: <testLibraryFragment>::@mixin::M::@method::noSuchMethod
          formalParameters
            requiredPositional invocation
              type: Invocation
''');
  }

  test_class_field_isPromotable_notFinal() async {
    var library = await buildLibrary(r'''
class A {
  int? _foo;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            _foo @17
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
  fieldNameNonPromotabilityInfo
    _foo
      conflictingFields
        <testLibraryFragment>::@class::A::@field::_foo
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @17
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
              setter2: <testLibraryFragment>::@class::A::@setter::_foo
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
          setter: <testLibraryFragment>::@class::A::@setter::_foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
      setters
        synthetic set _foo
          firstFragment: <testLibraryFragment>::@class::A::@setter::_foo
          formalParameters
            requiredPositional __foo
              type: int?
  fieldNameNonPromotabilityInfo
    _foo
      conflictingFields
        <testLibraryFragment>::@class::A::@field::_foo
''');
  }

  test_class_field_isPromotable_notPrivate() async {
    var library = await buildLibrary(r'''
class A {
  int? field;
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            field @17
              reference: <testLibraryFragment>::@class::A::@field::field
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            field @17
              reference: <testLibraryFragment>::@class::A::@field::field
              element: <testLibraryFragment>::@class::A::@field::field#element
              getter2: <testLibraryFragment>::@class::A::@getter::field
              setter2: <testLibraryFragment>::@class::A::@setter::field
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        field
          firstFragment: <testLibraryFragment>::@class::A::@field::field
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::field#element
          setter: <testLibraryFragment>::@class::A::@setter::field#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic get field
          firstFragment: <testLibraryFragment>::@class::A::@getter::field
      setters
        synthetic set field
          firstFragment: <testLibraryFragment>::@class::A::@setter::field
          formalParameters
            requiredPositional _field
              type: int?
''');
  }

  test_class_field_isPromotable_typeInference() async {
    // We decide that `_foo` is promotable before inferring the type of `bar`.
    var library = await buildLibrary(r'''
class A {
  final int? _foo;
  final bar = _foo != null ? _foo : 0;
  A(this._foo);
}
''');

    configuration.forPromotableFields(classNames: {'A'});
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            final promotable _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int?
            final bar @37
              reference: <testLibraryFragment>::@class::A::@field::bar
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
              shouldUseTypeForInitializerInference: false
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            _foo @23
              reference: <testLibraryFragment>::@class::A::@field::_foo
              element: <testLibraryFragment>::@class::A::@field::_foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::_foo
            hasInitializer bar @37
              reference: <testLibraryFragment>::@class::A::@field::bar
              element: <testLibraryFragment>::@class::A::@field::bar#element
              getter2: <testLibraryFragment>::@class::A::@getter::bar
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        final promotable _foo
          firstFragment: <testLibraryFragment>::@class::A::@field::_foo
          type: int?
          getter: <testLibraryFragment>::@class::A::@getter::_foo#element
        final hasInitializer bar
          firstFragment: <testLibraryFragment>::@class::A::@field::bar
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::bar#element
      constructors
        new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
          formalParameters
            requiredPositional final hasImplicitType _foo
              type: int?
      getters
        synthetic get _foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::_foo
        synthetic get bar
          firstFragment: <testLibraryFragment>::@class::A::@getter::bar
''');
  }

  test_class_field_missingName() async {
    var library = await buildLibrary('''
abstract class C {
  Object a,;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        abstract class C @15
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            a @28
              reference: <testLibraryFragment>::@class::C::@field::a
              enclosingElement3: <testLibraryFragment>::@class::C
              type: Object
            @30
              reference: <testLibraryFragment>::@class::C::@field::0
              enclosingElement3: <testLibraryFragment>::@class::C
              type: Object
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get a @-1
              reference: <testLibraryFragment>::@class::C::@getter::a
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: Object
            synthetic set a= @-1
              reference: <testLibraryFragment>::@class::C::@setter::a
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _a @-1
                  type: Object
              returnType: void
            synthetic get @-1
              reference: <testLibraryFragment>::@class::C::@getter::0
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: Object
            synthetic set = @-1
              reference: <testLibraryFragment>::@class::C::@setter::0
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _ @-1
                  type: Object
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @15
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            a @28
              reference: <testLibraryFragment>::@class::C::@field::a
              element: <testLibraryFragment>::@class::C::@field::a#element
              getter2: <testLibraryFragment>::@class::C::@getter::a
              setter2: <testLibraryFragment>::@class::C::@setter::a
            <null-name>
              reference: <testLibraryFragment>::@class::C::@field::0
              element: <testLibraryFragment>::@class::C::@field::0#element
              getter2: <testLibraryFragment>::@class::C::@getter::0
              setter2: <testLibraryFragment>::@class::C::@setter::0
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get a
              reference: <testLibraryFragment>::@class::C::@getter::a
              element: <testLibraryFragment>::@class::C::@getter::a#element
            synthetic get <null-name>
              reference: <testLibraryFragment>::@class::C::@getter::0
              element: <testLibraryFragment>::@class::C::@getter::0#element
          setters
            synthetic set a
              reference: <testLibraryFragment>::@class::C::@setter::a
              element: <testLibraryFragment>::@class::C::@setter::a#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::a::@parameter::_a#element
            synthetic set <null-name>
              reference: <testLibraryFragment>::@class::C::@setter::0
              element: <testLibraryFragment>::@class::C::@setter::0#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::0::@parameter::_#element
  classes
    abstract class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        a
          firstFragment: <testLibraryFragment>::@class::C::@field::a
          type: Object
          getter: <testLibraryFragment>::@class::C::@getter::a#element
          setter: <testLibraryFragment>::@class::C::@setter::a#element
        <null-name>
          firstFragment: <testLibraryFragment>::@class::C::@field::0
          type: Object
          getter: <testLibraryFragment>::@class::C::@getter::0#element
          setter: <testLibraryFragment>::@class::C::@setter::0#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get a
          firstFragment: <testLibraryFragment>::@class::C::@getter::a
        synthetic get <null-name>
          firstFragment: <testLibraryFragment>::@class::C::@getter::0
      setters
        synthetic set a
          firstFragment: <testLibraryFragment>::@class::C::@setter::a
          formalParameters
            requiredPositional _a
              type: Object
        synthetic set <null-name>
          firstFragment: <testLibraryFragment>::@class::C::@setter::0
          formalParameters
            requiredPositional _
              type: Object
''');
  }

  test_class_field_propagatedType_const_noDep() async {
    var library = await buildLibrary('''
class C {
  static const x = 0;
}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            static const x @25
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              shouldUseTypeForInitializerInference: false
              constantInitializer
                IntegerLiteral
                  literal: 0 @29
                  staticType: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic static get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer x @25
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        static const hasInitializer x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic static get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
''');
  }

  test_class_field_propagatedType_final_dep_inLib() async {
    newFile('$testPackageLibPath/a.dart', 'final a = 1;');
    var library = await buildLibrary('''
import "a.dart";
class C {
  final b = a / 2;
}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      libraryImports
        package:test/a.dart
          enclosingElement3: <testLibraryFragment>
      classes
        class C @23
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            final b @35
              reference: <testLibraryFragment>::@class::C::@field::b
              enclosingElement3: <testLibraryFragment>::@class::C
              type: double
              shouldUseTypeForInitializerInference: false
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get b @-1
              reference: <testLibraryFragment>::@class::C::@getter::b
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: double
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        package:test/a.dart
      classes
        class C @23
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer b @35
              reference: <testLibraryFragment>::@class::C::@field::b
              element: <testLibraryFragment>::@class::C::@field::b#element
              getter2: <testLibraryFragment>::@class::C::@getter::b
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get b
              reference: <testLibraryFragment>::@class::C::@getter::b
              element: <testLibraryFragment>::@class::C::@getter::b#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final hasInitializer b
          firstFragment: <testLibraryFragment>::@class::C::@field::b
          type: double
          getter: <testLibraryFragment>::@class::C::@getter::b#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get b
          firstFragment: <testLibraryFragment>::@class::C::@getter::b
''');
  }

  test_class_field_propagatedType_final_dep_inPart() async {
    newFile('$testPackageLibPath/a.dart', 'part of lib; final a = 1;');
    var library = await buildLibrary('''
library lib;
part "a.dart";
class C {
  final b = a / 2;
}''');
    checkElementText(library, r'''
library
  name: lib
  nameOffset: 8
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      parts
        part_0
          uri: package:test/a.dart
          enclosingElement3: <testLibraryFragment>
          unit: <testLibrary>::@fragment::package:test/a.dart
      classes
        class C @34
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            final b @46
              reference: <testLibraryFragment>::@class::C::@field::b
              enclosingElement3: <testLibraryFragment>::@class::C
              type: double
              shouldUseTypeForInitializerInference: false
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get b @-1
              reference: <testLibraryFragment>::@class::C::@getter::b
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: double
    <testLibrary>::@fragment::package:test/a.dart
      enclosingElement3: <testLibraryFragment>
      topLevelVariables
        static final a @19
          reference: <testLibrary>::@fragment::package:test/a.dart::@topLevelVariable::a
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          type: int
          shouldUseTypeForInitializerInference: false
      accessors
        synthetic static get a @-1
          reference: <testLibrary>::@fragment::package:test/a.dart::@getter::a
          enclosingElement3: <testLibrary>::@fragment::package:test/a.dart
          returnType: int
----------------------------------------
library
  reference: <testLibrary>
  name: lib
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      nextFragment: <testLibrary>::@fragment::package:test/a.dart
      classes
        class C @34
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer b @46
              reference: <testLibraryFragment>::@class::C::@field::b
              element: <testLibraryFragment>::@class::C::@field::b#element
              getter2: <testLibraryFragment>::@class::C::@getter::b
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get b
              reference: <testLibraryFragment>::@class::C::@getter::b
              element: <testLibraryFragment>::@class::C::@getter::b#element
    <testLibrary>::@fragment::package:test/a.dart
      element: <testLibrary>
      enclosingFragment: <testLibraryFragment>
      previousFragment: <testLibraryFragment>
      topLevelVariables
        hasInitializer a @19
          reference: <testLibrary>::@fragment::package:test/a.dart::@topLevelVariable::a
          element: <testLibrary>::@topLevelVariable::a
          getter2: <testLibrary>::@fragment::package:test/a.dart::@getter::a
      getters
        synthetic get a
          reference: <testLibrary>::@fragment::package:test/a.dart::@getter::a
          element: <testLibrary>::@fragment::package:test/a.dart::@getter::a#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final hasInitializer b
          firstFragment: <testLibraryFragment>::@class::C::@field::b
          type: double
          getter: <testLibraryFragment>::@class::C::@getter::b#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get b
          firstFragment: <testLibraryFragment>::@class::C::@getter::b
  topLevelVariables
    final hasInitializer a
      reference: <testLibrary>::@topLevelVariable::a
      firstFragment: <testLibrary>::@fragment::package:test/a.dart::@topLevelVariable::a
      type: int
      getter: <testLibrary>::@fragment::package:test/a.dart::@getter::a#element
  getters
    synthetic static get a
      firstFragment: <testLibrary>::@fragment::package:test/a.dart::@getter::a
''');
  }

  test_class_field_propagatedType_final_noDep_instance() async {
    var library = await buildLibrary('''
class C {
  final x = 0;
}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            final x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              shouldUseTypeForInitializerInference: false
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer x @18
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        final hasInitializer x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
''');
  }

  test_class_field_propagatedType_final_noDep_static() async {
    var library = await buildLibrary('''
class C {
  static final x = 0;
}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            static final x @25
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              shouldUseTypeForInitializerInference: false
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic static get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer x @25
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        static final hasInitializer x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic static get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
''');
  }

  test_class_field_static() async {
    var library = await buildLibrary('class C { static int i; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            static i @21
              reference: <testLibraryFragment>::@class::C::@field::i
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic static get i @-1
              reference: <testLibraryFragment>::@class::C::@getter::i
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic static set i= @-1
              reference: <testLibraryFragment>::@class::C::@setter::i
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _i @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            i @21
              reference: <testLibraryFragment>::@class::C::@field::i
              element: <testLibraryFragment>::@class::C::@field::i#element
              getter2: <testLibraryFragment>::@class::C::@getter::i
              setter2: <testLibraryFragment>::@class::C::@setter::i
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get i
              reference: <testLibraryFragment>::@class::C::@getter::i
              element: <testLibraryFragment>::@class::C::@getter::i#element
          setters
            synthetic set i
              reference: <testLibraryFragment>::@class::C::@setter::i
              element: <testLibraryFragment>::@class::C::@setter::i#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::i::@parameter::_i#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        static i
          firstFragment: <testLibraryFragment>::@class::C::@field::i
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::i#element
          setter: <testLibraryFragment>::@class::C::@setter::i#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic static get i
          firstFragment: <testLibraryFragment>::@class::C::@getter::i
      setters
        synthetic static set i
          firstFragment: <testLibraryFragment>::@class::C::@setter::i
          formalParameters
            requiredPositional _i
              type: int
''');
  }

  test_class_field_static_final_hasConstConstructor() async {
    var library = await buildLibrary('''
class C {
  static final f = 0;
  const C();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            static final f @25
              reference: <testLibraryFragment>::@class::C::@field::f
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              shouldUseTypeForInitializerInference: false
          constructors
            const @40
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic static get f @-1
              reference: <testLibraryFragment>::@class::C::@getter::f
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer f @25
              reference: <testLibraryFragment>::@class::C::@field::f
              element: <testLibraryFragment>::@class::C::@field::f#element
              getter2: <testLibraryFragment>::@class::C::@getter::f
          constructors
            const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              typeNameOffset: 40
          getters
            synthetic get f
              reference: <testLibraryFragment>::@class::C::@getter::f
              element: <testLibraryFragment>::@class::C::@getter::f#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        static final hasInitializer f
          firstFragment: <testLibraryFragment>::@class::C::@field::f
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::f#element
      constructors
        const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic static get f
          firstFragment: <testLibraryFragment>::@class::C::@getter::f
''');
  }

  test_class_field_static_final_untyped() async {
    var library = await buildLibrary('class C { static final x = 0; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            static final x @23
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              shouldUseTypeForInitializerInference: false
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic static get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer x @23
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        static final hasInitializer x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic static get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
''');
  }

  test_class_field_static_late() async {
    var library = await buildLibrary('class C { static late int i; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            static late i @26
              reference: <testLibraryFragment>::@class::C::@field::i
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic static get i @-1
              reference: <testLibraryFragment>::@class::C::@getter::i
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic static set i= @-1
              reference: <testLibraryFragment>::@class::C::@setter::i
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _i @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            i @26
              reference: <testLibraryFragment>::@class::C::@field::i
              element: <testLibraryFragment>::@class::C::@field::i#element
              getter2: <testLibraryFragment>::@class::C::@getter::i
              setter2: <testLibraryFragment>::@class::C::@setter::i
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get i
              reference: <testLibraryFragment>::@class::C::@getter::i
              element: <testLibraryFragment>::@class::C::@getter::i#element
          setters
            synthetic set i
              reference: <testLibraryFragment>::@class::C::@setter::i
              element: <testLibraryFragment>::@class::C::@setter::i#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::i::@parameter::_i#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        static late i
          firstFragment: <testLibraryFragment>::@class::C::@field::i
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::i#element
          setter: <testLibraryFragment>::@class::C::@setter::i#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic static get i
          firstFragment: <testLibraryFragment>::@class::C::@getter::i
      setters
        synthetic static set i
          firstFragment: <testLibraryFragment>::@class::C::@setter::i
          formalParameters
            requiredPositional _i
              type: int
''');
  }

  test_class_field_type_explicit() async {
    var library = await buildLibrary(r'''
class C {
  int a = 0;
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            a @16
              reference: <testLibraryFragment>::@class::C::@field::a
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              shouldUseTypeForInitializerInference: true
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get a @-1
              reference: <testLibraryFragment>::@class::C::@getter::a
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic set a= @-1
              reference: <testLibraryFragment>::@class::C::@setter::a
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _a @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer a @16
              reference: <testLibraryFragment>::@class::C::@field::a
              element: <testLibraryFragment>::@class::C::@field::a#element
              getter2: <testLibraryFragment>::@class::C::@getter::a
              setter2: <testLibraryFragment>::@class::C::@setter::a
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get a
              reference: <testLibraryFragment>::@class::C::@getter::a
              element: <testLibraryFragment>::@class::C::@getter::a#element
          setters
            synthetic set a
              reference: <testLibraryFragment>::@class::C::@setter::a
              element: <testLibraryFragment>::@class::C::@setter::a#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::a::@parameter::_a#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        hasInitializer a
          firstFragment: <testLibraryFragment>::@class::C::@field::a
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::a#element
          setter: <testLibraryFragment>::@class::C::@setter::a#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get a
          firstFragment: <testLibraryFragment>::@class::C::@getter::a
      setters
        synthetic set a
          firstFragment: <testLibraryFragment>::@class::C::@setter::a
          formalParameters
            requiredPositional _a
              type: int
''');
  }

  test_class_field_type_inferred_fromInitializer() async {
    var library = await buildLibrary(r'''
class C {
  var foo = 0;
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            foo @16
              reference: <testLibraryFragment>::@class::C::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              shouldUseTypeForInitializerInference: false
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get foo @-1
              reference: <testLibraryFragment>::@class::C::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic set foo= @-1
              reference: <testLibraryFragment>::@class::C::@setter::foo
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _foo @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer foo @16
              reference: <testLibraryFragment>::@class::C::@field::foo
              element: <testLibraryFragment>::@class::C::@field::foo#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo
              setter2: <testLibraryFragment>::@class::C::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::C::@getter::foo
              element: <testLibraryFragment>::@class::C::@getter::foo#element
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::C::@setter::foo
              element: <testLibraryFragment>::@class::C::@setter::foo#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::foo::@parameter::_foo#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::foo#element
          setter: <testLibraryFragment>::@class::C::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::C::@setter::foo
          formalParameters
            requiredPositional _foo
              type: int
''');
  }

  test_class_field_type_inferred_fromSuper() async {
    var library = await buildLibrary(r'''
abstract class A {
  int get foo;
}

class B extends A {
  final foo = 0;
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        abstract class A @15
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic foo @-1
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            abstract get foo @29
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
        class B @43
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          fields
            final foo @65
              reference: <testLibraryFragment>::@class::B::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::B
              type: int
              shouldUseTypeForInitializerInference: true
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
          accessors
            synthetic get foo @-1
              reference: <testLibraryFragment>::@class::B::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::B
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @15
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            get foo @29
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
        class B @43
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          fields
            hasInitializer foo @65
              reference: <testLibraryFragment>::@class::B::@field::foo
              element: <testLibraryFragment>::@class::B::@field::foo#element
              getter2: <testLibraryFragment>::@class::B::@getter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::B::@getter::foo
              element: <testLibraryFragment>::@class::B::@getter::foo#element
  classes
    abstract class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        abstract get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      fields
        final hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::B::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::B::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::B::@getter::foo
''');
  }

  test_class_field_type_inferred_Never() async {
    var library = await buildLibrary(r'''
class C {
  var a = throw 42;
}
''');

    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            a @16
              reference: <testLibraryFragment>::@class::C::@field::a
              enclosingElement3: <testLibraryFragment>::@class::C
              type: Never
              shouldUseTypeForInitializerInference: false
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get a @-1
              reference: <testLibraryFragment>::@class::C::@getter::a
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: Never
            synthetic set a= @-1
              reference: <testLibraryFragment>::@class::C::@setter::a
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _a @-1
                  type: Never
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer a @16
              reference: <testLibraryFragment>::@class::C::@field::a
              element: <testLibraryFragment>::@class::C::@field::a#element
              getter2: <testLibraryFragment>::@class::C::@getter::a
              setter2: <testLibraryFragment>::@class::C::@setter::a
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get a
              reference: <testLibraryFragment>::@class::C::@getter::a
              element: <testLibraryFragment>::@class::C::@getter::a#element
          setters
            synthetic set a
              reference: <testLibraryFragment>::@class::C::@setter::a
              element: <testLibraryFragment>::@class::C::@setter::a#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::a::@parameter::_a#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        hasInitializer a
          firstFragment: <testLibraryFragment>::@class::C::@field::a
          type: Never
          getter: <testLibraryFragment>::@class::C::@getter::a#element
          setter: <testLibraryFragment>::@class::C::@setter::a#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get a
          firstFragment: <testLibraryFragment>::@class::C::@getter::a
      setters
        synthetic set a
          firstFragment: <testLibraryFragment>::@class::C::@setter::a
          formalParameters
            requiredPositional _a
              type: Never
''');
  }

  test_class_field_typed() async {
    var library = await buildLibrary('class C { int x = 0; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              shouldUseTypeForInitializerInference: true
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        hasInitializer x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: int
''');
  }

  test_class_field_untyped() async {
    var library = await buildLibrary('class C { var x = 0; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              shouldUseTypeForInitializerInference: false
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get x @-1
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic set x= @-1
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _x @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer x @14
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get x
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            synthetic set x
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::_x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        hasInitializer x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        synthetic set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional _x
              type: int
''');
  }

  test_class_fields() async {
    var library = await buildLibrary('class C { int i; int j; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            i @14
              reference: <testLibraryFragment>::@class::C::@field::i
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
            j @21
              reference: <testLibraryFragment>::@class::C::@field::j
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get i @-1
              reference: <testLibraryFragment>::@class::C::@getter::i
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic set i= @-1
              reference: <testLibraryFragment>::@class::C::@setter::i
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _i @-1
                  type: int
              returnType: void
            synthetic get j @-1
              reference: <testLibraryFragment>::@class::C::@getter::j
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic set j= @-1
              reference: <testLibraryFragment>::@class::C::@setter::j
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _j @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            i @14
              reference: <testLibraryFragment>::@class::C::@field::i
              element: <testLibraryFragment>::@class::C::@field::i#element
              getter2: <testLibraryFragment>::@class::C::@getter::i
              setter2: <testLibraryFragment>::@class::C::@setter::i
            j @21
              reference: <testLibraryFragment>::@class::C::@field::j
              element: <testLibraryFragment>::@class::C::@field::j#element
              getter2: <testLibraryFragment>::@class::C::@getter::j
              setter2: <testLibraryFragment>::@class::C::@setter::j
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get i
              reference: <testLibraryFragment>::@class::C::@getter::i
              element: <testLibraryFragment>::@class::C::@getter::i#element
            synthetic get j
              reference: <testLibraryFragment>::@class::C::@getter::j
              element: <testLibraryFragment>::@class::C::@getter::j#element
          setters
            synthetic set i
              reference: <testLibraryFragment>::@class::C::@setter::i
              element: <testLibraryFragment>::@class::C::@setter::i#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::i::@parameter::_i#element
            synthetic set j
              reference: <testLibraryFragment>::@class::C::@setter::j
              element: <testLibraryFragment>::@class::C::@setter::j#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::j::@parameter::_j#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        i
          firstFragment: <testLibraryFragment>::@class::C::@field::i
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::i#element
          setter: <testLibraryFragment>::@class::C::@setter::i#element
        j
          firstFragment: <testLibraryFragment>::@class::C::@field::j
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::j#element
          setter: <testLibraryFragment>::@class::C::@setter::j#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get i
          firstFragment: <testLibraryFragment>::@class::C::@getter::i
        synthetic get j
          firstFragment: <testLibraryFragment>::@class::C::@getter::j
      setters
        synthetic set i
          firstFragment: <testLibraryFragment>::@class::C::@setter::i
          formalParameters
            requiredPositional _i
              type: int
        synthetic set j
          firstFragment: <testLibraryFragment>::@class::C::@setter::j
          formalParameters
            requiredPositional _j
              type: int
''');
  }

  test_class_fields_late() async {
    var library = await buildLibrary('''
class C {
  late int foo;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            late foo @21
              reference: <testLibraryFragment>::@class::C::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get foo @-1
              reference: <testLibraryFragment>::@class::C::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic set foo= @-1
              reference: <testLibraryFragment>::@class::C::@setter::foo
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _foo @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            foo @21
              reference: <testLibraryFragment>::@class::C::@field::foo
              element: <testLibraryFragment>::@class::C::@field::foo#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo
              setter2: <testLibraryFragment>::@class::C::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::C::@getter::foo
              element: <testLibraryFragment>::@class::C::@getter::foo#element
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::C::@setter::foo
              element: <testLibraryFragment>::@class::C::@setter::foo#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::foo::@parameter::_foo#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        late foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::foo#element
          setter: <testLibraryFragment>::@class::C::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::C::@setter::foo
          formalParameters
            requiredPositional _foo
              type: int
''');
  }

  test_class_fields_late_final() async {
    var library = await buildLibrary('''
class C {
  late final int foo;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            late final foo @27
              reference: <testLibraryFragment>::@class::C::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get foo @-1
              reference: <testLibraryFragment>::@class::C::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            synthetic set foo= @-1
              reference: <testLibraryFragment>::@class::C::@setter::foo
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional _foo @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            foo @27
              reference: <testLibraryFragment>::@class::C::@field::foo
              element: <testLibraryFragment>::@class::C::@field::foo#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo
              setter2: <testLibraryFragment>::@class::C::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::C::@getter::foo
              element: <testLibraryFragment>::@class::C::@getter::foo#element
          setters
            synthetic set foo
              reference: <testLibraryFragment>::@class::C::@setter::foo
              element: <testLibraryFragment>::@class::C::@setter::foo#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C::@setter::foo::@parameter::_foo#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        late final foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::foo#element
          setter: <testLibraryFragment>::@class::C::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo
      setters
        synthetic set foo
          firstFragment: <testLibraryFragment>::@class::C::@setter::foo
          formalParameters
            requiredPositional _foo
              type: int
''');
  }

  test_class_fields_late_final_initialized() async {
    var library = await buildLibrary('''
class C {
  late final int foo = 0;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            late final foo @27
              reference: <testLibraryFragment>::@class::C::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
              shouldUseTypeForInitializerInference: true
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            synthetic get foo @-1
              reference: <testLibraryFragment>::@class::C::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            hasInitializer foo @27
              reference: <testLibraryFragment>::@class::C::@field::foo
              element: <testLibraryFragment>::@class::C::@field::foo#element
              getter2: <testLibraryFragment>::@class::C::@getter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            synthetic get foo
              reference: <testLibraryFragment>::@class::C::@getter::foo
              element: <testLibraryFragment>::@class::C::@getter::foo#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        late final hasInitializer foo
          firstFragment: <testLibraryFragment>::@class::C::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        synthetic get foo
          firstFragment: <testLibraryFragment>::@class::C::@getter::foo
''');
  }

  test_class_fields_late_inference_usingSuper_methodInvocation() async {
    var library = await buildLibrary('''
class A {
  int foo() => 0;
}

class B extends A {
  late var f = super.foo();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          methods
            foo @16
              reference: <testLibraryFragment>::@class::A::@method::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
        class B @37
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          fields
            late f @62
              reference: <testLibraryFragment>::@class::B::@field::f
              enclosingElement3: <testLibraryFragment>::@class::B
              type: int
              shouldUseTypeForInitializerInference: false
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
          accessors
            synthetic get f @-1
              reference: <testLibraryFragment>::@class::B::@getter::f
              enclosingElement3: <testLibraryFragment>::@class::B
              returnType: int
            synthetic set f= @-1
              reference: <testLibraryFragment>::@class::B::@setter::f
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                requiredPositional _f @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          methods
            foo @16
              reference: <testLibraryFragment>::@class::A::@method::foo
              element: <testLibraryFragment>::@class::A::@method::foo#element
        class B @37
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          fields
            hasInitializer f @62
              reference: <testLibraryFragment>::@class::B::@field::f
              element: <testLibraryFragment>::@class::B::@field::f#element
              getter2: <testLibraryFragment>::@class::B::@getter::f
              setter2: <testLibraryFragment>::@class::B::@setter::f
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
          getters
            synthetic get f
              reference: <testLibraryFragment>::@class::B::@getter::f
              element: <testLibraryFragment>::@class::B::@getter::f#element
          setters
            synthetic set f
              reference: <testLibraryFragment>::@class::B::@setter::f
              element: <testLibraryFragment>::@class::B::@setter::f#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::B::@setter::f::@parameter::_f#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      methods
        foo
          reference: <testLibrary>::@class::A::@method::foo
          firstFragment: <testLibraryFragment>::@class::A::@method::foo
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      fields
        late hasInitializer f
          firstFragment: <testLibraryFragment>::@class::B::@field::f
          type: int
          getter: <testLibraryFragment>::@class::B::@getter::f#element
          setter: <testLibraryFragment>::@class::B::@setter::f#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
      getters
        synthetic get f
          firstFragment: <testLibraryFragment>::@class::B::@getter::f
      setters
        synthetic set f
          firstFragment: <testLibraryFragment>::@class::B::@setter::f
          formalParameters
            requiredPositional _f
              type: int
''');
  }

  test_class_fields_late_inference_usingSuper_propertyAccess() async {
    var library = await buildLibrary('''
class A {
  int get foo => 0;
}

class B extends A {
  late var f = super.foo;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic foo @-1
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            get foo @20
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
        class B @39
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          fields
            late f @64
              reference: <testLibraryFragment>::@class::B::@field::f
              enclosingElement3: <testLibraryFragment>::@class::B
              type: int
              shouldUseTypeForInitializerInference: false
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
          accessors
            synthetic get f @-1
              reference: <testLibraryFragment>::@class::B::@getter::f
              enclosingElement3: <testLibraryFragment>::@class::B
              returnType: int
            synthetic set f= @-1
              reference: <testLibraryFragment>::@class::B::@setter::f
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                requiredPositional _f @-1
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            get foo @20
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
        class B @39
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          fields
            hasInitializer f @64
              reference: <testLibraryFragment>::@class::B::@field::f
              element: <testLibraryFragment>::@class::B::@field::f#element
              getter2: <testLibraryFragment>::@class::B::@getter::f
              setter2: <testLibraryFragment>::@class::B::@setter::f
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
          getters
            synthetic get f
              reference: <testLibraryFragment>::@class::B::@getter::f
              element: <testLibraryFragment>::@class::B::@getter::f#element
          setters
            synthetic set f
              reference: <testLibraryFragment>::@class::B::@setter::f
              element: <testLibraryFragment>::@class::B::@setter::f#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::B::@setter::f::@parameter::_f#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      fields
        late hasInitializer f
          firstFragment: <testLibraryFragment>::@class::B::@field::f
          type: int
          getter: <testLibraryFragment>::@class::B::@getter::f#element
          setter: <testLibraryFragment>::@class::B::@setter::f#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
      getters
        synthetic get f
          firstFragment: <testLibraryFragment>::@class::B::@getter::f
      setters
        synthetic set f
          firstFragment: <testLibraryFragment>::@class::B::@setter::f
          formalParameters
            requiredPositional _f
              type: int
''');
  }

  test_class_final() async {
    var library = await buildLibrary('final class C {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        final class C @12
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @12
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    final class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_getter_abstract() async {
    var library = await buildLibrary('abstract class C { int get x; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        abstract class C @15
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            abstract get x @27
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @15
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            get x @27
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    abstract class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        abstract get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
''');
  }

  test_class_getter_external() async {
    var library = await buildLibrary('class C { external int get x; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            external get x @27
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            get x @27
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        external get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
''');
  }

  test_class_getter_implicit_return_type() async {
    var library = await buildLibrary('class C { get x => null; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            get x @14
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            get x @14
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
''');
  }

  test_class_getter_invokesSuperSelf_getter() async {
    var library = await buildLibrary(r'''
class A {
  int get foo {
    super.foo;
  }
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic foo @-1
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            get foo @20 invokesSuperSelf
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            get foo @20
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
''');
  }

  test_class_getter_invokesSuperSelf_getter_nestedInAssignment() async {
    var library = await buildLibrary(r'''
class A {
  int get foo {
    (super.foo).foo = 0;
  }
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic foo @-1
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            get foo @20 invokesSuperSelf
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            get foo @20
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
''');
  }

  test_class_getter_invokesSuperSelf_setter() async {
    var library = await buildLibrary(r'''
class A {
  int get foo {
    super.foo = 0;
  }
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic foo @-1
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            get foo @20
              reference: <testLibraryFragment>::@class::A::@getter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              getter2: <testLibraryFragment>::@class::A::@getter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            get foo @20
              reference: <testLibraryFragment>::@class::A::@getter::foo
              element: <testLibraryFragment>::@class::A::@getter::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        get foo
          firstFragment: <testLibraryFragment>::@class::A::@getter::foo
''');
  }

  test_class_getter_missingName() async {
    var library = await buildLibrary('''
class A {
  get () => 0;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          methods
            get @12
              reference: <testLibraryFragment>::@class::A::@method::get
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: dynamic
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          methods
            get @12
              reference: <testLibraryFragment>::@class::A::@method::get
              element: <testLibraryFragment>::@class::A::@method::get#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      methods
        get
          reference: <testLibrary>::@class::A::@method::get
          firstFragment: <testLibraryFragment>::@class::A::@method::get
''');
  }

  test_class_getter_native() async {
    var library = await buildLibrary('''
class C {
  int get x() native;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            external get x @20
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            get x @20
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        external get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
''');
  }

  test_class_getter_static() async {
    var library = await buildLibrary('class C { static int get x => null; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic static x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            static get x @25
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            get x @25
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic static x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        static get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
''');
  }

  test_class_getters() async {
    var library =
        await buildLibrary('class C { int get x => null; get y => null; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
            synthetic y @-1
              reference: <testLibraryFragment>::@class::C::@field::y
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            get x @18
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            get y @33
              reference: <testLibraryFragment>::@class::C::@getter::y
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
            synthetic y
              reference: <testLibraryFragment>::@class::C::@field::y
              element: <testLibraryFragment>::@class::C::@field::y#element
              getter2: <testLibraryFragment>::@class::C::@getter::y
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            get x @18
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
            get y @33
              reference: <testLibraryFragment>::@class::C::@getter::y
              element: <testLibraryFragment>::@class::C::@getter::y#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
        synthetic y
          firstFragment: <testLibraryFragment>::@class::C::@field::y
          type: dynamic
          getter: <testLibraryFragment>::@class::C::@getter::y#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
        get y
          firstFragment: <testLibraryFragment>::@class::C::@getter::y
''');
  }

  test_class_implicitField_getterFirst() async {
    var library = await buildLibrary('''
class C {
  int get x => 0;
  void set x(int value) {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            get x @20
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
            set x= @39
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional value @45
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            get x @20
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            set x @39
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @45
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional value
              type: int
''');
  }

  test_class_implicitField_setterFirst() async {
    var library = await buildLibrary('''
class C {
  void set x(int value) {}
  int get x => 0;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            set x= @21
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional value @27
                  type: int
              returnType: void
            get x @47
              reference: <testLibraryFragment>::@class::C::@getter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              getter2: <testLibraryFragment>::@class::C::@getter::x
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          getters
            get x @47
              reference: <testLibraryFragment>::@class::C::@getter::x
              element: <testLibraryFragment>::@class::C::@getter::x#element
          setters
            set x @21
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @27
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          getter: <testLibraryFragment>::@class::C::@getter::x#element
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      getters
        get x
          firstFragment: <testLibraryFragment>::@class::C::@getter::x
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional value
              type: int
''');
  }

  test_class_interface() async {
    var library = await buildLibrary('interface class C {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        interface class C @16
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @16
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    interface class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_interfaces() async {
    var library = await buildLibrary('''
class C implements D, E {}
class D {}
class E {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          interfaces
            D
            E
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
        class D @33
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
        class E @44
          reference: <testLibraryFragment>::@class::E
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::E::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::E
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @33
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @44
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      interfaces
        D
        E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
''');
  }

  test_class_interfaces_extensionType() async {
    var library = await buildLibrary('''
class A {}
extension type B(int it) {}
class C {}
class D implements A, B, C {}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
        class C @45
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
        class D @56
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          interfaces
            A
            C
      extensionTypes
        B @26
          reference: <testLibraryFragment>::@extensionType::B
          enclosingElement3: <testLibraryFragment>
          representation: <testLibraryFragment>::@extensionType::B::@field::it
          primaryConstructor: <testLibraryFragment>::@extensionType::B::@constructor::new
          typeErasure: int
          fields
            final it @32
              reference: <testLibraryFragment>::@extensionType::B::@field::it
              enclosingElement3: <testLibraryFragment>::@extensionType::B
              type: int
          accessors
            synthetic get it @-1
              reference: <testLibraryFragment>::@extensionType::B::@getter::it
              enclosingElement3: <testLibraryFragment>::@extensionType::B
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
        class C @45
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
        class D @56
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
      extensionTypes
        extension type B @26
          reference: <testLibraryFragment>::@extensionType::B
          element: <testLibrary>::@extensionType::B
          fields
            it @32
              reference: <testLibraryFragment>::@extensionType::B::@field::it
              element: <testLibraryFragment>::@extensionType::B::@field::it#element
              getter2: <testLibraryFragment>::@extensionType::B::@getter::it
          getters
            synthetic get it
              reference: <testLibraryFragment>::@extensionType::B::@getter::it
              element: <testLibraryFragment>::@extensionType::B::@getter::it#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      interfaces
        A
        C
  extensionTypes
    extension type B
      reference: <testLibrary>::@extensionType::B
      firstFragment: <testLibraryFragment>::@extensionType::B
      representation: <testLibraryFragment>::@extensionType::B::@field::it#element
      primaryConstructor: <testLibraryFragment>::@extensionType::B::@constructor::new#element
      typeErasure: int
      fields
        final it
          firstFragment: <testLibraryFragment>::@extensionType::B::@field::it
          type: int
          getter: <testLibraryFragment>::@extensionType::B::@getter::it#element
      getters
        synthetic get it
          firstFragment: <testLibraryFragment>::@extensionType::B::@getter::it
''');
  }

  test_class_interfaces_Function() async {
    var library = await buildLibrary('''
class A {}
class B {}
class C implements A, Function, B {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
        class B @17
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
        class C @28
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          interfaces
            A
            B
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @17
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
        class C @28
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      interfaces
        A
        B
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_interfaces_unresolved() async {
    var library = await buildLibrary(
        'class C implements X, Y, Z {} class X {} class Z {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          interfaces
            X
            Z
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
        class X @36
          reference: <testLibraryFragment>::@class::X
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::X::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::X
        class Z @47
          reference: <testLibraryFragment>::@class::Z
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::Z::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::Z
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class X @36
          reference: <testLibraryFragment>::@class::X
          element: <testLibrary>::@class::X
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::X::@constructor::new
              element: <testLibraryFragment>::@class::X::@constructor::new#element
              typeName: X
        class Z @47
          reference: <testLibraryFragment>::@class::Z
          element: <testLibrary>::@class::Z
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::Z::@constructor::new
              element: <testLibraryFragment>::@class::Z::@constructor::new#element
              typeName: Z
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      interfaces
        X
        Z
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class X
      reference: <testLibrary>::@class::X
      firstFragment: <testLibraryFragment>::@class::X
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::X::@constructor::new
    class Z
      reference: <testLibrary>::@class::Z
      firstFragment: <testLibraryFragment>::@class::Z
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::Z::@constructor::new
''');
  }

  test_class_lazy_constructors() async {
    var library = await buildLibrary('''
class A {
  A.named();
}
''');

    var constructors = library.getClass2('A')!.constructors2;
    expect(constructors, hasLength(1));
  }

  test_class_lazy_fields() async {
    var library = await buildLibrary('''
class A {
  int foo = 0;
}
''');

    var fields = library.getClass2('A')!.fields2;
    expect(fields, hasLength(1));
  }

  test_class_lazy_getters() async {
    var library = await buildLibrary('''
class A {
  int foo = 0;
}
''');

    var getters = library.getClass2('A')!.getters2;
    expect(getters, hasLength(1));
  }

  test_class_lazy_methods() async {
    var library = await buildLibrary('''
class A {
  void foo() {}
}
''');

    var methods = library.getClass2('A')!.methods2;
    expect(methods, hasLength(1));
  }

  test_class_lazy_setters() async {
    var library = await buildLibrary('''
class A {
  int foo = 0;
}
''');

    var setters = library.getClass2('A')!.setters2;
    expect(setters, hasLength(1));
  }

  test_class_method_abstract() async {
    var library = await buildLibrary('abstract class C { f(); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        abstract class C @15
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            abstract f @19
              reference: <testLibraryFragment>::@class::C::@method::f
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @15
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @19
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
  classes
    abstract class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        abstract f
          reference: <testLibrary>::@class::C::@method::f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
''');
  }

  test_class_method_async() async {
    var library = await buildLibrary(r'''
import 'dart:async';
class C {
  Future f() async {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      libraryImports
        dart:async
          enclosingElement3: <testLibraryFragment>
      classes
        class C @27
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            f @40 async
              reference: <testLibraryFragment>::@class::C::@method::f
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: Future<dynamic>
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        dart:async
      classes
        class C @27
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @40 async
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          reference: <testLibrary>::@class::C::@method::f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
''');
  }

  test_class_method_asyncStar() async {
    var library = await buildLibrary(r'''
import 'dart:async';
class C {
  Stream f() async* {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      libraryImports
        dart:async
          enclosingElement3: <testLibraryFragment>
      classes
        class C @27
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            f @40 async*
              reference: <testLibraryFragment>::@class::C::@method::f
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: Stream<dynamic>
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      libraryImports
        dart:async
      classes
        class C @27
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @40 async*
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          reference: <testLibrary>::@class::C::@method::f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
''');
  }

  test_class_method_documented() async {
    var library = await buildLibrary('''
class C {
  /**
   * Docs
   */
  f() {}
}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            f @34
              reference: <testLibraryFragment>::@class::C::@method::f
              enclosingElement3: <testLibraryFragment>::@class::C
              documentationComment: /**\n   * Docs\n   */
              returnType: dynamic
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @34
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
              documentationComment: /**\n   * Docs\n   */
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          reference: <testLibrary>::@class::C::@method::f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          documentationComment: /**\n   * Docs\n   */
''');
  }

  test_class_method_external() async {
    var library = await buildLibrary('class C { external f(); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            external f @19
              reference: <testLibraryFragment>::@class::C::@method::f
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @19
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        external f
          reference: <testLibrary>::@class::C::@method::f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
''');
  }

  test_class_method_hasImplicitReturnType_false() async {
    var library = await buildLibrary('''
class C {
  int m() => 0;
}
''');
    var c = library.definingCompilationUnit.classes.single;
    var m = c.methods.single;
    expect(m.hasImplicitReturnType, isFalse);
  }

  test_class_method_hasImplicitReturnType_true() async {
    var library = await buildLibrary('''
class C {
  m() => 0;
}
''');
    var c = library.definingCompilationUnit.classes.single;
    var m = c.methods.single;
    expect(m.hasImplicitReturnType, isTrue);
  }

  test_class_method_inferred_type_nonStatic_implicit_param() async {
    var library = await buildLibrary('class C extends D { void f(value) {} }'
        ' abstract class D { void f(int value); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: D
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              superConstructor: <testLibraryFragment>::@class::D::@constructor::new
          methods
            f @25
              reference: <testLibraryFragment>::@class::C::@method::f
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional hasImplicitType value @27
                  type: int
              returnType: void
        abstract class D @54
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
          methods
            abstract f @63
              reference: <testLibraryFragment>::@class::D::@method::f
              enclosingElement3: <testLibraryFragment>::@class::D
              parameters
                requiredPositional value @69
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              superConstructor: <testLibraryFragment>::@class::D::@constructor::new
          methods
            f @25
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
              formalParameters
                value @27
                  element: <testLibraryFragment>::@class::C::@method::f::@parameter::value#element
        class D @54
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
          methods
            f @63
              reference: <testLibraryFragment>::@class::D::@method::f
              element: <testLibraryFragment>::@class::D::@method::f#element
              formalParameters
                value @69
                  element: <testLibraryFragment>::@class::D::@method::f::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
      methods
        f
          reference: <testLibrary>::@class::C::@method::f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          formalParameters
            requiredPositional hasImplicitType value
              type: int
    abstract class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
      methods
        abstract f
          reference: <testLibrary>::@class::D::@method::f
          firstFragment: <testLibraryFragment>::@class::D::@method::f
          formalParameters
            requiredPositional value
              type: int
''');
  }

  test_class_method_inferred_type_nonStatic_implicit_return() async {
    var library = await buildLibrary('''
class C extends D {
  f() => null;
}
abstract class D {
  int f();
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: D
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              superConstructor: <testLibraryFragment>::@class::D::@constructor::new
          methods
            f @22
              reference: <testLibraryFragment>::@class::C::@method::f
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
        abstract class D @52
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
          methods
            abstract f @62
              reference: <testLibraryFragment>::@class::D::@method::f
              enclosingElement3: <testLibraryFragment>::@class::D
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              superConstructor: <testLibraryFragment>::@class::D::@constructor::new
          methods
            f @22
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
        class D @52
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
          methods
            f @62
              reference: <testLibraryFragment>::@class::D::@method::f
              element: <testLibraryFragment>::@class::D::@method::f#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
      methods
        f
          reference: <testLibrary>::@class::C::@method::f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
    abstract class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
      methods
        abstract f
          reference: <testLibrary>::@class::D::@method::f
          firstFragment: <testLibraryFragment>::@class::D::@method::f
''');
  }

  test_class_method_invokesSuperSelf() async {
    var library = await buildLibrary(r'''
class A {
  void foo() {
    super.foo();
  }
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          methods
            foo @17 invokesSuperSelf
              reference: <testLibraryFragment>::@class::A::@method::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          methods
            foo @17 invokesSuperSelf
              reference: <testLibraryFragment>::@class::A::@method::foo
              element: <testLibraryFragment>::@class::A::@method::foo#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      methods
        foo
          reference: <testLibrary>::@class::A::@method::foo
          firstFragment: <testLibraryFragment>::@class::A::@method::foo
''');
  }

  test_class_method_missingName() async {
    var library = await buildLibrary('''
class A {
  () {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          methods
            @12
              reference: <testLibraryFragment>::@class::A::@method::0
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: dynamic
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          methods
            <null-name>
              reference: <testLibraryFragment>::@class::A::@method::0
              element: <testLibraryFragment>::@class::A::@method::0#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      methods
        <null-name>
          reference: <testLibrary>::@class::A::@method::0
          firstFragment: <testLibraryFragment>::@class::A::@method::0
''');
  }

  test_class_method_namedAsSupertype() async {
    var library = await buildLibrary(r'''
class A {}
class B extends A {
  void A() {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
        class B @17
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
          methods
            A @38
              reference: <testLibraryFragment>::@class::B::@method::A
              enclosingElement3: <testLibraryFragment>::@class::B
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @17
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
          methods
            A @38
              reference: <testLibraryFragment>::@class::B::@method::A
              element: <testLibraryFragment>::@class::B::@method::A#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
      methods
        A
          reference: <testLibrary>::@class::B::@method::A
          firstFragment: <testLibraryFragment>::@class::B::@method::A
''');
  }

  test_class_method_native() async {
    var library = await buildLibrary('''
class C {
  int m() native;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            external m @16
              reference: <testLibraryFragment>::@class::C::@method::m
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            m @16
              reference: <testLibraryFragment>::@class::C::@method::m
              element: <testLibraryFragment>::@class::C::@method::m#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        external m
          reference: <testLibrary>::@class::C::@method::m
          firstFragment: <testLibraryFragment>::@class::C::@method::m
''');
  }

  test_class_method_params() async {
    var library = await buildLibrary('class C { f(x, y) {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            f @10
              reference: <testLibraryFragment>::@class::C::@method::f
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional hasImplicitType x @12
                  type: dynamic
                requiredPositional hasImplicitType y @15
                  type: dynamic
              returnType: dynamic
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @10
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
              formalParameters
                x @12
                  element: <testLibraryFragment>::@class::C::@method::f::@parameter::x#element
                y @15
                  element: <testLibraryFragment>::@class::C::@method::f::@parameter::y#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          reference: <testLibrary>::@class::C::@method::f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          formalParameters
            requiredPositional hasImplicitType x
              type: dynamic
            requiredPositional hasImplicitType y
              type: dynamic
''');
  }

  test_class_method_static() async {
    var library = await buildLibrary('class C { static f() {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            static f @17
              reference: <testLibraryFragment>::@class::C::@method::f
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @17
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        static f
          reference: <testLibrary>::@class::C::@method::f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
''');
  }

  test_class_method_syncStar() async {
    var library = await buildLibrary(r'''
class C {
  Iterable<int> f() sync* {
    yield 42;
  }
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            f @26 sync*
              reference: <testLibraryFragment>::@class::C::@method::f
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: Iterable<int>
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @26 sync*
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          reference: <testLibrary>::@class::C::@method::f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
''');
  }

  test_class_method_type_parameter() async {
    var library = await buildLibrary('class C { T f<T, U>(U u) => null; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            f @12
              reference: <testLibraryFragment>::@class::C::@method::f
              enclosingElement3: <testLibraryFragment>::@class::C
              typeParameters
                covariant T @14
                  defaultType: dynamic
                covariant U @17
                  defaultType: dynamic
              parameters
                requiredPositional u @22
                  type: U
              returnType: T
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @12
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
              typeParameters
                T @14
                  element: <not-implemented>
                U @17
                  element: <not-implemented>
              formalParameters
                u @22
                  element: <testLibraryFragment>::@class::C::@method::f::@parameter::u#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          reference: <testLibrary>::@class::C::@method::f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          typeParameters
            T
            U
          formalParameters
            requiredPositional u
              type: U
''');
  }

  test_class_method_type_parameter_in_generic_class() async {
    var library = await buildLibrary('''
class C<T, U> {
  V f<V, W>(T t, U u, W w) => null;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              defaultType: dynamic
            covariant U @11
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            f @20
              reference: <testLibraryFragment>::@class::C::@method::f
              enclosingElement3: <testLibraryFragment>::@class::C
              typeParameters
                covariant V @22
                  defaultType: dynamic
                covariant W @25
                  defaultType: dynamic
              parameters
                requiredPositional t @30
                  type: T
                requiredPositional u @35
                  type: U
                requiredPositional w @40
                  type: W
              returnType: V
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
            U @11
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @20
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
              typeParameters
                V @22
                  element: <not-implemented>
                W @25
                  element: <not-implemented>
              formalParameters
                t @30
                  element: <testLibraryFragment>::@class::C::@method::f::@parameter::t#element
                u @35
                  element: <testLibraryFragment>::@class::C::@method::f::@parameter::u#element
                w @40
                  element: <testLibraryFragment>::@class::C::@method::f::@parameter::w#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          reference: <testLibrary>::@class::C::@method::f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          typeParameters
            V
            W
          formalParameters
            requiredPositional t
              type: T
            requiredPositional u
              type: U
            requiredPositional w
              type: W
''');
  }

  test_class_method_type_parameter_with_function_typed_parameter() async {
    var library = await buildLibrary('class C { void f<T, U>(T x(U u)) {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            f @15
              reference: <testLibraryFragment>::@class::C::@method::f
              enclosingElement3: <testLibraryFragment>::@class::C
              typeParameters
                covariant T @17
                  defaultType: dynamic
                covariant U @20
                  defaultType: dynamic
              parameters
                requiredPositional x @25
                  type: T Function(U)
                  parameters
                    requiredPositional u @29
                      type: U
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @15
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
              typeParameters
                T @17
                  element: <not-implemented>
                U @20
                  element: <not-implemented>
              formalParameters
                x @25
                  element: <testLibraryFragment>::@class::C::@method::f::@parameter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          reference: <testLibrary>::@class::C::@method::f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
          typeParameters
            T
            U
          formalParameters
            requiredPositional x
              type: T Function(U)
              formalParameters
                requiredPositional u
                  type: U
''');
  }

  test_class_methods() async {
    var library = await buildLibrary('class C { f() {} g() {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            f @10
              reference: <testLibraryFragment>::@class::C::@method::f
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
            g @17
              reference: <testLibraryFragment>::@class::C::@method::g
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: dynamic
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            f @10
              reference: <testLibraryFragment>::@class::C::@method::f
              element: <testLibraryFragment>::@class::C::@method::f#element
            g @17
              reference: <testLibraryFragment>::@class::C::@method::g
              element: <testLibraryFragment>::@class::C::@method::g#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        f
          reference: <testLibrary>::@class::C::@method::f
          firstFragment: <testLibraryFragment>::@class::C::@method::f
        g
          reference: <testLibrary>::@class::C::@method::g
          firstFragment: <testLibraryFragment>::@class::C::@method::g
''');
  }

  test_class_missingName() async {
    var library = await buildLibrary(r'''
class {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class @6
          reference: <testLibraryFragment>::@class::0
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::0::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::0
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class <null-name>
          reference: <testLibraryFragment>::@class::0
          element: <testLibrary>::@class::0
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::0::@constructor::new
              element: <testLibraryFragment>::@class::0::@constructor::new#element
              typeName: null
  classes
    class 
      reference: <testLibrary>::@class::0
      firstFragment: <testLibraryFragment>::@class::0
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::0::@constructor::new
''');
  }

  test_class_mixin_class() async {
    var library = await buildLibrary('mixin class C {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        mixin class C @12
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @12
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    mixin class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_mixins() async {
    var library = await buildLibrary('''
class C extends D with E, F, G {}
class D {}
class E {}
class F {}
class G {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: D
          mixins
            E
            F
            G
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              superConstructor: <testLibraryFragment>::@class::D::@constructor::new
        class D @40
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
        class E @51
          reference: <testLibraryFragment>::@class::E
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::E::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::E
        class F @62
          reference: <testLibraryFragment>::@class::F
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::F::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::F
        class G @73
          reference: <testLibraryFragment>::@class::G
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::G::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::G
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              superConstructor: <testLibraryFragment>::@class::D::@constructor::new
        class D @40
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @51
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
        class F @62
          reference: <testLibraryFragment>::@class::F
          element: <testLibrary>::@class::F
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::F::@constructor::new
              element: <testLibraryFragment>::@class::F::@constructor::new#element
              typeName: F
        class G @73
          reference: <testLibraryFragment>::@class::G
          element: <testLibrary>::@class::G
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::G::@constructor::new
              element: <testLibraryFragment>::@class::G::@constructor::new#element
              typeName: G
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D
      mixins
        E
        F
        G
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
    class F
      reference: <testLibrary>::@class::F
      firstFragment: <testLibraryFragment>::@class::F
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::F::@constructor::new
    class G
      reference: <testLibrary>::@class::G
      firstFragment: <testLibraryFragment>::@class::G
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::G::@constructor::new
''');
  }

  test_class_mixins_extensionType() async {
    var library = await buildLibrary('''
mixin A {}
extension type B(int it) {}
mixin C {}
class D extends Object with A, B, C {}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class D @56
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          supertype: Object
          mixins
            A
            C
      extensionTypes
        B @26
          reference: <testLibraryFragment>::@extensionType::B
          enclosingElement3: <testLibraryFragment>
          representation: <testLibraryFragment>::@extensionType::B::@field::it
          primaryConstructor: <testLibraryFragment>::@extensionType::B::@constructor::new
          typeErasure: int
          fields
            final it @32
              reference: <testLibraryFragment>::@extensionType::B::@field::it
              enclosingElement3: <testLibraryFragment>::@extensionType::B
              type: int
          accessors
            synthetic get it @-1
              reference: <testLibraryFragment>::@extensionType::B::@getter::it
              enclosingElement3: <testLibraryFragment>::@extensionType::B
              returnType: int
      mixins
        mixin A @6
          reference: <testLibraryFragment>::@mixin::A
          enclosingElement3: <testLibraryFragment>
          superclassConstraints
            Object
        mixin C @45
          reference: <testLibraryFragment>::@mixin::C
          enclosingElement3: <testLibraryFragment>
          superclassConstraints
            Object
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class D @56
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
      extensionTypes
        extension type B @26
          reference: <testLibraryFragment>::@extensionType::B
          element: <testLibrary>::@extensionType::B
          fields
            it @32
              reference: <testLibraryFragment>::@extensionType::B::@field::it
              element: <testLibraryFragment>::@extensionType::B::@field::it#element
              getter2: <testLibraryFragment>::@extensionType::B::@getter::it
          getters
            synthetic get it
              reference: <testLibraryFragment>::@extensionType::B::@getter::it
              element: <testLibraryFragment>::@extensionType::B::@getter::it#element
      mixins
        mixin A @6
          reference: <testLibraryFragment>::@mixin::A
          element: <testLibrary>::@mixin::A
        mixin C @45
          reference: <testLibraryFragment>::@mixin::C
          element: <testLibrary>::@mixin::C
  classes
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      supertype: Object
      mixins
        A
        C
  extensionTypes
    extension type B
      reference: <testLibrary>::@extensionType::B
      firstFragment: <testLibraryFragment>::@extensionType::B
      representation: <testLibraryFragment>::@extensionType::B::@field::it#element
      primaryConstructor: <testLibraryFragment>::@extensionType::B::@constructor::new#element
      typeErasure: int
      fields
        final it
          firstFragment: <testLibraryFragment>::@extensionType::B::@field::it
          type: int
          getter: <testLibraryFragment>::@extensionType::B::@getter::it#element
      getters
        synthetic get it
          firstFragment: <testLibraryFragment>::@extensionType::B::@getter::it
  mixins
    mixin A
      reference: <testLibrary>::@mixin::A
      firstFragment: <testLibraryFragment>::@mixin::A
      superclassConstraints
        Object
    mixin C
      reference: <testLibrary>::@mixin::C
      firstFragment: <testLibraryFragment>::@mixin::C
      superclassConstraints
        Object
''');
  }

  test_class_mixins_generic() async {
    var library = await buildLibrary('''
class Z extends A with B<int>, C<double> {}
class A {}
class B<B1> {}
class C<C1> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class Z @6
          reference: <testLibraryFragment>::@class::Z
          enclosingElement3: <testLibraryFragment>
          supertype: A
          mixins
            B<int>
            C<double>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::Z::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::Z
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
        class A @50
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
        class B @61
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant B1 @63
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
        class C @76
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant C1 @78
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class Z @6
          reference: <testLibraryFragment>::@class::Z
          element: <testLibrary>::@class::Z
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::Z::@constructor::new
              element: <testLibraryFragment>::@class::Z::@constructor::new#element
              typeName: Z
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
        class A @50
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @61
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          typeParameters
            B1 @63
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
        class C @76
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            C1 @78
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class Z
      reference: <testLibrary>::@class::Z
      firstFragment: <testLibraryFragment>::@class::Z
      supertype: A
      mixins
        B<int>
        C<double>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::Z::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      typeParameters
        B1
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        C1
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_mixins_generic_superAfter() async {
    var library = await buildLibrary('''
mixin M<T extends num> {}
mixin M2<T extends num> on M<T> {}
class Z extends S with M2 {}
class S with M<int> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class Z @67
          reference: <testLibraryFragment>::@class::Z
          enclosingElement3: <testLibraryFragment>
          supertype: S
          mixins
            M2<int>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::Z::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::Z
              superConstructor: <testLibraryFragment>::@class::S::@constructor::new
        class S @96
          reference: <testLibraryFragment>::@class::S
          enclosingElement3: <testLibraryFragment>
          supertype: Object
          mixins
            M<int>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::S::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::S
      mixins
        mixin M @6
          reference: <testLibraryFragment>::@mixin::M
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: num
              defaultType: num
          superclassConstraints
            Object
        mixin M2 @32
          reference: <testLibraryFragment>::@mixin::M2
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @35
              bound: num
              defaultType: num
          superclassConstraints
            M<T>
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class Z @67
          reference: <testLibraryFragment>::@class::Z
          element: <testLibrary>::@class::Z
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::Z::@constructor::new
              element: <testLibraryFragment>::@class::Z::@constructor::new#element
              typeName: Z
              superConstructor: <testLibraryFragment>::@class::S::@constructor::new
        class S @96
          reference: <testLibraryFragment>::@class::S
          element: <testLibrary>::@class::S
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::S::@constructor::new
              element: <testLibraryFragment>::@class::S::@constructor::new#element
              typeName: S
      mixins
        mixin M @6
          reference: <testLibraryFragment>::@mixin::M
          element: <testLibrary>::@mixin::M
          typeParameters
            T @8
              element: <not-implemented>
        mixin M2 @32
          reference: <testLibraryFragment>::@mixin::M2
          element: <testLibrary>::@mixin::M2
          typeParameters
            T @35
              element: <not-implemented>
  classes
    class Z
      reference: <testLibrary>::@class::Z
      firstFragment: <testLibraryFragment>::@class::Z
      supertype: S
      mixins
        M2<int>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::Z::@constructor::new
          superConstructor: <testLibraryFragment>::@class::S::@constructor::new#element
    class S
      reference: <testLibrary>::@class::S
      firstFragment: <testLibraryFragment>::@class::S
      supertype: Object
      mixins
        M<int>
        M<int>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::S::@constructor::new
  mixins
    mixin M
      reference: <testLibrary>::@mixin::M
      firstFragment: <testLibraryFragment>::@mixin::M
      typeParameters
        T
          bound: num
      superclassConstraints
        Object
    mixin M2
      reference: <testLibrary>::@mixin::M2
      firstFragment: <testLibraryFragment>::@mixin::M2
      typeParameters
        T
          bound: num
      superclassConstraints
        M<T>
''');
  }

  test_class_mixins_genericMixin_tooManyArguments() async {
    var library = await buildLibrary('''
mixin M<T> {}
class A extends Object with M<int, String> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @20
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          supertype: Object
          mixins
            M<dynamic>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
      mixins
        mixin M @6
          reference: <testLibraryFragment>::@mixin::M
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              defaultType: dynamic
          superclassConstraints
            Object
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @20
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
      mixins
        mixin M @6
          reference: <testLibraryFragment>::@mixin::M
          element: <testLibrary>::@mixin::M
          typeParameters
            T @8
              element: <not-implemented>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      supertype: Object
      mixins
        M<dynamic>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
  mixins
    mixin M
      reference: <testLibrary>::@mixin::M
      firstFragment: <testLibraryFragment>::@mixin::M
      typeParameters
        T
      superclassConstraints
        Object
''');
  }

  test_class_mixins_typeParameter() async {
    var library = await buildLibrary('''
mixin M1 {}
mixin M2 {}
class A<T> extends Object with M1, T<int>, M2 {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @30
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @32
              defaultType: dynamic
          supertype: Object
          mixins
            M1
            M2
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
      mixins
        mixin M1 @6
          reference: <testLibraryFragment>::@mixin::M1
          enclosingElement3: <testLibraryFragment>
          superclassConstraints
            Object
        mixin M2 @18
          reference: <testLibraryFragment>::@mixin::M2
          enclosingElement3: <testLibraryFragment>
          superclassConstraints
            Object
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @30
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @32
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
      mixins
        mixin M1 @6
          reference: <testLibraryFragment>::@mixin::M1
          element: <testLibrary>::@mixin::M1
        mixin M2 @18
          reference: <testLibraryFragment>::@mixin::M2
          element: <testLibrary>::@mixin::M2
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
      supertype: Object
      mixins
        M1
        M2
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
  mixins
    mixin M1
      reference: <testLibrary>::@mixin::M1
      firstFragment: <testLibraryFragment>::@mixin::M1
      superclassConstraints
        Object
    mixin M2
      reference: <testLibrary>::@mixin::M2
      firstFragment: <testLibraryFragment>::@mixin::M2
      superclassConstraints
        Object
''');
  }

  test_class_mixins_unresolved() async {
    var library = await buildLibrary(
        'class C extends Object with X, Y, Z {} class X {} class Z {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: Object
          mixins
            X
            Z
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
        class X @45
          reference: <testLibraryFragment>::@class::X
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::X::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::X
        class Z @56
          reference: <testLibraryFragment>::@class::Z
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::Z::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::Z
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class X @45
          reference: <testLibraryFragment>::@class::X
          element: <testLibrary>::@class::X
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::X::@constructor::new
              element: <testLibraryFragment>::@class::X::@constructor::new#element
              typeName: X
        class Z @56
          reference: <testLibraryFragment>::@class::Z
          element: <testLibrary>::@class::Z
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::Z::@constructor::new
              element: <testLibraryFragment>::@class::Z::@constructor::new#element
              typeName: Z
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: Object
      mixins
        X
        Z
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class X
      reference: <testLibrary>::@class::X
      firstFragment: <testLibraryFragment>::@class::X
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::X::@constructor::new
    class Z
      reference: <testLibrary>::@class::Z
      firstFragment: <testLibraryFragment>::@class::Z
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::Z::@constructor::new
''');
  }

  test_class_notSimplyBounded_circularity_via_typeAlias_recordType() async {
    var library = await buildLibrary('''
class C<T extends A> {}
typedef A = (C, int);
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: dynamic
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
      typeAliases
        notSimplyBounded A @32
          reference: <testLibraryFragment>::@typeAlias::A
          aliasedType: (C<dynamic>, int)
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
      typeAliases
        A @32
          reference: <testLibraryFragment>::@typeAlias::A
          element: <testLibrary>::@typeAlias::A
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: dynamic
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
  typeAliases
    notSimplyBounded A
      firstFragment: <testLibraryFragment>::@typeAlias::A
      aliasedType: (C<dynamic>, int)
''');
  }

  test_class_notSimplyBounded_circularity_via_typedef() async {
    // C's type parameter T is not simply bounded because its bound, F, expands
    // to `dynamic F(C)`, which refers to C.
    var library = await buildLibrary('''
class C<T extends F> {}
typedef F(C value);
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: dynamic
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
      typeAliases
        functionTypeAliasBased notSimplyBounded F @32
          reference: <testLibraryFragment>::@typeAlias::F
          aliasedType: dynamic Function(C<dynamic>)
          aliasedElement: GenericFunctionTypeElement
            parameters
              requiredPositional value @36
                type: C<dynamic>
            returnType: dynamic
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
      typeAliases
        F @32
          reference: <testLibraryFragment>::@typeAlias::F
          element: <testLibrary>::@typeAlias::F
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: dynamic
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
  typeAliases
    notSimplyBounded F
      firstFragment: <testLibraryFragment>::@typeAlias::F
      aliasedType: dynamic Function(C<dynamic>)
''');
  }

  test_class_notSimplyBounded_circularity_with_type_params() async {
    // C's type parameter T is simply bounded because even though it refers to
    // C, it specifies a bound.
    var library = await buildLibrary('''
class C<T extends C<dynamic>> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: C<dynamic>
              defaultType: C<dynamic>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: C<dynamic>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_notSimplyBounded_complex_by_cycle_class() async {
    var library = await buildLibrary('''
class C<T extends D> {}
class D<T extends C> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: D<dynamic>
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
        notSimplyBounded class D @30
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @32
              bound: C<dynamic>
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @30
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          typeParameters
            T @32
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: D<dynamic>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      typeParameters
        T
          bound: C<dynamic>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
''');
  }

  test_class_notSimplyBounded_complex_by_cycle_typedef_functionType() async {
    var library = await buildLibrary('''
typedef C<T extends D> = void Function();
typedef D<T extends C> = void Function();
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      typeAliases
        notSimplyBounded C @8
          reference: <testLibraryFragment>::@typeAlias::C
          typeParameters
            unrelated T @10
              bound: dynamic
              defaultType: dynamic
          aliasedType: void Function()
          aliasedElement: GenericFunctionTypeElement
            returnType: void
        notSimplyBounded D @50
          reference: <testLibraryFragment>::@typeAlias::D
          typeParameters
            unrelated T @52
              bound: dynamic
              defaultType: dynamic
          aliasedType: void Function()
          aliasedElement: GenericFunctionTypeElement
            returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      typeAliases
        C @8
          reference: <testLibraryFragment>::@typeAlias::C
          element: <testLibrary>::@typeAlias::C
          typeParameters
            T @10
              element: <not-implemented>
        D @50
          reference: <testLibraryFragment>::@typeAlias::D
          element: <testLibrary>::@typeAlias::D
          typeParameters
            T @52
              element: <not-implemented>
  typeAliases
    notSimplyBounded C
      firstFragment: <testLibraryFragment>::@typeAlias::C
      typeParameters
        T
          bound: dynamic
      aliasedType: void Function()
    notSimplyBounded D
      firstFragment: <testLibraryFragment>::@typeAlias::D
      typeParameters
        T
          bound: dynamic
      aliasedType: void Function()
''');
  }

  test_class_notSimplyBounded_complex_by_cycle_typedef_interfaceType() async {
    var library = await buildLibrary('''
typedef C<T extends D> = List<T>;
typedef D<T extends C> = List<T>;
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      typeAliases
        notSimplyBounded C @8
          reference: <testLibraryFragment>::@typeAlias::C
          typeParameters
            covariant T @10
              bound: dynamic
              defaultType: dynamic
          aliasedType: List<T>
        notSimplyBounded D @42
          reference: <testLibraryFragment>::@typeAlias::D
          typeParameters
            covariant T @44
              bound: dynamic
              defaultType: dynamic
          aliasedType: List<T>
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      typeAliases
        C @8
          reference: <testLibraryFragment>::@typeAlias::C
          element: <testLibrary>::@typeAlias::C
          typeParameters
            T @10
              element: <not-implemented>
        D @42
          reference: <testLibraryFragment>::@typeAlias::D
          element: <testLibrary>::@typeAlias::D
          typeParameters
            T @44
              element: <not-implemented>
  typeAliases
    notSimplyBounded C
      firstFragment: <testLibraryFragment>::@typeAlias::C
      typeParameters
        T
          bound: dynamic
      aliasedType: List<T>
    notSimplyBounded D
      firstFragment: <testLibraryFragment>::@typeAlias::D
      typeParameters
        T
          bound: dynamic
      aliasedType: List<T>
''');
  }

  test_class_notSimplyBounded_complex_by_reference_to_cycle() async {
    var library = await buildLibrary('''
class C<T extends D> {}
class D<T extends D> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: D<dynamic>
              defaultType: D<dynamic>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
        notSimplyBounded class D @30
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @32
              bound: D<dynamic>
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @30
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          typeParameters
            T @32
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: D<dynamic>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      typeParameters
        T
          bound: D<dynamic>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
''');
  }

  test_class_notSimplyBounded_complex_by_use_of_parameter() async {
    var library = await buildLibrary('''
class C<T extends D<T>> {}
class D<T> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: D<T>
              defaultType: D<dynamic>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
        class D @33
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @35
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @33
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          typeParameters
            T @35
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: D<T>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      typeParameters
        T
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
''');
  }

  test_class_notSimplyBounded_dependency_with_type_params() async {
    // C's type parameter T is simply bounded because even though it refers to
    // non-simply-bounded type D, it specifies a bound.
    var library = await buildLibrary('''
class C<T extends D<dynamic>> {}
class D<T extends D<T>> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: D<dynamic>
              defaultType: D<dynamic>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
        notSimplyBounded class D @39
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @41
              bound: D<T>
              defaultType: D<dynamic>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @39
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          typeParameters
            T @41
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: D<dynamic>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      typeParameters
        T
          bound: D<T>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
''');
  }

  test_class_notSimplyBounded_function_typed_bound_complex_via_parameter_type() async {
    var library = await buildLibrary('''
class C<T extends void Function(T)> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: void Function(T)
              defaultType: void Function(Never)
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: void Function(T)
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_notSimplyBounded_function_typed_bound_complex_via_return_type() async {
    var library = await buildLibrary('''
class C<T extends T Function()> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: T Function()
              defaultType: dynamic Function()
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: T Function()
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_notSimplyBounded_function_typed_bound_simple() async {
    var library = await buildLibrary('''
class C<T extends void Function()> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: void Function()
              defaultType: void Function()
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: void Function()
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_notSimplyBounded_refers_to_circular_typedef() async {
    // C's type parameter T has a bound of F, which is a circular typedef.  This
    // is illegal in Dart, but we need to make sure it doesn't lead to a crash
    // or infinite loop.
    var library = await buildLibrary('''
class C<T extends F> {}
typedef F(G value);
typedef G(F value);
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: dynamic
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
      typeAliases
        functionTypeAliasBased notSimplyBounded F @32
          reference: <testLibraryFragment>::@typeAlias::F
          aliasedType: dynamic Function(dynamic)
          aliasedElement: GenericFunctionTypeElement
            parameters
              requiredPositional value @36
                type: dynamic
            returnType: dynamic
        functionTypeAliasBased notSimplyBounded G @52
          reference: <testLibraryFragment>::@typeAlias::G
          aliasedType: dynamic Function(dynamic)
          aliasedElement: GenericFunctionTypeElement
            parameters
              requiredPositional value @56
                type: dynamic
            returnType: dynamic
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
      typeAliases
        F @32
          reference: <testLibraryFragment>::@typeAlias::F
          element: <testLibrary>::@typeAlias::F
        G @52
          reference: <testLibraryFragment>::@typeAlias::G
          element: <testLibrary>::@typeAlias::G
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: dynamic
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
  typeAliases
    notSimplyBounded F
      firstFragment: <testLibraryFragment>::@typeAlias::F
      aliasedType: dynamic Function(dynamic)
    notSimplyBounded G
      firstFragment: <testLibraryFragment>::@typeAlias::G
      aliasedType: dynamic Function(dynamic)
''');
  }

  test_class_notSimplyBounded_self() async {
    var library = await buildLibrary('''
class C<T extends C> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: C<dynamic>
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: C<dynamic>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_notSimplyBounded_simple_because_non_generic() async {
    // If no type parameters are specified, then the class is simply bounded, so
    // there is no reason to assign it a slot.
    var library = await buildLibrary('''
class C {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_notSimplyBounded_simple_by_lack_of_cycles() async {
    var library = await buildLibrary('''
class C<T extends D> {}
class D<T> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: D<dynamic>
              defaultType: D<dynamic>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
        class D @30
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @32
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @30
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          typeParameters
            T @32
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: D<dynamic>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      typeParameters
        T
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
''');
  }

  test_class_notSimplyBounded_simple_by_syntax() async {
    // If no bounds are specified, then the class is simply bounded by syntax
    // alone, so there is no reason to assign it a slot.
    var library = await buildLibrary('''
class C<T> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_operator() async {
    var library =
        await buildLibrary('class C { C operator+(C other) => null; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            + @20
              reference: <testLibraryFragment>::@class::C::@method::+
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional other @24
                  type: C
              returnType: C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            + @20
              reference: <testLibraryFragment>::@class::C::@method::+
              element: <testLibraryFragment>::@class::C::@method::+#element
              formalParameters
                other @24
                  element: <testLibraryFragment>::@class::C::@method::+::@parameter::other#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        +
          reference: <testLibrary>::@class::C::@method::+
          firstFragment: <testLibraryFragment>::@class::C::@method::+
          formalParameters
            requiredPositional other
              type: C
''');
  }

  test_class_operator_equal() async {
    var library = await buildLibrary('''
class C {
  bool operator==(Object other) => false;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            == @25
              reference: <testLibraryFragment>::@class::C::@method::==
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional other @35
                  type: Object
              returnType: bool
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            == @25
              reference: <testLibraryFragment>::@class::C::@method::==
              element: <testLibraryFragment>::@class::C::@method::==#element
              formalParameters
                other @35
                  element: <testLibraryFragment>::@class::C::@method::==::@parameter::other#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        ==
          reference: <testLibrary>::@class::C::@method::==
          firstFragment: <testLibraryFragment>::@class::C::@method::==
          formalParameters
            requiredPositional other
              type: Object
''');
  }

  test_class_operator_external() async {
    var library =
        await buildLibrary('class C { external C operator+(C other); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            external + @29
              reference: <testLibraryFragment>::@class::C::@method::+
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional other @33
                  type: C
              returnType: C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            + @29
              reference: <testLibraryFragment>::@class::C::@method::+
              element: <testLibraryFragment>::@class::C::@method::+#element
              formalParameters
                other @33
                  element: <testLibraryFragment>::@class::C::@method::+::@parameter::other#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        external +
          reference: <testLibrary>::@class::C::@method::+
          firstFragment: <testLibraryFragment>::@class::C::@method::+
          formalParameters
            requiredPositional other
              type: C
''');
  }

  test_class_operator_greater_equal() async {
    var library = await buildLibrary('''
class C {
  bool operator>=(C other) => false;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            >= @25
              reference: <testLibraryFragment>::@class::C::@method::>=
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional other @30
                  type: C
              returnType: bool
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            >= @25
              reference: <testLibraryFragment>::@class::C::@method::>=
              element: <testLibraryFragment>::@class::C::@method::>=#element
              formalParameters
                other @30
                  element: <testLibraryFragment>::@class::C::@method::>=::@parameter::other#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        >=
          reference: <testLibrary>::@class::C::@method::>=
          firstFragment: <testLibraryFragment>::@class::C::@method::>=
          formalParameters
            requiredPositional other
              type: C
''');
  }

  test_class_operator_index() async {
    var library =
        await buildLibrary('class C { bool operator[](int i) => null; }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            [] @23
              reference: <testLibraryFragment>::@class::C::@method::[]
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional i @30
                  type: int
              returnType: bool
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            [] @23
              reference: <testLibraryFragment>::@class::C::@method::[]
              element: <testLibraryFragment>::@class::C::@method::[]#element
              formalParameters
                i @30
                  element: <testLibraryFragment>::@class::C::@method::[]::@parameter::i#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        []
          reference: <testLibrary>::@class::C::@method::[]
          firstFragment: <testLibraryFragment>::@class::C::@method::[]
          formalParameters
            requiredPositional i
              type: int
''');
  }

  test_class_operator_index_set() async {
    var library = await buildLibrary('''
class C {
  void operator[]=(int i, bool v) {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            []= @25
              reference: <testLibraryFragment>::@class::C::@method::[]=
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional i @33
                  type: int
                requiredPositional v @41
                  type: bool
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            []= @25
              reference: <testLibraryFragment>::@class::C::@method::[]=
              element: <testLibraryFragment>::@class::C::@method::[]=#element
              formalParameters
                i @33
                  element: <testLibraryFragment>::@class::C::@method::[]=::@parameter::i#element
                v @41
                  element: <testLibraryFragment>::@class::C::@method::[]=::@parameter::v#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        []=
          reference: <testLibrary>::@class::C::@method::[]=
          firstFragment: <testLibraryFragment>::@class::C::@method::[]=
          formalParameters
            requiredPositional i
              type: int
            requiredPositional v
              type: bool
''');
  }

  test_class_operator_less_equal() async {
    var library = await buildLibrary('''
class C {
  bool operator<=(C other) => false;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          methods
            <= @25
              reference: <testLibraryFragment>::@class::C::@method::<=
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional other @30
                  type: C
              returnType: bool
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          methods
            <= @25
              reference: <testLibraryFragment>::@class::C::@method::<=
              element: <testLibraryFragment>::@class::C::@method::<=#element
              formalParameters
                other @30
                  element: <testLibraryFragment>::@class::C::@method::<=::@parameter::other#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      methods
        <=
          reference: <testLibrary>::@class::C::@method::<=
          firstFragment: <testLibraryFragment>::@class::C::@method::<=
          formalParameters
            requiredPositional other
              type: C
''');
  }

  test_class_operator_minus() async {
    var library = await buildLibrary('''
class A {
  int operator -(int other) => 0;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          methods
            - @25
              reference: <testLibraryFragment>::@class::A::@method::-
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional other @31
                  type: int
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          methods
            - @25
              reference: <testLibraryFragment>::@class::A::@method::-
              element: <testLibraryFragment>::@class::A::@method::-#element
              formalParameters
                other @31
                  element: <testLibraryFragment>::@class::A::@method::-::@parameter::other#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      methods
        -
          reference: <testLibrary>::@class::A::@method::-
          firstFragment: <testLibraryFragment>::@class::A::@method::-
          formalParameters
            requiredPositional other
              type: int
''');
  }

  test_class_operator_minus_unary() async {
    var library = await buildLibrary('''
class A {
  int operator -() => 0;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          methods
            unary- @25
              reference: <testLibraryFragment>::@class::A::@method::unary-
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          methods
            - @25
              reference: <testLibraryFragment>::@class::A::@method::unary-
              element: <testLibraryFragment>::@class::A::@method::unary-#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      methods
        -
          reference: <testLibrary>::@class::A::@method::unary-
          firstFragment: <testLibraryFragment>::@class::A::@method::unary-
''');
  }

  test_class_ref_nullability_none() async {
    var library = await buildLibrary('''
class C {}
C c;
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
      topLevelVariables
        static c @13
          reference: <testLibraryFragment>::@topLevelVariable::c
          enclosingElement3: <testLibraryFragment>
          type: C
      accessors
        synthetic static get c @-1
          reference: <testLibraryFragment>::@getter::c
          enclosingElement3: <testLibraryFragment>
          returnType: C
        synthetic static set c= @-1
          reference: <testLibraryFragment>::@setter::c
          enclosingElement3: <testLibraryFragment>
          parameters
            requiredPositional _c @-1
              type: C
          returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
      topLevelVariables
        c @13
          reference: <testLibraryFragment>::@topLevelVariable::c
          element: <testLibrary>::@topLevelVariable::c
          getter2: <testLibraryFragment>::@getter::c
          setter2: <testLibraryFragment>::@setter::c
      getters
        synthetic get c
          reference: <testLibraryFragment>::@getter::c
          element: <testLibraryFragment>::@getter::c#element
      setters
        synthetic set c
          reference: <testLibraryFragment>::@setter::c
          element: <testLibraryFragment>::@setter::c#element
          formalParameters
            <null-name>
              element: <testLibraryFragment>::@setter::c::@parameter::_c#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
  topLevelVariables
    c
      reference: <testLibrary>::@topLevelVariable::c
      firstFragment: <testLibraryFragment>::@topLevelVariable::c
      type: C
      getter: <testLibraryFragment>::@getter::c#element
      setter: <testLibraryFragment>::@setter::c#element
  getters
    synthetic static get c
      firstFragment: <testLibraryFragment>::@getter::c
  setters
    synthetic static set c
      firstFragment: <testLibraryFragment>::@setter::c
      formalParameters
        requiredPositional _c
          type: C
''');
  }

  test_class_ref_nullability_question() async {
    var library = await buildLibrary('''
class C {}
C? c;
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
      topLevelVariables
        static c @14
          reference: <testLibraryFragment>::@topLevelVariable::c
          enclosingElement3: <testLibraryFragment>
          type: C?
      accessors
        synthetic static get c @-1
          reference: <testLibraryFragment>::@getter::c
          enclosingElement3: <testLibraryFragment>
          returnType: C?
        synthetic static set c= @-1
          reference: <testLibraryFragment>::@setter::c
          enclosingElement3: <testLibraryFragment>
          parameters
            requiredPositional _c @-1
              type: C?
          returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
      topLevelVariables
        c @14
          reference: <testLibraryFragment>::@topLevelVariable::c
          element: <testLibrary>::@topLevelVariable::c
          getter2: <testLibraryFragment>::@getter::c
          setter2: <testLibraryFragment>::@setter::c
      getters
        synthetic get c
          reference: <testLibraryFragment>::@getter::c
          element: <testLibraryFragment>::@getter::c#element
      setters
        synthetic set c
          reference: <testLibraryFragment>::@setter::c
          element: <testLibraryFragment>::@setter::c#element
          formalParameters
            <null-name>
              element: <testLibraryFragment>::@setter::c::@parameter::_c#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
  topLevelVariables
    c
      reference: <testLibrary>::@topLevelVariable::c
      firstFragment: <testLibraryFragment>::@topLevelVariable::c
      type: C?
      getter: <testLibraryFragment>::@getter::c#element
      setter: <testLibraryFragment>::@setter::c#element
  getters
    synthetic static get c
      firstFragment: <testLibraryFragment>::@getter::c
  setters
    synthetic static set c
      firstFragment: <testLibraryFragment>::@setter::c
      formalParameters
        requiredPositional _c
          type: C?
''');
  }

  test_class_sealed() async {
    var library = await buildLibrary('sealed class C {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        abstract sealed class C @13
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @13
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    abstract sealed class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_sealed_induced_base_extends_base() async {
    var library = await buildLibrary('''
base class A {}
sealed class B extends A {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        base class A @11
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
        abstract sealed base class B @29
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @11
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @29
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    base class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    abstract sealed base class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_sealed_induced_base_implements_base() async {
    var library = await buildLibrary('''
base class A {}
sealed class B implements A {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        base class A @11
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
        abstract sealed base class B @29
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          interfaces
            A
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @11
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @29
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
  classes
    base class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    abstract sealed base class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      interfaces
        A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
''');
  }

  test_class_sealed_induced_base_implements_final() async {
    var library = await buildLibrary('''
final class A {}
sealed class B implements A {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        final class A @12
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
        abstract sealed base class B @30
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          interfaces
            A
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @12
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @30
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
  classes
    final class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    abstract sealed base class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      interfaces
        A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
''');
  }

  test_class_sealed_induced_final_extends_final() async {
    var library = await buildLibrary('''
final class A {}
sealed class B extends A {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        final class A @12
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
        abstract sealed final class B @30
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @12
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @30
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    final class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    abstract sealed final class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_sealed_induced_final_with_base_mixin() async {
    var library = await buildLibrary('''
base mixin A {}
interface class B {}
sealed class C extends B with A {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        interface class B @32
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
        abstract sealed final class C @50
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: B
          mixins
            A
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              superConstructor: <testLibraryFragment>::@class::B::@constructor::new
      mixins
        base mixin A @11
          reference: <testLibraryFragment>::@mixin::A
          enclosingElement3: <testLibraryFragment>
          superclassConstraints
            Object
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class B @32
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
        class C @50
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              superConstructor: <testLibraryFragment>::@class::B::@constructor::new
      mixins
        mixin A @11
          reference: <testLibraryFragment>::@mixin::A
          element: <testLibrary>::@mixin::A
  classes
    interface class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
    abstract sealed final class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: B
      mixins
        A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::B::@constructor::new#element
  mixins
    base mixin A
      reference: <testLibrary>::@mixin::A
      firstFragment: <testLibraryFragment>::@mixin::A
      superclassConstraints
        Object
''');
  }

  test_class_sealed_induced_interface_extends_interface() async {
    var library = await buildLibrary('''
interface class A {}
sealed class B extends A {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        interface class A @16
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
        abstract sealed interface class B @34
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @16
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @34
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    interface class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    abstract sealed interface class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_sealed_induced_none_implements_interface() async {
    var library = await buildLibrary('''
interface class A {}
sealed class B implements A {}''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        interface class A @16
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
        abstract sealed class B @34
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          interfaces
            A
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @16
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @34
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
  classes
    interface class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    abstract sealed class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      interfaces
        A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
''');
  }

  test_class_setter_abstract() async {
    var library =
        await buildLibrary('abstract class C { void set x(int value); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        abstract class C @15
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            abstract set x= @28
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional value @34
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @15
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @28
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @34
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
  classes
    abstract class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        abstract set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional value
              type: int
''');
  }

  test_class_setter_covariant() async {
    var library =
        await buildLibrary('class C { void set x(covariant int value); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            abstract set x= @19
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional covariant value @35
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @19
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @35
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        abstract set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional covariant value
              type: int
''');
  }

  test_class_setter_external() async {
    var library =
        await buildLibrary('class C { external void set x(int value); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            external set x= @28
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional value @34
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @28
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @34
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        external set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional value
              type: int
''');
  }

  test_class_setter_implicit_param_type() async {
    var library = await buildLibrary('class C { void set x(value) {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            set x= @19
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional hasImplicitType value @21
                  type: dynamic
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @19
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @21
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional hasImplicitType value
              type: dynamic
''');
  }

  test_class_setter_implicit_return_type() async {
    var library = await buildLibrary('class C { set x(int value) {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            set x= @14
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional value @20
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @14
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @20
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional value
              type: int
''');
  }

  test_class_setter_inferred_type_conflictingInheritance() async {
    var library = await buildLibrary('''
class A {
  int t;
}
class B extends A {
  double t;
}
class C extends A implements B {
}
class D extends C {
  void set t(p) {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            t @16
              reference: <testLibraryFragment>::@class::A::@field::t
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            synthetic get t @-1
              reference: <testLibraryFragment>::@class::A::@getter::t
              enclosingElement3: <testLibraryFragment>::@class::A
              returnType: int
            synthetic set t= @-1
              reference: <testLibraryFragment>::@class::A::@setter::t
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional _t @-1
                  type: int
              returnType: void
        class B @27
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
          fields
            t @50
              reference: <testLibraryFragment>::@class::B::@field::t
              enclosingElement3: <testLibraryFragment>::@class::B
              type: double
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
          accessors
            synthetic get t @-1
              reference: <testLibraryFragment>::@class::B::@getter::t
              enclosingElement3: <testLibraryFragment>::@class::B
              returnType: double
            synthetic set t= @-1
              reference: <testLibraryFragment>::@class::B::@setter::t
              enclosingElement3: <testLibraryFragment>::@class::B
              parameters
                requiredPositional _t @-1
                  type: double
              returnType: void
        class C @61
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: A
          interfaces
            B
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
        class D @96
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          supertype: C
          fields
            synthetic t @-1
              reference: <testLibraryFragment>::@class::D::@field::t
              enclosingElement3: <testLibraryFragment>::@class::D
              type: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
              superConstructor: <testLibraryFragment>::@class::C::@constructor::new
          accessors
            set t= @121
              reference: <testLibraryFragment>::@class::D::@setter::t
              enclosingElement3: <testLibraryFragment>::@class::D
              parameters
                requiredPositional hasImplicitType p @123
                  type: dynamic
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            t @16
              reference: <testLibraryFragment>::@class::A::@field::t
              element: <testLibraryFragment>::@class::A::@field::t#element
              getter2: <testLibraryFragment>::@class::A::@getter::t
              setter2: <testLibraryFragment>::@class::A::@setter::t
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          getters
            synthetic get t
              reference: <testLibraryFragment>::@class::A::@getter::t
              element: <testLibraryFragment>::@class::A::@getter::t#element
          setters
            synthetic set t
              reference: <testLibraryFragment>::@class::A::@setter::t
              element: <testLibraryFragment>::@class::A::@setter::t#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::A::@setter::t::@parameter::_t#element
        class B @27
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          fields
            t @50
              reference: <testLibraryFragment>::@class::B::@field::t
              element: <testLibraryFragment>::@class::B::@field::t#element
              getter2: <testLibraryFragment>::@class::B::@getter::t
              setter2: <testLibraryFragment>::@class::B::@setter::t
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
          getters
            synthetic get t
              reference: <testLibraryFragment>::@class::B::@getter::t
              element: <testLibraryFragment>::@class::B::@getter::t#element
          setters
            synthetic set t
              reference: <testLibraryFragment>::@class::B::@setter::t
              element: <testLibraryFragment>::@class::B::@setter::t#element
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::B::@setter::t::@parameter::_t#element
        class C @61
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
        class D @96
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          fields
            synthetic t
              reference: <testLibraryFragment>::@class::D::@field::t
              element: <testLibraryFragment>::@class::D::@field::t#element
              setter2: <testLibraryFragment>::@class::D::@setter::t
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
              superConstructor: <testLibraryFragment>::@class::C::@constructor::new
          setters
            set t @121
              reference: <testLibraryFragment>::@class::D::@setter::t
              element: <testLibraryFragment>::@class::D::@setter::t#element
              formalParameters
                p @123
                  element: <testLibraryFragment>::@class::D::@setter::t::@parameter::p#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        t
          firstFragment: <testLibraryFragment>::@class::A::@field::t
          type: int
          getter: <testLibraryFragment>::@class::A::@getter::t#element
          setter: <testLibraryFragment>::@class::A::@setter::t#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      getters
        synthetic get t
          firstFragment: <testLibraryFragment>::@class::A::@getter::t
      setters
        synthetic set t
          firstFragment: <testLibraryFragment>::@class::A::@setter::t
          formalParameters
            requiredPositional _t
              type: int
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
      fields
        t
          firstFragment: <testLibraryFragment>::@class::B::@field::t
          type: double
          getter: <testLibraryFragment>::@class::B::@getter::t#element
          setter: <testLibraryFragment>::@class::B::@setter::t#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
      getters
        synthetic get t
          firstFragment: <testLibraryFragment>::@class::B::@getter::t
      setters
        synthetic set t
          firstFragment: <testLibraryFragment>::@class::B::@setter::t
          formalParameters
            requiredPositional _t
              type: double
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: A
      interfaces
        B
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      supertype: C
      fields
        synthetic t
          firstFragment: <testLibraryFragment>::@class::D::@field::t
          type: dynamic
          setter: <testLibraryFragment>::@class::D::@setter::t#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
          superConstructor: <testLibraryFragment>::@class::C::@constructor::new#element
      setters
        set t
          firstFragment: <testLibraryFragment>::@class::D::@setter::t
          formalParameters
            requiredPositional hasImplicitType p
              type: dynamic
''');
  }

  test_class_setter_inferred_type_nonStatic_implicit_param() async {
    var library =
        await buildLibrary('class C extends D { void set f(value) {} }'
            ' abstract class D { void set f(int value); }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: D
          fields
            synthetic f @-1
              reference: <testLibraryFragment>::@class::C::@field::f
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              superConstructor: <testLibraryFragment>::@class::D::@constructor::new
          accessors
            set f= @29
              reference: <testLibraryFragment>::@class::C::@setter::f
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional hasImplicitType value @31
                  type: int
              returnType: void
        abstract class D @58
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic f @-1
              reference: <testLibraryFragment>::@class::D::@field::f
              enclosingElement3: <testLibraryFragment>::@class::D
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
          accessors
            abstract set f= @71
              reference: <testLibraryFragment>::@class::D::@setter::f
              enclosingElement3: <testLibraryFragment>::@class::D
              parameters
                requiredPositional value @77
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic f
              reference: <testLibraryFragment>::@class::C::@field::f
              element: <testLibraryFragment>::@class::C::@field::f#element
              setter2: <testLibraryFragment>::@class::C::@setter::f
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              superConstructor: <testLibraryFragment>::@class::D::@constructor::new
          setters
            set f @29
              reference: <testLibraryFragment>::@class::C::@setter::f
              element: <testLibraryFragment>::@class::C::@setter::f#element
              formalParameters
                value @31
                  element: <testLibraryFragment>::@class::C::@setter::f::@parameter::value#element
        class D @58
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          fields
            synthetic f
              reference: <testLibraryFragment>::@class::D::@field::f
              element: <testLibraryFragment>::@class::D::@field::f#element
              setter2: <testLibraryFragment>::@class::D::@setter::f
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
          setters
            set f @71
              reference: <testLibraryFragment>::@class::D::@setter::f
              element: <testLibraryFragment>::@class::D::@setter::f#element
              formalParameters
                value @77
                  element: <testLibraryFragment>::@class::D::@setter::f::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D
      fields
        synthetic f
          firstFragment: <testLibraryFragment>::@class::C::@field::f
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::f#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
      setters
        set f
          firstFragment: <testLibraryFragment>::@class::C::@setter::f
          formalParameters
            requiredPositional hasImplicitType value
              type: int
    abstract class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      fields
        synthetic f
          firstFragment: <testLibraryFragment>::@class::D::@field::f
          type: int
          setter: <testLibraryFragment>::@class::D::@setter::f#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
      setters
        abstract set f
          firstFragment: <testLibraryFragment>::@class::D::@setter::f
          formalParameters
            requiredPositional value
              type: int
''');
  }

  test_class_setter_inferred_type_static_implicit_return() async {
    var library = await buildLibrary('''
class C {
  static set f(int value) {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic static f @-1
              reference: <testLibraryFragment>::@class::C::@field::f
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            static set f= @23
              reference: <testLibraryFragment>::@class::C::@setter::f
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional value @29
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic f
              reference: <testLibraryFragment>::@class::C::@field::f
              element: <testLibraryFragment>::@class::C::@field::f#element
              setter2: <testLibraryFragment>::@class::C::@setter::f
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set f @23
              reference: <testLibraryFragment>::@class::C::@setter::f
              element: <testLibraryFragment>::@class::C::@setter::f#element
              formalParameters
                value @29
                  element: <testLibraryFragment>::@class::C::@setter::f::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic static f
          firstFragment: <testLibraryFragment>::@class::C::@field::f
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::f#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        static set f
          firstFragment: <testLibraryFragment>::@class::C::@setter::f
          formalParameters
            requiredPositional value
              type: int
''');
  }

  test_class_setter_invalid_named_parameter() async {
    var library = await buildLibrary('class C { void set x({a}) {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            set x= @19
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                optionalNamed default hasImplicitType a @22
                  reference: <testLibraryFragment>::@class::C::@setter::x::@parameter::a
                  type: dynamic
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @19
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                default a @22
                  reference: <testLibraryFragment>::@class::C::@setter::x::@parameter::a
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::a#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            optionalNamed hasImplicitType a
              firstFragment: <testLibraryFragment>::@class::C::@setter::x::@parameter::a
              type: dynamic
''');
  }

  test_class_setter_invalid_no_parameter() async {
    var library = await buildLibrary('class C { void set x() {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            set x= @19
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @19
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
''');
  }

  test_class_setter_invalid_optional_parameter() async {
    var library = await buildLibrary('class C { void set x([a]) {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            set x= @19
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                optionalPositional default hasImplicitType a @22
                  type: dynamic
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @19
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                default a @22
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::a#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            optionalPositional hasImplicitType a
              type: dynamic
''');
  }

  test_class_setter_invalid_too_many_parameters() async {
    var library = await buildLibrary('class C { void set x(a, b) {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            set x= @19
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional hasImplicitType a @21
                  type: dynamic
                requiredPositional hasImplicitType b @24
                  type: dynamic
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @19
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                a @21
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::a#element
                b @24
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::b#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: dynamic
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional hasImplicitType a
              type: dynamic
            requiredPositional hasImplicitType b
              type: dynamic
''');
  }

  test_class_setter_invokesSuperSelf_getter() async {
    var library = await buildLibrary(r'''
class A {
  set foo(int _) {
    super.foo;
  }
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic foo @-1
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            set foo= @16
              reference: <testLibraryFragment>::@class::A::@setter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional _ @24
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              setter2: <testLibraryFragment>::@class::A::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          setters
            set foo @16
              reference: <testLibraryFragment>::@class::A::@setter::foo
              element: <testLibraryFragment>::@class::A::@setter::foo#element
              formalParameters
                _ @24
                  element: <testLibraryFragment>::@class::A::@setter::foo::@parameter::_#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          setter: <testLibraryFragment>::@class::A::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      setters
        set foo
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo
          formalParameters
            requiredPositional _
              type: int
''');
  }

  test_class_setter_invokesSuperSelf_setter() async {
    var library = await buildLibrary(r'''
class A {
  set foo(int _) {
    super.foo = 0;
  }
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic foo @-1
              reference: <testLibraryFragment>::@class::A::@field::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          accessors
            set foo= @16 invokesSuperSelf
              reference: <testLibraryFragment>::@class::A::@setter::foo
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional _ @24
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          fields
            synthetic foo
              reference: <testLibraryFragment>::@class::A::@field::foo
              element: <testLibraryFragment>::@class::A::@field::foo#element
              setter2: <testLibraryFragment>::@class::A::@setter::foo
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          setters
            set foo @16
              reference: <testLibraryFragment>::@class::A::@setter::foo
              element: <testLibraryFragment>::@class::A::@setter::foo#element
              formalParameters
                _ @24
                  element: <testLibraryFragment>::@class::A::@setter::foo::@parameter::_#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      fields
        synthetic foo
          firstFragment: <testLibraryFragment>::@class::A::@field::foo
          type: int
          setter: <testLibraryFragment>::@class::A::@setter::foo#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      setters
        set foo
          firstFragment: <testLibraryFragment>::@class::A::@setter::foo
          formalParameters
            requiredPositional _
              type: int
''');
  }

  test_class_setter_missingName() async {
    var library = await buildLibrary('''
class A {
  set (int _) {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
          methods
            set @12
              reference: <testLibraryFragment>::@class::A::@method::set
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional _ @21
                  type: int
              returnType: dynamic
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
          methods
            set @12
              reference: <testLibraryFragment>::@class::A::@method::set
              element: <testLibraryFragment>::@class::A::@method::set#element
              formalParameters
                _ @21
                  element: <testLibraryFragment>::@class::A::@method::set::@parameter::_#element
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
      methods
        set
          reference: <testLibrary>::@class::A::@method::set
          firstFragment: <testLibraryFragment>::@class::A::@method::set
          formalParameters
            requiredPositional _
              type: int
''');
  }

  test_class_setter_native() async {
    var library = await buildLibrary('''
class C {
  void set x(int value) native;
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            external set x= @21
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional value @27
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @21
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @27
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        external set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional value
              type: int
''');
  }

  test_class_setter_static() async {
    var library =
        await buildLibrary('class C { static void set x(int value) {} }');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic static x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            static set x= @26
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional value @32
                  type: int
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @26
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @32
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic static x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::x#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        static set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional value
              type: int
''');
  }

  test_class_setters() async {
    var library = await buildLibrary('''
class C {
  void set x(int value) {}
  set y(value) {}
}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          fields
            synthetic x @-1
              reference: <testLibraryFragment>::@class::C::@field::x
              enclosingElement3: <testLibraryFragment>::@class::C
              type: int
            synthetic y @-1
              reference: <testLibraryFragment>::@class::C::@field::y
              enclosingElement3: <testLibraryFragment>::@class::C
              type: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
          accessors
            set x= @21
              reference: <testLibraryFragment>::@class::C::@setter::x
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional value @27
                  type: int
              returnType: void
            set y= @43
              reference: <testLibraryFragment>::@class::C::@setter::y
              enclosingElement3: <testLibraryFragment>::@class::C
              parameters
                requiredPositional hasImplicitType value @45
                  type: dynamic
              returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          fields
            synthetic x
              reference: <testLibraryFragment>::@class::C::@field::x
              element: <testLibraryFragment>::@class::C::@field::x#element
              setter2: <testLibraryFragment>::@class::C::@setter::x
            synthetic y
              reference: <testLibraryFragment>::@class::C::@field::y
              element: <testLibraryFragment>::@class::C::@field::y#element
              setter2: <testLibraryFragment>::@class::C::@setter::y
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
          setters
            set x @21
              reference: <testLibraryFragment>::@class::C::@setter::x
              element: <testLibraryFragment>::@class::C::@setter::x#element
              formalParameters
                value @27
                  element: <testLibraryFragment>::@class::C::@setter::x::@parameter::value#element
            set y @43
              reference: <testLibraryFragment>::@class::C::@setter::y
              element: <testLibraryFragment>::@class::C::@setter::y#element
              formalParameters
                value @45
                  element: <testLibraryFragment>::@class::C::@setter::y::@parameter::value#element
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      fields
        synthetic x
          firstFragment: <testLibraryFragment>::@class::C::@field::x
          type: int
          setter: <testLibraryFragment>::@class::C::@setter::x#element
        synthetic y
          firstFragment: <testLibraryFragment>::@class::C::@field::y
          type: dynamic
          setter: <testLibraryFragment>::@class::C::@setter::y#element
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
      setters
        set x
          firstFragment: <testLibraryFragment>::@class::C::@setter::x
          formalParameters
            requiredPositional value
              type: int
        set y
          firstFragment: <testLibraryFragment>::@class::C::@setter::y
          formalParameters
            requiredPositional hasImplicitType value
              type: dynamic
''');
  }

  test_class_supertype() async {
    var library = await buildLibrary('''
class A {}
class B extends A {}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
        class B @17
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
        class B @17
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A
''');
  }

  test_class_supertype_dynamic() async {
    var library = await buildLibrary('''
class A extends dynamic {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
''');
  }

  test_class_supertype_extensionType() async {
    var library = await buildLibrary('''
extension type A(int it) {}
class B extends A {}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class B @34
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
      extensionTypes
        A @15
          reference: <testLibraryFragment>::@extensionType::A
          enclosingElement3: <testLibraryFragment>
          representation: <testLibraryFragment>::@extensionType::A::@field::it
          primaryConstructor: <testLibraryFragment>::@extensionType::A::@constructor::new
          typeErasure: int
          fields
            final it @21
              reference: <testLibraryFragment>::@extensionType::A::@field::it
              enclosingElement3: <testLibraryFragment>::@extensionType::A
              type: int
          accessors
            synthetic get it @-1
              reference: <testLibraryFragment>::@extensionType::A::@getter::it
              enclosingElement3: <testLibraryFragment>::@extensionType::A
              returnType: int
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class B @34
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
      extensionTypes
        extension type A @15
          reference: <testLibraryFragment>::@extensionType::A
          element: <testLibrary>::@extensionType::A
          fields
            it @21
              reference: <testLibraryFragment>::@extensionType::A::@field::it
              element: <testLibraryFragment>::@extensionType::A::@field::it#element
              getter2: <testLibraryFragment>::@extensionType::A::@getter::it
          getters
            synthetic get it
              reference: <testLibraryFragment>::@extensionType::A::@getter::it
              element: <testLibraryFragment>::@extensionType::A::@getter::it#element
  classes
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
  extensionTypes
    extension type A
      reference: <testLibrary>::@extensionType::A
      firstFragment: <testLibraryFragment>::@extensionType::A
      representation: <testLibraryFragment>::@extensionType::A::@field::it#element
      primaryConstructor: <testLibraryFragment>::@extensionType::A::@constructor::new#element
      typeErasure: int
      fields
        final it
          firstFragment: <testLibraryFragment>::@extensionType::A::@field::it
          type: int
          getter: <testLibraryFragment>::@extensionType::A::@getter::it#element
      getters
        synthetic get it
          firstFragment: <testLibraryFragment>::@extensionType::A::@getter::it
''');
  }

  test_class_supertype_genericClass() async {
    var library = await buildLibrary('''
class C extends D<int, double> {}
class D<T1, T2> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: D<int, double>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              superConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::D::@constructor::new
                substitution: {T1: int, T2: double}
        class D @40
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T1 @42
              defaultType: dynamic
            covariant T2 @46
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              superConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::D::@constructor::new
                substitution: {T1: int, T2: double}
        class D @40
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          typeParameters
            T1 @42
              element: <not-implemented>
            T2 @46
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D<int, double>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      typeParameters
        T1
        T2
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
''');
  }

  test_class_supertype_genericClass_tooManyArguments() async {
    var library = await buildLibrary('''
class A<T> {}
class B extends A<int, String> {}
''');
    configuration.withConstructors = false;
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              defaultType: dynamic
        class B @20
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A<dynamic>
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @8
              element: <not-implemented>
        class B @20
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A<dynamic>
''');
  }

  test_class_supertype_typeArguments_self() async {
    var library = await buildLibrary('''
class A<T> {}
class B extends A<B> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
        class B @20
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          supertype: A<B>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
              superConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::A::@constructor::new
                substitution: {T: B}
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class B @20
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
              superConstructor: ConstructorMember
                base: <testLibraryFragment>::@class::A::@constructor::new
                substitution: {T: B}
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      supertype: A<B>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_class_supertype_typeParameter() async {
    var library = await buildLibrary('''
class A<T> extends T<int> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
''');
  }

  test_class_supertype_unresolved() async {
    var library = await buildLibrary('class C extends D {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters() async {
    var library = await buildLibrary('class C<T, U> {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              defaultType: dynamic
            covariant U @11
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
            U @11
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters_bound() async {
    var library = await buildLibrary('''
class C<T extends Object, U extends D> {}
class D {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: Object
              defaultType: Object
            covariant U @26
              bound: D
              defaultType: D
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
        class D @48
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
            U @26
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
        class D @48
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: Object
        U
          bound: D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
''');
  }

  test_class_typeParameters_cycle_1of1() async {
    var library = await buildLibrary('class C<T extends T> {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: dynamic
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: dynamic
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters_cycle_2of3() async {
    var library = await buildLibrary(r'''
class C<T extends V, U, V extends T> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: dynamic
              defaultType: dynamic
            covariant U @21
              defaultType: dynamic
            covariant V @24
              bound: dynamic
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
            U @21
              element: <not-implemented>
            V @24
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: dynamic
        U
        V
          bound: dynamic
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters_defaultType_cycle_genericFunctionType() async {
    var library = await buildLibrary(r'''
class A<T extends void Function(A)> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: void Function(A<dynamic>)
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
          bound: void Function(A<dynamic>)
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
''');
  }

  test_class_typeParameters_defaultType_cycle_genericFunctionType2() async {
    var library = await buildLibrary(r'''
class C<T extends void Function<U extends C>()> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: void Function<U extends C<dynamic>>()
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: void Function<U extends C<dynamic>>()
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters_defaultType_functionTypeAlias_contravariant() async {
    var library = await buildLibrary(r'''
typedef F<X> = void Function(X);

class A<X extends F<X>> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class A @40
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant X @42
              bound: void Function(X)
                alias: <testLibraryFragment>::@typeAlias::F
                  typeArguments
                    X
              defaultType: void Function(Never)
                alias: <testLibraryFragment>::@typeAlias::F
                  typeArguments
                    Never
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
      typeAliases
        F @8
          reference: <testLibraryFragment>::@typeAlias::F
          typeParameters
            contravariant X @10
              defaultType: dynamic
          aliasedType: void Function(X)
          aliasedElement: GenericFunctionTypeElement
            parameters
              requiredPositional @-1
                type: X
            returnType: void
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @40
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            X @42
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
      typeAliases
        F @8
          reference: <testLibraryFragment>::@typeAlias::F
          element: <testLibrary>::@typeAlias::F
          typeParameters
            X @10
              element: <not-implemented>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        X
          bound: void Function(X)
            alias: <testLibraryFragment>::@typeAlias::F
              typeArguments
                X
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
  typeAliases
    F
      firstFragment: <testLibraryFragment>::@typeAlias::F
      typeParameters
        X
      aliasedType: void Function(X)
''');
  }

  test_class_typeParameters_defaultType_functionTypeAlias_covariant() async {
    var library = await buildLibrary(r'''
typedef F<X> = X Function();

class A<X extends F<X>> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class A @36
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant X @38
              bound: X Function()
                alias: <testLibraryFragment>::@typeAlias::F
                  typeArguments
                    X
              defaultType: dynamic Function()
                alias: <testLibraryFragment>::@typeAlias::F
                  typeArguments
                    dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
      typeAliases
        F @8
          reference: <testLibraryFragment>::@typeAlias::F
          typeParameters
            covariant X @10
              defaultType: dynamic
          aliasedType: X Function()
          aliasedElement: GenericFunctionTypeElement
            returnType: X
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @36
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            X @38
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
      typeAliases
        F @8
          reference: <testLibraryFragment>::@typeAlias::F
          element: <testLibrary>::@typeAlias::F
          typeParameters
            X @10
              element: <not-implemented>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        X
          bound: X Function()
            alias: <testLibraryFragment>::@typeAlias::F
              typeArguments
                X
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
  typeAliases
    F
      firstFragment: <testLibraryFragment>::@typeAlias::F
      typeParameters
        X
      aliasedType: X Function()
''');
  }

  test_class_typeParameters_defaultType_functionTypeAlias_invariant() async {
    var library = await buildLibrary(r'''
typedef F<X> = X Function(X);

class A<X extends F<X>> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class A @37
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant X @39
              bound: X Function(X)
                alias: <testLibraryFragment>::@typeAlias::F
                  typeArguments
                    X
              defaultType: dynamic Function(dynamic)
                alias: <testLibraryFragment>::@typeAlias::F
                  typeArguments
                    dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
      typeAliases
        F @8
          reference: <testLibraryFragment>::@typeAlias::F
          typeParameters
            invariant X @10
              defaultType: dynamic
          aliasedType: X Function(X)
          aliasedElement: GenericFunctionTypeElement
            parameters
              requiredPositional @-1
                type: X
            returnType: X
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @37
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            X @39
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
      typeAliases
        F @8
          reference: <testLibraryFragment>::@typeAlias::F
          element: <testLibrary>::@typeAlias::F
          typeParameters
            X @10
              element: <not-implemented>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        X
          bound: X Function(X)
            alias: <testLibraryFragment>::@typeAlias::F
              typeArguments
                X
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
  typeAliases
    F
      firstFragment: <testLibraryFragment>::@typeAlias::F
      typeParameters
        X
      aliasedType: X Function(X)
''');
  }

  test_class_typeParameters_defaultType_functionTypeAlias_invariant_legacy() async {
    var library = await buildLibrary(r'''
typedef F<X> = X Function(X);

class A<X extends F<X>> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class A @37
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant X @39
              bound: X Function(X)
                alias: <testLibraryFragment>::@typeAlias::F
                  typeArguments
                    X
              defaultType: dynamic Function(dynamic)
                alias: <testLibraryFragment>::@typeAlias::F
                  typeArguments
                    dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
      typeAliases
        F @8
          reference: <testLibraryFragment>::@typeAlias::F
          typeParameters
            invariant X @10
              defaultType: dynamic
          aliasedType: X Function(X)
          aliasedElement: GenericFunctionTypeElement
            parameters
              requiredPositional @-1
                type: X
            returnType: X
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @37
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            X @39
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
      typeAliases
        F @8
          reference: <testLibraryFragment>::@typeAlias::F
          element: <testLibrary>::@typeAlias::F
          typeParameters
            X @10
              element: <not-implemented>
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        X
          bound: X Function(X)
            alias: <testLibraryFragment>::@typeAlias::F
              typeArguments
                X
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
  typeAliases
    F
      firstFragment: <testLibraryFragment>::@typeAlias::F
      typeParameters
        X
      aliasedType: X Function(X)
''');
  }

  test_class_typeParameters_defaultType_genericFunctionType_both() async {
    var library = await buildLibrary(r'''
class A<X extends X Function(X)> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant X @8
              bound: X Function(X)
              defaultType: dynamic Function(Never)
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            X @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        X
          bound: X Function(X)
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
''');
  }

  test_class_typeParameters_defaultType_genericFunctionType_contravariant() async {
    var library = await buildLibrary(r'''
class A<X extends void Function(X)> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant X @8
              bound: void Function(X)
              defaultType: void Function(Never)
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            X @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        X
          bound: void Function(X)
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
''');
  }

  test_class_typeParameters_defaultType_genericFunctionType_covariant() async {
    var library = await buildLibrary(r'''
class A<X extends X Function()> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant X @8
              bound: X Function()
              defaultType: dynamic Function()
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            X @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        X
          bound: X Function()
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
''');
  }

  test_class_typeParameters_defaultType_genericFunctionType_covariant_legacy() async {
    var library = await buildLibrary(r'''
class A<X extends X Function()> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant X @8
              bound: X Function()
              defaultType: dynamic Function()
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            X @8
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        X
          bound: X Function()
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
''');
  }

  test_class_typeParameters_defaultType_typeAlias_interface_contravariant() async {
    var library = await buildLibrary(r'''
typedef A<X> = List<void Function(X)>;

class B<X extends A<X>> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class B @46
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant X @48
              bound: List<void Function(X)>
                alias: <testLibraryFragment>::@typeAlias::A
                  typeArguments
                    X
              defaultType: List<void Function(Never)>
                alias: <testLibraryFragment>::@typeAlias::A
                  typeArguments
                    Never
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
      typeAliases
        A @8
          reference: <testLibraryFragment>::@typeAlias::A
          typeParameters
            contravariant X @10
              defaultType: dynamic
          aliasedType: List<void Function(X)>
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class B @46
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          typeParameters
            X @48
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
      typeAliases
        A @8
          reference: <testLibraryFragment>::@typeAlias::A
          element: <testLibrary>::@typeAlias::A
          typeParameters
            X @10
              element: <not-implemented>
  classes
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      typeParameters
        X
          bound: List<void Function(X)>
            alias: <testLibraryFragment>::@typeAlias::A
              typeArguments
                X
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
  typeAliases
    A
      firstFragment: <testLibraryFragment>::@typeAlias::A
      typeParameters
        X
      aliasedType: List<void Function(X)>
''');
  }

  test_class_typeParameters_defaultType_typeAlias_interface_covariant() async {
    var library = await buildLibrary(r'''
typedef A<X> = Map<X, int>;

class B<X extends A<X>> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class B @35
          reference: <testLibraryFragment>::@class::B
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant X @37
              bound: Map<X, int>
                alias: <testLibraryFragment>::@typeAlias::A
                  typeArguments
                    X
              defaultType: Map<dynamic, int>
                alias: <testLibraryFragment>::@typeAlias::A
                  typeArguments
                    dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::B::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::B
      typeAliases
        A @8
          reference: <testLibraryFragment>::@typeAlias::A
          typeParameters
            covariant X @10
              defaultType: dynamic
          aliasedType: Map<X, int>
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class B @35
          reference: <testLibraryFragment>::@class::B
          element: <testLibrary>::@class::B
          typeParameters
            X @37
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::B::@constructor::new
              element: <testLibraryFragment>::@class::B::@constructor::new#element
              typeName: B
      typeAliases
        A @8
          reference: <testLibraryFragment>::@typeAlias::A
          element: <testLibrary>::@typeAlias::A
          typeParameters
            X @10
              element: <not-implemented>
  classes
    class B
      reference: <testLibrary>::@class::B
      firstFragment: <testLibraryFragment>::@class::B
      typeParameters
        X
          bound: Map<X, int>
            alias: <testLibraryFragment>::@typeAlias::A
              typeArguments
                X
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::B::@constructor::new
  typeAliases
    A
      firstFragment: <testLibraryFragment>::@typeAlias::A
      typeParameters
        X
      aliasedType: Map<X, int>
''');
  }

  test_class_typeParameters_f_bound_complex() async {
    var library = await buildLibrary('class C<T extends List<U>, U> {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: List<U>
              defaultType: List<dynamic>
            covariant U @27
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
            U @27
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: List<U>
        U
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters_f_bound_simple() async {
    var library = await buildLibrary('class C<T extends U, U> {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        notSimplyBounded class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              bound: U
              defaultType: dynamic
            covariant U @21
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @8
              element: <not-implemented>
            U @21
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
          bound: U
        U
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters_missingName() async {
    var library = await buildLibrary(r'''
class A<T,> {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @8
              defaultType: dynamic
            covariant @10
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          typeParameters
            T @8
              element: <not-implemented>
            <null-name>
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      typeParameters
        T
        <null-name>
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
''');
  }

  test_class_typeParameters_variance_contravariant() async {
    var library = await buildLibrary('class C<in T> {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            contravariant T @11
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @11
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters_variance_covariant() async {
    var library = await buildLibrary('class C<out T> {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            covariant T @12
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @12
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters_variance_invariant() async {
    var library = await buildLibrary('class C<inout T> {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            invariant T @14
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @14
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_class_typeParameters_variance_multiple() async {
    var library = await buildLibrary('class C<inout T, in U, out V> {}');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          typeParameters
            invariant T @14
              defaultType: dynamic
            contravariant U @20
              defaultType: dynamic
            covariant V @27
              defaultType: dynamic
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          typeParameters
            T @14
              element: <not-implemented>
            U @20
              element: <not-implemented>
            V @27
              element: <not-implemented>
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
  classes
    class C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      typeParameters
        T
        U
        V
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
''');
  }

  test_classAlias() async {
    var library = await buildLibrary('''
class C = D with E, F, G;
class D {}
class E {}
class F {}
class G {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class alias C @6
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: D
          mixins
            E
            F
            G
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @0
                  argumentList: ArgumentList
                    leftParenthesis: ( @0
                    rightParenthesis: ) @0
                  staticElement: <testLibraryFragment>::@class::D::@constructor::new
                  element: <testLibraryFragment>::@class::D::@constructor::new#element
              superConstructor: <testLibraryFragment>::@class::D::@constructor::new
        class D @32
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
        class E @43
          reference: <testLibraryFragment>::@class::E
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::E::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::E
        class F @54
          reference: <testLibraryFragment>::@class::F
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::F::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::F
        class G @65
          reference: <testLibraryFragment>::@class::G
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::G::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::G
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @6
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @0
                  argumentList: ArgumentList
                    leftParenthesis: ( @0
                    rightParenthesis: ) @0
                  staticElement: <testLibraryFragment>::@class::D::@constructor::new
                  element: <testLibraryFragment>::@class::D::@constructor::new#element
              superConstructor: <testLibraryFragment>::@class::D::@constructor::new
        class D @32
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @43
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
        class F @54
          reference: <testLibraryFragment>::@class::F
          element: <testLibrary>::@class::F
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::F::@constructor::new
              element: <testLibraryFragment>::@class::F::@constructor::new#element
              typeName: F
        class G @65
          reference: <testLibraryFragment>::@class::G
          element: <testLibrary>::@class::G
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::G::@constructor::new
              element: <testLibraryFragment>::@class::G::@constructor::new#element
              typeName: G
  classes
    class alias C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D
      mixins
        E
        F
        G
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
    class F
      reference: <testLibrary>::@class::F
      firstFragment: <testLibraryFragment>::@class::F
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::F::@constructor::new
    class G
      reference: <testLibrary>::@class::G
      firstFragment: <testLibraryFragment>::@class::G
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::G::@constructor::new
''');
  }

  test_classAlias_abstract() async {
    var library = await buildLibrary('''
abstract class C = D with E;
class D {}
class E {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        abstract class alias C @15
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: D
          mixins
            E
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @0
                  argumentList: ArgumentList
                    leftParenthesis: ( @0
                    rightParenthesis: ) @0
                  staticElement: <testLibraryFragment>::@class::D::@constructor::new
                  element: <testLibraryFragment>::@class::D::@constructor::new#element
              superConstructor: <testLibraryFragment>::@class::D::@constructor::new
        class D @35
          reference: <testLibraryFragment>::@class::D
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::D::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::D
        class E @46
          reference: <testLibraryFragment>::@class::E
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::E::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::E
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @15
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @0
                  argumentList: ArgumentList
                    leftParenthesis: ( @0
                    rightParenthesis: ) @0
                  staticElement: <testLibraryFragment>::@class::D::@constructor::new
                  element: <testLibraryFragment>::@class::D::@constructor::new#element
              superConstructor: <testLibraryFragment>::@class::D::@constructor::new
        class D @35
          reference: <testLibraryFragment>::@class::D
          element: <testLibrary>::@class::D
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::D::@constructor::new
              element: <testLibraryFragment>::@class::D::@constructor::new#element
              typeName: D
        class E @46
          reference: <testLibraryFragment>::@class::E
          element: <testLibrary>::@class::E
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::E::@constructor::new
              element: <testLibraryFragment>::@class::E::@constructor::new#element
              typeName: E
  classes
    abstract class alias C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: D
      mixins
        E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
          superConstructor: <testLibraryFragment>::@class::D::@constructor::new#element
    class D
      reference: <testLibrary>::@class::D
      firstFragment: <testLibraryFragment>::@class::D
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::D::@constructor::new
    class E
      reference: <testLibrary>::@class::E
      firstFragment: <testLibraryFragment>::@class::E
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::E::@constructor::new
''');
  }

  test_classAlias_base() async {
    var library = await buildLibrary('''
base class C = Object with M;
mixin M {}
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        base class alias C @11
          reference: <testLibraryFragment>::@class::C
          enclosingElement3: <testLibraryFragment>
          supertype: Object
          mixins
            M
          constructors
            synthetic const @-1
              reference: <testLibraryFragment>::@class::C::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @0
                  argumentList: ArgumentList
                    leftParenthesis: ( @0
                    rightParenthesis: ) @0
                  staticElement: dart:core::<fragment>::@class::Object::@constructor::new
                  element: dart:core::<fragment>::@class::Object::@constructor::new#element
      mixins
        mixin M @36
          reference: <testLibraryFragment>::@mixin::M
          enclosingElement3: <testLibraryFragment>
          superclassConstraints
            Object
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class C @11
          reference: <testLibraryFragment>::@class::C
          element: <testLibrary>::@class::C
          constructors
            synthetic const new
              reference: <testLibraryFragment>::@class::C::@constructor::new
              element: <testLibraryFragment>::@class::C::@constructor::new#element
              typeName: C
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @0
                  argumentList: ArgumentList
                    leftParenthesis: ( @0
                    rightParenthesis: ) @0
                  staticElement: dart:core::<fragment>::@class::Object::@constructor::new
                  element: dart:core::<fragment>::@class::Object::@constructor::new#element
      mixins
        mixin M @36
          reference: <testLibraryFragment>::@mixin::M
          element: <testLibrary>::@mixin::M
  classes
    base class alias C
      reference: <testLibrary>::@class::C
      firstFragment: <testLibraryFragment>::@class::C
      supertype: Object
      mixins
        M
      constructors
        synthetic const new
          firstFragment: <testLibraryFragment>::@class::C::@constructor::new
  mixins
    mixin M
      reference: <testLibrary>::@mixin::M
      firstFragment: <testLibraryFragment>::@mixin::M
      superclassConstraints
        Object
''');
  }

  test_classAlias_constructors_beforeOtherProperties() async {
    // https://github.com/dart-lang/sdk/issues/57035
    var library = await buildLibrary('''
abstract mixin class A {}
mixin M {}
class X = A with M;
''');

    var X = library.getClass2('X')!;
    expect(X.constructors2, hasLength(1));
  }

  test_classAlias_constructors_default() async {
    var library = await buildLibrary('''
class A {}
mixin class M {}
class X = A with M;
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
        mixin class M @23
          reference: <testLibraryFragment>::@class::M
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::M::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::M
        class alias X @34
          reference: <testLibraryFragment>::@class::X
          enclosingElement3: <testLibraryFragment>
          supertype: A
          mixins
            M
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::X::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::X
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @0
                  argumentList: ArgumentList
                    leftParenthesis: ( @0
                    rightParenthesis: ) @0
                  staticElement: <testLibraryFragment>::@class::A::@constructor::new
                  element: <testLibraryFragment>::@class::A::@constructor::new#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
        class M @23
          reference: <testLibraryFragment>::@class::M
          element: <testLibrary>::@class::M
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::M::@constructor::new
              element: <testLibraryFragment>::@class::M::@constructor::new#element
              typeName: M
        class X @34
          reference: <testLibraryFragment>::@class::X
          element: <testLibrary>::@class::X
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::X::@constructor::new
              element: <testLibraryFragment>::@class::X::@constructor::new#element
              typeName: X
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @0
                  argumentList: ArgumentList
                    leftParenthesis: ( @0
                    rightParenthesis: ) @0
                  staticElement: <testLibraryFragment>::@class::A::@constructor::new
                  element: <testLibraryFragment>::@class::A::@constructor::new#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
  classes
    class A
      reference: <testLibrary>::@class::A
      firstFragment: <testLibraryFragment>::@class::A
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::A::@constructor::new
    mixin class M
      reference: <testLibrary>::@class::M
      firstFragment: <testLibraryFragment>::@class::M
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::M::@constructor::new
    class alias X
      reference: <testLibrary>::@class::X
      firstFragment: <testLibraryFragment>::@class::X
      supertype: A
      mixins
        M
      constructors
        synthetic new
          firstFragment: <testLibraryFragment>::@class::X::@constructor::new
          superConstructor: <testLibraryFragment>::@class::A::@constructor::new#element
''');
  }

  test_classAlias_constructors_dependencies() async {
    var library = await buildLibrary('''
class A {
  A(int i);
}
mixin class M1 {}
mixin class M2 {}

class C2 = C1 with M2;
class C1 = A with M1;
''');
    checkElementText(library, r'''
library
  reference: <testLibrary>
  definingUnit: <testLibraryFragment>
  units
    <testLibraryFragment>
      enclosingElement3: <null>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          enclosingElement3: <testLibraryFragment>
          constructors
            @12
              reference: <testLibraryFragment>::@class::A::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::A
              parameters
                requiredPositional i @18
                  type: int
        mixin class M1 @36
          reference: <testLibraryFragment>::@class::M1
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::M1::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::M1
        mixin class M2 @54
          reference: <testLibraryFragment>::@class::M2
          enclosingElement3: <testLibraryFragment>
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::M2::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::M2
        class alias C2 @67
          reference: <testLibraryFragment>::@class::C2
          enclosingElement3: <testLibraryFragment>
          supertype: C1
          mixins
            M2
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C2::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C2
              parameters
                requiredPositional i @-1
                  type: int
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @0
                  argumentList: ArgumentList
                    leftParenthesis: ( @0
                    arguments
                      SimpleIdentifier
                        token: i @-1
                        staticElement: <testLibraryFragment>::@class::C2::@constructor::new::@parameter::i
                        element: <testLibraryFragment>::@class::C2::@constructor::new::@parameter::i#element
                        staticType: int
                    rightParenthesis: ) @0
                  staticElement: <testLibraryFragment>::@class::C1::@constructor::new
                  element: <testLibraryFragment>::@class::C1::@constructor::new#element
              superConstructor: <testLibraryFragment>::@class::C1::@constructor::new
        class alias C1 @90
          reference: <testLibraryFragment>::@class::C1
          enclosingElement3: <testLibraryFragment>
          supertype: A
          mixins
            M1
          constructors
            synthetic @-1
              reference: <testLibraryFragment>::@class::C1::@constructor::new
              enclosingElement3: <testLibraryFragment>::@class::C1
              parameters
                requiredPositional i @-1
                  type: int
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @0
                  argumentList: ArgumentList
                    leftParenthesis: ( @0
                    arguments
                      SimpleIdentifier
                        token: i @-1
                        staticElement: <testLibraryFragment>::@class::C1::@constructor::new::@parameter::i
                        element: <testLibraryFragment>::@class::C1::@constructor::new::@parameter::i#element
                        staticType: int
                    rightParenthesis: ) @0
                  staticElement: <testLibraryFragment>::@class::A::@constructor::new
                  element: <testLibraryFragment>::@class::A::@constructor::new#element
              superConstructor: <testLibraryFragment>::@class::A::@constructor::new
----------------------------------------
library
  reference: <testLibrary>
  fragments
    <testLibraryFragment>
      element: <testLibrary>
      classes
        class A @6
          reference: <testLibraryFragment>::@class::A
          element: <testLibrary>::@class::A
          constructors
            new
              reference: <testLibraryFragment>::@class::A::@constructor::new
              element: <testLibraryFragment>::@class::A::@constructor::new#element
              typeName: A
              typeNameOffset: 12
              formalParameters
                i @18
                  element: <testLibraryFragment>::@class::A::@constructor::new::@parameter::i#element
        class M1 @36
          reference: <testLibraryFragment>::@class::M1
          element: <testLibrary>::@class::M1
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::M1::@constructor::new
              element: <testLibraryFragment>::@class::M1::@constructor::new#element
              typeName: M1
        class M2 @54
          reference: <testLibraryFragment>::@class::M2
          element: <testLibrary>::@class::M2
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::M2::@constructor::new
              element: <testLibraryFragment>::@class::M2::@constructor::new#element
              typeName: M2
        class C2 @67
          reference: <testLibraryFragment>::@class::C2
          element: <testLibrary>::@class::C2
          constructors
            synthetic new
              reference: <testLibraryFragment>::@class::C2::@constructor::new
              element: <testLibraryFragment>::@class::C2::@constructor::new#element
              typeName: C2
              formalParameters
                <null-name>
                  element: <testLibraryFragment>::@class::C2::@constructor::new::@parameter::i#element
              constantInitializers
                SuperConstructorInvocation
                  superKeyword: super @0
                  argumentList: ArgumentList
                    leftParenthesis: ( @0
                    arguments
                      SimpleIdentifier
                        token: i @-1
                        staticElement: <testLibrary