/**
  @file assets.h

  This file containts assets to be used in the game. Textures are stored by
  columns for cache friendliness (as rendering also happens by columns),
  4 bits per pixel, which means an index to 16 color subpalette stored at the
  beginning of the image. Images can be converted to this format with the
  provided pything script:

  python img2array.py -t -c -x32 -y32 -ppalette565.png image.png

  by Miloslav Ciz (drummyfish), 2019

  Released under CC0 1.0 (https://creativecommons.org/publicdomain/zero/1.0/)
  plus a waiver of all other intellectual property. The goal of this work is to
  be and remain completely in the public domain forever, available for any use
  whatsoever.
*/

#ifndef SFG_IMAGES_H
#define SFG_IMAGES_H

#include <stdint.h>

#ifndef SFG_PROGRAM_MEMORY
#define SFG_PROGRAM_MEMORY static const
#endif

#ifndef SFG_PROGRAM_MEMORY_U8
#define SFG_PROGRAM_MEMORY_U8(addr) ((uint8_t)(*(addr)))
#endif

#define SFG_TEXTURE_SIZE 32

#define SFG_TEXTURE_STORE_SIZE (16 + (SFG_TEXTURE_SIZE * SFG_TEXTURE_SIZE) / 2)

/**
  Color index which will in textures and sprites be considered transparent.
*/
#define SFG_TRANSPARENT_COLOR 175

/**
  Special index of an implicit texture that consists of only transparent pixels.
*/
#define SFG_TRANSPARENT_TEXTURE 255

static inline uint8_t SFG_getTexel(const uint8_t *texture, uint8_t x, uint8_t y)
{
  x &= 0x1f;
  y &= 0x1f;

  return SFG_PROGRAM_MEMORY_U8(texture +
    ((SFG_PROGRAM_MEMORY_U8(texture + 16 + (x * SFG_TEXTURE_SIZE + y) / 2) >> 
     (4 * (y % 2 == 0))) & 0x0f));
}

#define SFG_WALL_TEXTURE_COUNT 16

SFG_PROGRAM_MEMORY uint8_t
    SFG_wallTextures[SFG_WALL_TEXTURE_COUNT * SFG_TEXTURE_STORE_SIZE] = {
#include "images_walls.txtdata"
};

SFG_PROGRAM_MEMORY uint8_t SFG_itemSprites[13 * SFG_TEXTURE_STORE_SIZE] = {
#include "images_items.txtdata"
};

SFG_PROGRAM_MEMORY uint8_t SFG_backgroundImages[3 * SFG_TEXTURE_STORE_SIZE] = {
#include "images_backgrounds.txtdata"
};

SFG_PROGRAM_MEMORY uint8_t SFG_weaponImages[6 * SFG_TEXTURE_STORE_SIZE] = {
#include "images_weapons.txtdata"
};

SFG_PROGRAM_MEMORY uint8_t SFG_effectSprites[4 * SFG_TEXTURE_STORE_SIZE] = {
#include "images_effects.txtdata"
};

SFG_PROGRAM_MEMORY uint8_t SFG_monsterSprites[19 * SFG_TEXTURE_STORE_SIZE] = {
#include "images_monsters.txtdata"
};

SFG_PROGRAM_MEMORY uint8_t SFG_logoImage[SFG_TEXTURE_STORE_SIZE] = {
#include "images_logo.txtdata"
};

uint8_t SFG_charToFontIndex(char c)
{
  if (c >= 'a' && c <= 'z')
    return c - 'a';

  if (c >= 'A' && c <= 'Z')
    return c - 'A';

  if (c >= '0' && c <= '9')
    return c - '0' + 31;

  switch (c)
  {
    case ' ': return 26; break;
    case '.': return 27; break;
    case ',': return 28; break;
    case '!': return 29; break;
    case '/': return 41; break;
    case '-': return 42; break;
    case '+': return 43; break;
    case '(': return 44; break;
    case ')': return 45; break;
    case '%': return 46; break;
    default:  return 30; break; // "?"
  }
}

#define SFG_FONT_CHARACTER_SIZE 4 ///< width (= height) of font char. in pixels

/**
  4x4 font, each character stored as 16 bits.
*/
static const uint16_t SFG_font[47] =
{
#include "images_font.txtdata"
};

#endif // guard
