// SPDX-License-Identifier: CC0-1.0
/**
  A tool that transforms button images and textures into format used by the STC
  library.

  Written in 2025 by Marcin Serwin <marcin@serwin.dev>

  To the extent possible under law, the author(s) have dedicated all copyright
  and related and neighboring rights to this software to the public domain
  worldwide. This software is distributed without any warranty.

  You should have received a copy of the CC0 Public Domain Dedication along with
  this software. If not, see <http://creativecommons.org/publicdomain/zero/1.0/>
*/

#include "qoi_decoder.h"
#include <assert.h>
#include <stdbool.h>
#include <stdio.h>

bool isWhite(Color c) { return c.r == 255 && c.g == 255 && c.b == 255; }

void dumpData(Header hdr, Color pixels[], const char *name) {

  printf("#define %s_WIDTH %d\n", name, hdr.width);
  printf("#define %s_HEIGHT %d\n", name, hdr.height);
  printf("#define %s_PITCH %d\n", name, (hdr.width + 7) / 8);
  printf("static const unsigned char %s_PIXELS[] = {\n", name);
  int counter = 0;
  uint8_t cur = 0;

  for (uint32_t i = 0; i < hdr.height; i++) {
    for (uint32_t j = 0; j < hdr.width; j++) {
      cur |= (uint8_t)(isWhite(pixels[i * hdr.width + j]) ? 1 << counter : 0);
      if (++counter == 8) {
        printf("0x%.2x, ", cur);
        counter = cur = 0;
      }
    }
    if (counter > 0) {
      printf("0x%.2x, ", cur);
      counter = cur = 0;
    }
    printf("\n");
  }
  printf("};\n\n");
}

int main(int argc, char *argv[]) {
  assert(argc % 2 == 1);
  for (int i = 1; i < argc; i += 2) {
    FILE *f = fopen(argv[i], "rb");
    assert(f);
    static Color pixels[MAX_SIZE * MAX_SIZE];
    Header hdr = readPixels(f, pixels);
    fclose(f);

    dumpData(hdr, pixels, argv[i + 1]);
  }
  return 0;
}
