// SPDX-License-Identifier: CC0-1.0
/**
  A tool that transforms image based Anarch levels into C structures. Based on
  the `img2map.py` tool by Miloslav Číž.

  Written in 2025 by Marcin Serwin <marcin@serwin.dev>

  To the extent possible under law, the author(s) have dedicated all copyright
  and related and neighboring rights to this software to the public domain
  worldwide. This software is distributed without any warranty.

  You should have received a copy of the CC0 Public Domain Dedication along with
  this software. If not, see <http://creativecommons.org/publicdomain/zero/1.0/>
*/

#include "qoi_decoder.h"

#include <limits.h>
#include <stdbool.h>
#include <stdio.h>

static const char *const elementTypes[] = {
    "NONE",
    "BARREL",
    "HEALTH",
    "BULLETS",
    "ROCKETS",
    "PLASMA",
    "TREE",
    "FINISH",
    "TELEPORTER",
    "TERMINAL",
    "COLUMN",
    "RUIN",
    "LAMP",
    "CARD0",
    "CARD1",
    "CARD2",
    "LOCK0",
    "LOCK1",
    "LOCK2",
    "BLOCKER",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "",
    "MONSTER_SPIDER",
    "MONSTER_DESTROYER",
    "MONSTER_WARRIOR",
    "MONSTER_PLASMABOT",
    "MONSTER_ENDER",
    "MONSTER_TURRET",
    "MONSTER_EXPLODER",
};

#define ELEMENT_TYPE_COUNT (int)(sizeof(elementTypes) / sizeof(*elementTypes))

static const char *const propertyTypes[] = {"ELEVATOR", "SQUEEZER", "DOOR"};

#define getPixel(i, j) pixels[(i) * hdr.width + (j)]

int colorToIndex(Color c) { return (c.r << 16) + (c.g << 8) + c.b; }

#define getInversePixel(i, j) paletteInverse[colorToIndex(getPixel(i, j))]

void readPalette(Header hdr, Color const pixels[], int paletteInverse[]) {
#define PALETTE_ROW_LENGTH 64

  for (int i = 0; i < 4; i++) {
    for (int j = 0; j < PALETTE_ROW_LENGTH; j++) {
      getInversePixel(i + 70, j + 5) = i * PALETTE_ROW_LENGTH + j;
    }
  }
}

struct Tile {
  int texture;
  int height;
};

void loadTileDict(Header hdr, const Color pixels[], const int paletteInverse[],
                  int i, int j, struct Tile dict[]) {
  for (int k = 0; k < 64; k++) {
    struct Tile *t = &dict[k];
    t->texture = getInversePixel(i + 31, j + k);
    assert(t->texture <= 7);
    t->height = 0;
    while (t->height < 31 && getInversePixel(i + 30 - t->height, j + k) != 7) {
      t->height++;
    }
  }
}

struct MapTile {
  char tile;
  bool isSpecial;
};

struct Define {
  int tile;
  int prop;
};

#define MAX_DEFINES 128
#define MAP_SIDE_LEN 64

int loadLevelMap(Header hdr, const Color pixels[], const int paletteInverse[],
                 struct MapTile levelMap[MAP_SIDE_LEN][MAP_SIDE_LEN],
                 struct Define defines[MAX_DEFINES]) {
  int defLen = 0;

  for (int i = 0; i < MAP_SIDE_LEN; i++) {
    for (int j = 0; j < MAP_SIDE_LEN; j++) {
      int tile = getInversePixel(5 + i, 70 + j);
      struct MapTile *mt = &levelMap[i][MAP_SIDE_LEN - 1 - j];

      mt->isSpecial = tile >= 64;
      if (!mt->isSpecial) {
        mt->tile = (char)tile;
      } else {
        int prop = tile / 64 - 1;
        tile %= 64;

        int defNum = -1;
        for (int i = 0; i < defLen; i++) {
          if (defines[i].tile == tile && defines[i].prop == prop) {
            defNum = i;
            break;
          }
        }
        if (defNum == -1) {
          assert(defLen < MAX_DEFINES);
          defNum = defLen++;
          defines[defNum].tile = tile;
          defines[defNum].prop = prop;
        }

        assert(defNum < CHAR_MAX);
        mt->tile = (char)defNum;
      }
    }
  }
  return defLen;
}

struct Element {
  int type;
  int x;
  int y;
};

#define ELEMENTS_LEN 128

struct Player {
  int x;
  int y;
  int direction;
};

void loadElements(Header hdr, const Color pixels[], const int paletteInverse[],
                  struct Element elements[ELEMENTS_LEN],
                  struct Player *player) {

  int elemCount = 0;
  bool playerFound = false;

  for (int i = 0; i < MAP_SIDE_LEN; i++) {
    for (int j = 0; j < MAP_SIDE_LEN; j++) {
      int tile = getInversePixel(70 + i, 70 + j);
      if (tile < ELEMENT_TYPE_COUNT) {
        assert(elemCount < ELEMENTS_LEN);
        elements[elemCount].type = tile;
        elements[elemCount].x = 63 - j;
        elements[elemCount].y = i;
        elemCount++;
      } else if (tile >= 240) {
        assert(!playerFound);
        playerFound = true;

        player->x = 63 - j;
        player->y = i;
        player->direction = (tile - 240) * 16;
      }
    }
  }
  assert(playerFound);
}

struct Level {
  char *name;
  struct Tile floorDict[64];
  struct Tile ceilDict[64];
  int floorColor;
  int ceilColor;
  int backgroundTex;
  int doorTex;
  struct MapTile map[MAP_SIDE_LEN][MAP_SIDE_LEN];

  int definesLength;
  struct Define defines[MAX_DEFINES];

  struct Element elements[ELEMENTS_LEN];
  struct Player player;

  int textures[7];
};

void readLevel(Header hdr, Color *pixels, int *paletteInverse,
               struct Level *level) {
  loadTileDict(hdr, pixels, paletteInverse, 37, 5, level->floorDict);
  loadTileDict(hdr, pixels, paletteInverse, 5, 5, level->ceilDict);

  level->floorColor = getInversePixel(122, 41);
  level->ceilColor = getInversePixel(118, 41);
  level->backgroundTex = getInversePixel(126, 41);
  level->doorTex = getInversePixel(130, 41);

  level->definesLength =
      loadLevelMap(hdr, pixels, paletteInverse, level->map, level->defines);

  loadElements(hdr, pixels, paletteInverse, level->elements, &level->player);

  for (int i = 0; i < 7; i++) {
    level->textures[i] = getInversePixel(114, 41 + i * 4);
  }
}

void printLevel(const struct Level level) {

#define defName(i) 'A' + i, 'A' + i
#define mapXScale()                                                            \
  printf("    // ");                                                           \
  for (int i = 0; i < 64; i++) {                                               \
    printf("%-2d ", i);                                                        \
  }                                                                            \
  printf("\n")

  printf("SFG_PROGRAM_MEMORY SFG_Level %s =\n"
         "  {          // level\n"
         "    {        // mapArray\n"
         "    #define o 0\n",
         level.name);

  for (int i = 0; i < level.definesLength; i++) {
    printf("    #define %c%c (%d | SFG_TILE_PROPERTY_%s)\n", defName(i),
           level.defines[i].tile, propertyTypes[level.defines[i].prop]);
  }

  mapXScale();

  for (int i = 0; i < 64; i++) {
    printf("/*%-2d*/ ", i);
    for (int j = 0; j < 64; j++) {
      struct MapTile mt = level.map[i][j];
      if (!mt.isSpecial) {
        if (mt.tile) {
          printf("%-2d", mt.tile);
        } else {
          printf("o ");
        }
      } else {
        char c = mt.tile + 'A';
        printf("%c%c", c, c);
      }

      printf(i < 63 || j < 63 ? "," : " ");
    }
    printf(" /*%-2d*/ \n", i);
  }

  mapXScale();

  for (int i = 0; i < level.definesLength; i++) {
    printf("    #undef %c%c\n", defName(i));
  }
  printf("    #undef o\n"
         "    },\n"
         "    {        // tileDictionary\n      ");

  for (int i = 0; i < 64; i++) {
    printf("SFG_TD(%2d,%2d,%d,%d)", level.floorDict[i].height,
           level.ceilDict[i].height, level.floorDict[i].texture,
           level.ceilDict[i].texture);

    printf(i != 63 ? "," : " ");
    if ((i + 1) % 4 == 0) {
      printf(" // %d \n      ", i - 3);
    }
  }
  printf("},                    // tileDictionary\n");
  printf("    {");
  for (int i = 0; i < 7; i++) {
    printf(i == 0 ? "%-2d" : ",%-2d", level.textures[i]);
  }
  printf("}, // textureIndices\n");
#define numAlign(n) n, n < 10 ? " " : ""
  printf("    %d,%s                     // doorTextureIndex\n",
         numAlign(level.doorTex));
  printf("    %d,%s                     // floorColor\n",
         numAlign(level.floorColor));
  printf("    %d,%s                     // ceilingColor\n",
         numAlign(level.ceilColor));
  printf("    {%-2d, %-2d, %-3d},          // player start: x, y, direction\n",
         level.player.x, level.player.y, level.player.direction);
  printf("    %d,%s                     // backgroundImage\n",
         numAlign(level.backgroundTex));
  printf("    {                       // elements\n");
  for (int i = 0; i < ELEMENTS_LEN; i++) {
    struct Element e = level.elements[i];
    if (i % 2 == 0) {
      printf("      ");
    }
    printf("{SFG_LEVEL_ELEMENT_%s, {%d,%d}}", elementTypes[e.type], e.x, e.y);
    if (i < 127) {
      printf(",");
    }
    if (i % 2 == 1) {
      printf("\n");
    }
  }
  printf("    }, // elements\n"
         "  } // level\n"
         "  ;\n\n");
}

int main(int argc, char *argv[]) {
  assert(argc % 2 == 1);
  for (int i = 1; i < argc; i += 2) {
    FILE *f = fopen(argv[i], "rb");
    assert(f);
    static Color pixels[MAX_SIZE * MAX_SIZE];
    Header hdr = readPixels(f, pixels);

    static int paletteInverse[256 * 256 * 256];
    readPalette(hdr, pixels, paletteInverse);

    struct Level level = {0};
    level.name = argv[i + 1];
    readLevel(hdr, pixels, paletteInverse, &level);
    fclose(f);

    printLevel(level);
  }
  printf("#define SFG_NUMBER_OF_LEVELS %d\n\n", argc / 2);
  printf("static const SFG_Level * SFG_levels[SFG_NUMBER_OF_LEVELS] = {\n");
  for (int i = 2; i < argc; i += 2) {
    printf("  &%s,\n", argv[i]);
  }
  printf("};\n");
}
