// SPDX-License-Identifier: CC0-1.0
/**
  An SDL3 implementation of the Anarch game frontend. Based on the SDL2
  frontend present in `main_sdl.c` by Miloslav Číž.

  Written in 2025 by Marcin Serwin <marcin@serwin.dev>

  To the extent possible under law, the author(s) have dedicated all copyright
  and related and neighboring rights to this software to the public domain
  worldwide. This software is distributed without any warranty.

  You should have received a copy of the CC0 Public Domain Dedication along with
  this software. If not, see <http://creativecommons.org/publicdomain/zero/1.0/>
*/

#define SDL_MAIN_USE_CALLBACKS
#include <SDL3/SDL.h>
#include <SDL3/SDL_main.h>

#ifdef __EMSCRIPTEN__
#include <emscripten.h>
#define SFG_CAN_EXIT 0
#endif

#define SFG_HEADBOB_SHEAR (-1 * SFG_SCREEN_RESOLUTION_Y / 80)
#define SFG_FULLSCREEN 1
#define SFG_BACKGROUND_BLUR 1
#define SFG_DITHERED_SHADOW 1
#define SFG_LOG(str) SDL_LogInfo(SDL_LOG_CATEGORY_APPLICATION, "%s", str);

static inline uint8_t getCurrentLanguage(void);
#define SFG_DEFAULT_LANG getCurrentLanguage()

#if !defined(SDL_PLATFORM_ANDROID) && !defined(SDL_PLATFORM_IOS)
/*
  SDL is easier to play thanks to nice controls so make the player take full
  damage to make it a bit harder.
*/
#define SFG_PLAYER_DAMAGE_MULTIPLIER 1024
#endif

#include "game.h"
#include "settings.h"
#include "sounds.h"

#if SFG_TOUCHCONTROLS_SUPPORT
#define STC_IMPLEMENTATION
#define STC_PUBLIC_API static
#include "stc.h"
#endif

#define GAMEPAD_DEADZONE 10000
#define GAMEPAD_AXIS_SCALE 1024
#define TOUCH_CAMERA_SCALE 2

#define STREAM_COUNT 8
#define MUSIC_STREAM STREAM_COUNT
#define MUSIC_VOLUME 0.2f

#if SFG_TOUCHCONTROLS_SUPPORT
enum Touch {
  TOUCH_DISABLED,
  TOUCH_AUTO_HIDDEN,
  TOUCH_AUTO_SHOWN,
  TOUCH_ENABLED,
};
#endif

struct InputData {
  float wheel;
  SDL_Gamepad *gamepad;
  bool backButton;
#if SFG_TOUCHCONTROLS_SUPPORT
  struct STC_TouchData touchData;
  enum Touch touchControls;
#endif
};

static struct AppData {
  SDL_Window *window;
  SDL_Renderer *renderer;
  SDL_AudioDeviceID audioDevice;

  SDL_Texture *texture;
  uint16_t *textureBuf;
  SDL_FRect renderTarget;

  SDL_AudioStream *streams[STREAM_COUNT + 1];
  int curStream;
  bool musicOn;

  struct InputData input;

  bool isFullscreen;

#ifdef __EMSCRIPTEN__
  bool inited;
  bool idbfsLoaded;
#endif
} appData;

#ifdef __EMSCRIPTEN__
void idbfsSynced(int err) {
  if (err) {
    SDL_LogWarn(SDL_LOG_CATEGORY_SYSTEM, "Mounting IDBFS failed.");
    return;
  }
  appData.idbfsLoaded = true;
  SDL_Event event;
  event.type = SDL_EVENT_USER;
  SDL_PushEvent(&event);
}
#endif

static inline void SFG_setPixel(uint16_t x, uint16_t y, uint8_t colorIndex) {
  appData.textureBuf[y * SFG_SCREEN_RESOLUTION_X + x] =
      paletteRGB565[colorIndex];
}

void SFG_setMusic(uint8_t value) {
  switch (value) {
  case SFG_MUSIC_TURN_ON:
    appData.musicOn = true;
    break;
  case SFG_MUSIC_TURN_OFF:
    appData.musicOn = false;
    break;
  case SFG_MUSIC_NEXT:
    SFG_nextMusicTrack();
    break;
  }
}

void SFG_playSound(uint8_t soundIndex, uint8_t volume) {
  if (!appData.audioDevice) {
    return;
  }
  SDL_AudioStream *s = appData.streams[appData.curStream];
  SDL_SetAudioStreamGain(s, (float)volume / 255);
  SDL_PutAudioStreamData(s, SFG_sounds[soundIndex], SFG_SFX_SAMPLE_COUNT);
  if (++appData.curStream == STREAM_COUNT) {
    appData.curStream = 0;
  }
}

void SFG_sleepMs(uint16_t timeMs) { SDL_Delay(timeMs); }

uint32_t SFG_getTimeMs(void) { return (uint32_t)SDL_GetTicks(); }

void SFG_processEvent(uint8_t event, uint8_t data) {
  SDL_Gamepad *gamepad = appData.input.gamepad;
  switch (event) {

  case SFG_EVENT_VIBRATE:
    SDL_RumbleGamepad(gamepad, 0xffff, 0xffff, 500);
    break;

  case SFG_EVENT_PLAYER_TELEPORTS:
    SDL_RumbleGamepad(gamepad, 0xffff, 0xffff, 200);
    break;

  case SFG_EVENT_BORDER_CHANGES_COLOR:
    switch (data) {
    case 0:
      SDL_SetGamepadLED(gamepad, 0x00, 0x00, 0x00);
      break;
    case 1:
      SDL_SetGamepadLED(gamepad, 0xff, 0x00, 0x00);
      break;
    case 2:
      SDL_SetGamepadLED(gamepad, 0x00, 0xff, 0x00);
      break;
    }
    break;

  case SFG_EVENT_GAME_STATE_CHANGED:
    SDL_SetWindowRelativeMouseMode(appData.window,
                                   data != SFG_GAME_STATE_MENU &&
                                       data != SFG_GAME_STATE_INIT);
    break;
  }
}

#define MAX_STORAGE_WAIT_MS 100

static SDL_Storage *openUserStorage(void) {
  SDL_Storage *strg =
      SDL_OpenUserStorage(SFG_ORG_NAME_STRING, SFG_NAME_NOSPACE_STRING, 0);
  if (!strg) {
    SDL_LogWarn(SDL_LOG_CATEGORY_SYSTEM,
                "Failed to open user storage: %s. Game will not be saved.",
                SDL_GetError());
    return NULL;
  }

  for (int i = 0; !SDL_StorageReady(strg) && i < MAX_STORAGE_WAIT_MS; i++) {
    SDL_Delay(1);
  }

  if (!SDL_StorageReady(strg)) {
    SDL_LogWarn(
        SDL_LOG_CATEGORY_SYSTEM,
        "User storage is not ready after wait period. Game will not be saved.");
    SDL_CloseStorage(strg);
    return NULL;
  }
  return strg;
}

void SFG_save(uint8_t data[SFG_SAVE_SIZE]) {
  SDL_Storage *strg = openUserStorage();
  if (!strg) {
    return;
  }

  if (!SDL_WriteStorageFile(strg, SFG_SAVE_FILE_PATH, data, SFG_SAVE_SIZE)) {
    SDL_LogWarn(SDL_LOG_CATEGORY_SYSTEM,
                "Writing to user storage failed: %s. Game will not be saved.",
                SDL_GetError());
  }
  SDL_CloseStorage(strg);
}

uint8_t SFG_load(uint8_t data[SFG_SAVE_SIZE]) {
#ifdef __EMSCRIPTEN__
  if (!appData.idbfsLoaded) {
    return 0;
  }
#endif
  SDL_Storage *strg = openUserStorage();
  if (!strg) {
    return 0;
  }

  Uint64 len = 0;
  if (!SDL_GetStorageFileSize(strg, SFG_SAVE_FILE_PATH, &len)) {
    SDL_LogWarn(SDL_LOG_CATEGORY_SYSTEM,
                "Couldn't determine save file size: %s.", SDL_GetError());
  } else {
    if (!SDL_ReadStorageFile(strg, SFG_SAVE_FILE_PATH, data, len)) {
      SDL_LogWarn(SDL_LOG_CATEGORY_SYSTEM, "Loading save file failed: %s.",
                  SDL_GetError());
    } else {
      for (int i = len; i < SFG_SAVE_SIZE; i++) {
        data[i] = 0;
      }
    }
  }

  SDL_CloseStorage(strg);
  return 1;
}

void musicCallback(void *userdata, SDL_AudioStream *stream, int am, int total) {
  (void)userdata;
  (void)total;

  if (!appData.musicOn || !appData.audioDevice) {
    return;
  }
  uint8_t buf[1024];
  while (am > 0) {
    const int cur = RCL_max(am, 1024);
    for (int i = 0; i < cur; i++) {
      buf[i] = SFG_getNextMusicSample();
    }
    SDL_PutAudioStreamData(stream, buf, cur);
    am -= cur;
  }
}

int8_t SFG_keyPressed(uint8_t key) {
  const bool *sdlKeyboardState = SDL_GetKeyboardState(NULL);
  const SDL_MouseButtonFlags sdlMouseState = SDL_GetMouseState(NULL, NULL);

#define k(x) sdlKeyboardState[SDL_SCANCODE_##x]

#if SFG_TOUCHCONTROLS_SUPPORT
#define t(x) STC_isPressed(&appData.input.touchData, STC_BUTTON_##x)
#else
#define t(x) false
#endif

#define b(x)                                                                   \
  (appData.input.gamepad &&                                                    \
   SDL_GetGamepadButton(appData.input.gamepad, SDL_GAMEPAD_BUTTON_##x))
#define a(x)                                                                   \
  (appData.input.gamepad                                                       \
       ? SDL_GetGamepadAxis(appData.input.gamepad, SDL_GAMEPAD_AXIS_##x)       \
       : 0)
#define m(x) sdlMouseState &SDL_BUTTON_MASK(SDL_BUTTON_##x)
#define w(x) (appData.input.wheel x 0 ? appData.input.wheel = 0, true : false);
#define bb                                                                     \
  appData.input.backButton ? appData.input.backButton = false, true : false

  switch (key) {
  case SFG_KEY_UP:
    return k(UP) || k(W) || k(KP_8) || t(UP) || b(DPAD_UP) ||
           a(LEFTY) < -GAMEPAD_DEADZONE;
  case SFG_KEY_RIGHT:
    return k(RIGHT) || k(E) || k(KP_6) || b(DPAD_RIGHT);
  case SFG_KEY_DOWN:
    return k(DOWN) || k(S) || k(KP_5) || k(KP_2) || t(DOWN) || b(DPAD_DOWN) ||
           a(LEFTY) > GAMEPAD_DEADZONE;
  case SFG_KEY_LEFT:
    return k(LEFT) || k(Q) || k(KP_4) || b(DPAD_LEFT);
  case SFG_KEY_A:
    return k(J) || k(RETURN) || k(LCTRL) || k(RCTRL) || t(A) || b(SOUTH) ||
           m(LEFT) || a(RIGHT_TRIGGER) > GAMEPAD_DEADZONE;
  case SFG_KEY_B:
    return k(K) || k(LSHIFT) || b(EAST) || t(B);
  case SFG_KEY_C:
    return k(L) || b(NORTH) || t(C);
  case SFG_KEY_JUMP:
    return k(SPACE) || b(WEST);
  case SFG_KEY_STRAFE_LEFT:
    return k(A) || k(KP_7) || t(LEFT) || a(LEFTX) < -GAMEPAD_DEADZONE;
  case SFG_KEY_STRAFE_RIGHT:
    return k(D) || k(KP_9) || t(RIGHT) || a(LEFTX) > GAMEPAD_DEADZONE;
  case SFG_KEY_MAP:
    return k(TAB) || b(BACK);
  case SFG_KEY_CYCLE_WEAPON:
    return k(F) || m(MIDDLE);
  case SFG_KEY_TOGGLE_FREELOOK:
    return b(RIGHT_STICK) || m(RIGHT);
  case SFG_KEY_MENU:
    return k(ESCAPE) || b(START) || bb;
  case SFG_KEY_NEXT_WEAPON:
    return k(P) || k(X) || b(RIGHT_SHOULDER) || w(>);
  case SFG_KEY_PREVIOUS_WEAPON:
    return k(O) || k(Y) || k(Z) || b(LEFT_SHOULDER) || w(<);
  default:
    return false;
  }
#undef k
#undef b
#undef m
#undef w
#undef t
}

void SFG_getMouseOffset(int16_t *x, int16_t *y) {
  float mdx, mdy;
  SDL_GetRelativeMouseState(&mdx, &mdy);

  int adx = a(RIGHTX), ady = a(RIGHTY);

  *x = (int16_t)mdx + (adx > GAMEPAD_DEADZONE || adx < -GAMEPAD_DEADZONE) *
                          adx / GAMEPAD_AXIS_SCALE;
  *y = (int16_t)mdy + (ady > GAMEPAD_DEADZONE || ady < -GAMEPAD_DEADZONE) *
                          ady / GAMEPAD_AXIS_SCALE;

#if SFG_TOUCHCONTROLS_SUPPORT
  float tdx, tdy;
  STC_getCameraDiff(&appData.input.touchData, &tdx, &tdy);
  *x += (int16_t)tdx * TOUCH_CAMERA_SCALE;
  *y += (int16_t)tdy * TOUCH_CAMERA_SCALE;
#endif

#undef a
}

static inline uint8_t getCurrentLanguage(void) {
  int count;
  SDL_Locale **locales = SDL_GetPreferredLocales(&count);
  for (int i = 0; i < count; i++) {
    const char *lang = locales[i]->language;
    if (strcmp(lang, "en") == 0) {
      return LANG_EN;
    }
    if (strcmp(lang, "cs") == 0) {
      return LANG_CS;
    }
    if (strcmp(lang, "pl") == 0) {
      return LANG_PL;
    }
  }
  return LANG_EN;
}

static SDL_AppResult fail(void) {
  char *error = SDL_strdup(SDL_GetError());

  SDL_ShowSimpleMessageBox(SDL_MESSAGEBOX_ERROR, "Critical error", error, NULL);
  SDL_LogCritical(SDL_LOG_CATEGORY_ERROR, "Critical error: %s\n", error);

  SDL_free(error);
  return SDL_APP_FAILURE;
}

static SDL_FRect getLetterboxedTarget(int w, int h) {
  SDL_FRect result;
  if (w * SFG_SCREEN_RESOLUTION_Y > SFG_SCREEN_RESOLUTION_X * h) {
    result.h = h;
    result.w = (float)SFG_SCREEN_RESOLUTION_X * h / SFG_SCREEN_RESOLUTION_Y;
    result.x = (w - result.w) / 2.0f;
    result.y = 0;
  } else {
    result.h = (float)w * SFG_SCREEN_RESOLUTION_Y / SFG_SCREEN_RESOLUTION_X;
    result.w = w;
    result.x = 0;
    result.y = (h - result.h) / 2.0f;
  }
  return result;
}

static void setAppMetadata(void) {
  SDL_SetAppMetadataProperty(SDL_PROP_APP_METADATA_NAME_STRING,
                             SFG_NAME_STRING);
  SDL_SetAppMetadataProperty(SDL_PROP_APP_METADATA_VERSION_STRING,
                             SFG_VERSION_STRING);
  SDL_SetAppMetadataProperty(SDL_PROP_APP_METADATA_IDENTIFIER_STRING,
                             SFG_IDENTIFIER_STRING);
  SDL_SetAppMetadataProperty(SDL_PROP_APP_METADATA_COPYRIGHT_STRING,
                             SFG_COPYRIGHT_STRING);
  SDL_SetAppMetadataProperty(SDL_PROP_APP_METADATA_URL_STRING, SFG_URL_STRING);
  SDL_SetAppMetadataProperty(SDL_PROP_APP_METADATA_TYPE_STRING, "game");
}

static void printHelp(char *name) {
  char *prefPath =
      SDL_GetPrefPath(SFG_ORG_NAME_STRING, SFG_NAME_NOSPACE_STRING);
  SDL_GetUserFolder(SDL_FOLDER_SAVEDGAMES);
  SDL_Log(SFG_NAME_STRING
          ", version " SFG_VERSION_STRING " - a suckless FPS game.\n"
          "This is free and unencumbered software released into the public "
          "domain.\n"
          "Written by Miloslav Číž. SDL3 frontend by Marcin Serwin. SDL3 was "
          "created by Sam Lantinga.\n\n");
  SDL_Log("Usage: %s [OPTION]...\n\n", name);
  SDL_Log("Options:\n"
          "  -f, --fullscreen  start in fullscreen mode"
#if SFG_FULLSCREEN
          " (the default)"
#endif
          "\n"
          "  -w, --windowed    start in windowed mode"
#if !SFG_FULLSCREEN
          " (the default)"
#endif
          "\n"
#if SFG_TOUCHCONTROLS_SUPPORT
          "  -t, --touch MODE  enable touch controls (auto|always|never)\n"
#endif
          "  -h, --help        display this help and exit\n"
          "  -v, --version     display version and exit\n"
          "\n"
          "Save file data is stored in: %s\n",
          prefPath);

  SDL_free(prefPath);
}

typedef struct Args {
  SDL_AppResult exit;
  bool fullscreen;
#if SFG_TOUCHCONTROLS_SUPPORT
  enum Touch touch;
#endif
} Args;

static Args parseArgs(int argc, char *argv[]) {
#define arg(s) SDL_strcmp(argv[i], s) == 0
  Args args;
  args.exit = SDL_APP_CONTINUE;
  args.fullscreen = SFG_FULLSCREEN;
#if SFG_TOUCHCONTROLS_SUPPORT
  args.touch = TOUCH_AUTO_SHOWN;
#endif

  for (int i = 1; i < argc; i++) {
    if (arg("-h") || arg("--help")) {
      printHelp(argv[0]);
      args.exit = SDL_APP_SUCCESS;
      break;
    } else if (arg("-v") || arg("--version")) {
      SDL_Log(SFG_VERSION_STRING);
      args.exit = SDL_APP_SUCCESS;
      break;
    } else if (arg("-f") || arg("--fullscreen")) {
      args.fullscreen = true;
    } else if (arg("-w") || arg("--windowed")) {
      args.fullscreen = false;
#if SFG_TOUCHCONTROLS_SUPPORT
    } else if (arg("-t") || arg("--touch")) {
      i++;
      if (arg("auto")) {
        args.touch = TOUCH_AUTO_SHOWN;
      } else if (arg("never")) {
        args.touch = TOUCH_DISABLED;
      } else if (arg("always")) {
        args.touch = TOUCH_ENABLED;
      } else {
        SDL_LogCritical(SDL_LOG_CATEGORY_APPLICATION, "Unknown touch mode: %s",
                        argv[i]);
        args.exit = SDL_APP_FAILURE;
        break;
      }
#endif
    } else {
      SDL_LogCritical(SDL_LOG_CATEGORY_APPLICATION, "Unknown argument: %s",
                      argv[i]);
      args.exit = SDL_APP_FAILURE;
      break;
    }
  }
#undef arg
  return args;
}

SDL_AppResult SDL_AppInit(void **data, int argc, char *argv[]) {
  *data = &appData;
  struct AppData *appData = *(struct AppData **)data;
  setAppMetadata();

  Args args = parseArgs(argc, argv);
  if (args.exit != SDL_APP_CONTINUE) {
    return args.exit;
  }

  if (!SDL_InitSubSystem(SDL_INIT_VIDEO | SDL_INIT_AUDIO | SDL_INIT_GAMEPAD)) {
    return fail();
  }
#if SFG_TOUCHCONTROLS_SUPPORT
  if (args.touch == TOUCH_AUTO_SHOWN) {
#if defined(SDL_PLATFORM_ANDROID) || defined(SDL_PLATFORM_IOS)
    appData->input.touchControls = TOUCH_AUTO_SHOWN;
#else
    appData->input.touchControls = TOUCH_AUTO_HIDDEN;
#endif
  } else {
    appData->input.touchControls = args.touch;
  }
#endif
  SDL_SetHint(SDL_HINT_TOUCH_MOUSE_EVENTS, "0");
  SDL_SetHint(SDL_HINT_MOUSE_TOUCH_EVENTS, "0");
  if (!SDL_CreateWindowAndRenderer(
          SFG_NAME_STRING " | " SFG_VERSION_STRING, SFG_SCREEN_RESOLUTION_X,
          SFG_SCREEN_RESOLUTION_Y,
          SDL_WINDOW_RESIZABLE | (args.fullscreen ? SDL_WINDOW_FULLSCREEN : 0),
          &appData->window, &appData->renderer)) {
    return fail();
  }
  int w, h;
  SDL_GetWindowSize(appData->window, &w, &h);
  appData->renderTarget = getLetterboxedTarget(w, h);
  appData->texture = SDL_CreateTexture(
      appData->renderer, SDL_PIXELFORMAT_RGB565, SDL_TEXTUREACCESS_STREAMING,
      SFG_SCREEN_RESOLUTION_X, SFG_SCREEN_RESOLUTION_Y);
  if (!appData->texture) {
    return fail();
  }
  SDL_SetTextureScaleMode(appData->texture, SDL_SCALEMODE_NEAREST);

  appData->audioDevice =
      SDL_OpenAudioDevice(SDL_AUDIO_DEVICE_DEFAULT_PLAYBACK, NULL);
  if (!appData->audioDevice) {
    SDL_LogWarn(SDL_LOG_CATEGORY_SYSTEM, "Failed to open audio device: %s",
                SDL_GetError());
  } else {
    SDL_AudioSpec spec;
    spec.format = SDL_AUDIO_U8;
    spec.channels = 1;
    spec.freq = 8000;

    for (int i = 0; i < STREAM_COUNT + 1; i++) {
      appData->streams[i] = SDL_CreateAudioStream(&spec, NULL);
    }
    SDL_BindAudioStreams(appData->audioDevice, appData->streams,
                         STREAM_COUNT + 1);

    SDL_SetAudioStreamGain(appData->streams[MUSIC_STREAM], MUSIC_VOLUME);
    SDL_SetAudioStreamGetCallback(appData->streams[MUSIC_STREAM], musicCallback,
                                  NULL);
  }

#if SFG_TOUCHCONTROLS_SUPPORT
  if (appData->input.touchControls != TOUCH_DISABLED &&
      !STC_init(&appData->input.touchData, appData->renderer)) {
    return fail();
  }
#endif

#ifdef __EMSCRIPTEN__
  SDL_assert_release(SDL_RegisterEvents(1) == SDL_EVENT_USER);
  const char *prefpath = SDL_GetPrefPath(NULL, "anarch");
  if (!prefpath) {
    idbfsSynced(1);
  } else {
    EM_ASM(
        {
          FS.mount(IDBFS, {autoPersist : true}, UTF8ToString($0));
          FS.syncfs(
              true, function(err) {
                ccall("idbfsSynced", "void", ["number"], err ? 1 : 0);
              });
        },
        prefpath);
  }
#else
  SFG_init();
#endif

  return SDL_APP_CONTINUE;
}

SDL_AppResult SDL_AppIterate(void *data) {
  struct AppData *appData = (struct AppData *)data;
#ifdef __EMSCRIPTEN__
  if (!appData->inited) {
    return SDL_APP_CONTINUE;
  }
#endif

  int pitch;
  if (!SDL_LockTexture(appData->texture, NULL, (void **)&appData->textureBuf,
                       &pitch)) {
    return fail();
  }
  SDL_assert(pitch == SFG_GAME_RESOLUTION_X * sizeof(uint16_t));

  bool cont = SFG_mainLoopBody();

  SDL_UnlockTexture(appData->texture);

  SDL_SetRenderDrawColor(appData->renderer, 0x00, 0x00, 0x00, SDL_ALPHA_OPAQUE);
  SDL_RenderClear(appData->renderer);

  SDL_RenderTexture(appData->renderer, appData->texture, NULL,
                    &appData->renderTarget);

#if SFG_TOUCHCONTROLS_SUPPORT
  if (appData->input.touchControls == TOUCH_ENABLED ||
      appData->input.touchControls == TOUCH_AUTO_SHOWN) {
    STC_render(&appData->input.touchData, appData->renderer);
  }
#endif
  SDL_RenderPresent(appData->renderer);

  return cont ? SDL_APP_CONTINUE : SDL_APP_SUCCESS;
}

SDL_AppResult SDL_AppEvent(void *data, SDL_Event *event) {
  struct AppData *appData = (struct AppData *)data;
  switch (event->type) {
  case SDL_EVENT_QUIT:
    return SDL_APP_SUCCESS;

  case SDL_EVENT_WINDOW_RESIZED:
    appData->renderTarget =
        getLetterboxedTarget(event->window.data1, event->window.data2);
#if SFG_TOUCHCONTROLS_SUPPORT
    STC_updateSafeArea(&appData->input.touchData, appData->renderer);
#endif
    break;

#ifdef __EMSCRIPTEN__
  case SDL_EVENT_USER:
    SFG_init();
    appData->inited = true;
    break;
#endif

  case SDL_EVENT_WINDOW_ENTER_FULLSCREEN:
    appData->isFullscreen = true;
    break;

  case SDL_EVENT_WINDOW_LEAVE_FULLSCREEN:
    appData->isFullscreen = false;
    break;

  case SDL_EVENT_KEY_DOWN:
    switch (event->key.scancode) {
    case SDL_SCANCODE_F11:
      SDL_SetWindowFullscreen(appData->window, !appData->isFullscreen);
      break;
    case SDL_SCANCODE_AC_BACK:
      if (SFG_game.state == SFG_GAME_STATE_MENU) {
        return SDL_APP_SUCCESS;
      }
      appData->input.backButton = true;
      break;
    default:
#if SFG_TOUCHCONTROLS_SUPPORT
      if (appData->input.touchControls == TOUCH_AUTO_SHOWN) {
        appData->input.touchControls = TOUCH_AUTO_HIDDEN;
      }
#endif
      break;
    }
    break;

#if SFG_TOUCHCONTROLS_SUPPORT
  case SDL_EVENT_FINGER_DOWN:
  case SDL_EVENT_FINGER_MOTION:
  case SDL_EVENT_FINGER_UP:
    if (appData->input.touchControls != TOUCH_DISABLED) {
      SDL_ConvertEventToRenderCoordinates(appData->renderer, event);
      STC_handleTouchEvent(&appData->input.touchData, event->tfinger);
    }
    if (appData->input.touchControls == TOUCH_AUTO_HIDDEN) {
      appData->input.touchControls = TOUCH_AUTO_SHOWN;
    }
    break;
#endif

  case SDL_EVENT_MOUSE_WHEEL:
    appData->input.wheel += event->wheel.y;
    break;

  case SDL_EVENT_GAMEPAD_ADDED:
    if (!appData->input.gamepad) {
      appData->input.gamepad = SDL_OpenGamepad(event->gdevice.which);
    }
    break;

  case SDL_EVENT_GAMEPAD_REMOVED:
    if (SDL_GetGamepadID(appData->input.gamepad) == event->gdevice.which) {
      SDL_CloseGamepad(appData->input.gamepad);
      appData->input.gamepad = NULL;
    }
    break;
  }
  return SDL_APP_CONTINUE;
}

void SDL_AppQuit(void *data, SDL_AppResult result) {
  (void)result;

  struct AppData *appData = (struct AppData *)data;
#if SFG_TOUCHCONTROLS_SUPPORT
  if (appData->input.touchControls != TOUCH_DISABLED) {
    STC_destroy(&appData->input.touchData);
  }
#endif

  if (appData->input.gamepad) {
    SDL_CloseGamepad(appData->input.gamepad);
  }
  for (int i = 0; i < STREAM_COUNT + 1; i++) {
    SDL_DestroyAudioStream(appData->streams[i]);
  }
  SDL_CloseAudioDevice(appData->audioDevice);
  SDL_DestroyTexture(appData->texture);
  SDL_DestroyRenderer(appData->renderer);
  SDL_DestroyWindow(appData->window);
  SDL_Quit();
}
