# SPDX-License-Identifier: CC0-1.0
#
# Flake specification that describes the development environment for the Anarch
# RE project.
#
# Written in 2025 by Marcin Serwin <marcin@serwin.dev>
#
# To the extent possible under law, the author(s) have dedicated all copyright
# and related and neighboring rights to this software to the public domain
# worldwide. This software is distributed without any warranty.
#
# You should have received a copy of the CC0 Public Domain Dedication along with
# this software. If not, see <http://creativecommons.org/publicdomain/zero/1.0/>

{
  inputs = {
    nixpkgs.url = "nixpkgs/nixos-unstable";
  };

  outputs =
    {
      self,
      nixpkgs,
    }:
    let
      supportedSystems = [
        "x86_64-linux"
        "aarch64-linux"
        "x86_64-darwin"
        "aarch64-darwin"
      ];
      forEachSupportedSystem =
        f: nixpkgs.lib.genAttrs supportedSystems (system: f { pkgs = nixpkgs.legacyPackages.${system}; });
    in
    {
      packages = forEachSupportedSystem (
        { pkgs }:
        let
          anarch-re-cross =
            pkgs':
            pkgs'.callPackage ./package.nix {
              sdl3 = pkgs'.sdl3.override {
                ibusSupport = false;
                x11Support = false;
                testSupport = false;
                pulseaudioSupport = false;
              };

            };
        in
        rec {
          anarch-re-win = anarch-re-cross pkgs.pkgsCross.ucrt64;
          anarch-re-amd64 = anarch-re-cross pkgs.pkgsCross.gnu64;
          anarch-re-aarch64 = anarch-re-cross pkgs.pkgsCross.aarch64-multiplatform;
          anarch-re = pkgs.callPackage ./package.nix { };
          default = anarch-re;
        }
      );
      devShells = forEachSupportedSystem (
        { pkgs }:
        {
          default = pkgs.mkShell {
            packages = with pkgs; [
              pkg-config
              pdpmake
              sdl3
              gdb
              clang-tools
            ];
            shellHook = ''
              export NIX_HARDENING_ENABLE="";
              export UBSAN_OPTIONS=abort_on_error=1:halt_on_error=1
              export CFLAGS="-std=c99 -O0 -fsanitize=undefined -g3 \
                -Wall -Wextra -Wpedantic -Werror -Wdouble-promotion \
                -Wno-unused-parameter -Wno-missing-field-initializers"
            '';
          };
        }
      );
    };
}
