// SPDX-License-Identifier: CC0-1.0
/**
  A simple header library allowing to decode images encoded using the Quite OK
  Image Format <https://qoiformat.org/>.

  Written in 2025 by Marcin Serwin <marcin@serwin.dev>

  To the extent possible under law, the author(s) have dedicated all copyright
  and related and neighboring rights to this software to the public domain
  worldwide. This software is distributed without any warranty.

  You should have received a copy of the CC0 Public Domain Dedication along with
  this software. If not, see <http://creativecommons.org/publicdomain/zero/1.0/>
*/

#ifndef QOI_DECODER
#define QOI_DECODER

#include <assert.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>

#ifndef MAX_SIZE
#define MAX_SIZE 256
#endif

typedef struct Header {
  uint32_t width;
  uint32_t height;
  uint8_t channels;
  uint8_t colorspace;
} Header;

typedef struct Color {
  uint8_t r;
  uint8_t g;
  uint8_t b;
} Color;

static uint32_t readIntBE(uint8_t buf[4]) {
  return ((uint32_t)buf[0] << 24) | ((uint32_t)buf[1] << 16) |
         ((uint32_t)buf[2] << 8) | ((uint32_t)buf[3] << 0);
}

static Header readHeader(FILE *f) {
  uint8_t buf[14];
  size_t read = fread(buf, sizeof *buf, 14, f);
  assert(read == 14);
  assert(memcmp(buf, "qoif", 4) == 0);

  Header header;
  uint8_t *iter = buf + 4;
  header.width = readIntBE(iter);
  iter += 4;
  header.height = readIntBE(iter);
  iter += 4;
  header.channels = *iter;
  iter++;
  header.colorspace = *iter;
  return header;
}

static int hash(Color c) {
  return (c.r * 3 + c.g * 5 + c.b * 7 + 255 * 11) % 64;
}

static uint8_t fgetu8(FILE *f) {
  int c = fgetc(f);
  assert(c != EOF);
  return (uint8_t)c;
}

static Header readPixels(FILE *f, Color pixels[]) {
  Header h = readHeader(f);
  assert(h.width < MAX_SIZE);
  assert(h.height < MAX_SIZE);
  assert(h.channels == 3);
  assert(h.colorspace == 0);

  Color seen[64] = {0};
  Color p = {0};
  for (uint32_t i = 0; i < h.width * h.height; i++) {
    uint8_t c = fgetu8(f);
    assert(c != 0xFF);
    if (c == 0xFE) {
      p.r = fgetu8(f);
      p.g = fgetu8(f);
      p.b = fgetu8(f);
    } else {
      switch (c >> 6) {
      case 0:
        p = seen[c & 0x3F];
        break;
      case 1:
        p.r += (uint8_t)(((c >> 4) & 3) - 2);
        p.g += (uint8_t)(((c >> 2) & 3) - 2);
        p.b += (uint8_t)(((c >> 0) & 3) - 2);
        break;
      case 2: {
        uint8_t dg = (c & 0x3F) - 32;
        c = fgetu8(f);
        uint8_t dr_dg = (c >> 4) - 8;
        uint8_t db_dg = (c & 0xF) - 8;
        p.r += (uint8_t)(dr_dg + dg);
        p.g += dg;
        p.b += (uint8_t)(db_dg + dg);
        break;
      }
      case 3:
        c &= 0x3f;
        while (c--) {
          pixels[i++] = p;
        }
        break;
      default:
        assert(0);
      }
    }
    seen[hash(p)] = p;
    pixels[i] = p;
  }
  char buf[8];
  size_t read = fread(buf, sizeof *buf, 8, f);
  assert(read == 8);
  assert(memcmp(buf, "\0\0\0\0\0\0\0\1", 8) == 0);
  return h;
}

#endif // guard
