// SPDX-License-Identifier: CC0-1.0
/**
  A tool that transforms images and textures into format used by the Anarch
  game. Based on the `img2array.py` tool by Miloslav Číž.

  Written in 2025 by Marcin Serwin <marcin@serwin.dev>

  To the extent possible under law, the author(s) have dedicated all copyright
  and related and neighboring rights to this software to the public domain
  worldwide. This software is distributed without any warranty.

  You should have received a copy of the CC0 Public Domain Dedication along with
  this software. If not, see <http://creativecommons.org/publicdomain/zero/1.0/>
*/

#include "qoi_decoder.h"
#include <assert.h>
#include <limits.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define MAX_SIZE 256

uint8_t findClosestColor(Color p, const Color *palette, int pallen) {
  assert(pallen <= 256);
  int index = -1;
  int bestDiff = INT_MAX;
  for (int i = 0; i < pallen; i++) {
    Color c = palette[i];
    int diff = abs(p.r - c.r) + abs(p.g - c.g) + abs(p.b - c.b);
    if (diff < bestDiff) {
      index = i;
      bestDiff = diff;
    }
  }
  assert(index != -1);
  return (uint8_t)index;
}

int encodeImage(Header h, const Color *pixels, const Color *palette, int pallen,
                uint8_t *output) {
#define REDUCED_PALETTE_LEN 16
  static int histogram[MAX_SIZE];
  memset(histogram, 0, sizeof(histogram));
  for (uint32_t i = 0; i < h.width * h.height; i++) {
    histogram[findClosestColor(pixels[i], palette, pallen)]++;
  }

  for (int i = 0; i < REDUCED_PALETTE_LEN; i++) {
    int maxVal = 0;
    uint8_t maxIdx = 0;
    for (int j = 0; j < pallen; j++) {
      if (histogram[j] > maxVal) {
        maxVal = histogram[j];
        maxIdx = (uint8_t)j;
      }
    }
    output[i] = maxIdx;
    histogram[maxIdx] = 0;
  }

  Color reducedPalette[REDUCED_PALETTE_LEN];
  for (int i = 0; i < REDUCED_PALETTE_LEN; i++) {
    reducedPalette[i] = palette[output[i]];
  }
  output += REDUCED_PALETTE_LEN;
  bool even = true;
  for (uint32_t j = 0; j < h.width; j++) {
    for (uint32_t i = 0; i < h.height; i++) {
/* This matches the original python script, but is more complicated and less
accurate compared to the version below */
#if 0 
      Color c =
          palette[findClosestColor(pixels[i * h.width + j], palette, pallen)];
#else
      Color c = pixels[i * h.width + j];
#endif
      uint8_t idx = findClosestColor(c, reducedPalette, REDUCED_PALETTE_LEN);
      if (even) {
        *output = (uint8_t)(idx << 4);
      } else {
        *output |= idx;
        output++;
      }
      even = !even;
    }
  }
  return REDUCED_PALETTE_LEN + h.width * h.height / 2;
}

void dumpImage(const char *filename, const char *description, int i,
               Color *palette, int pallen) {
  printf("// %d, %s\n", i, description);
  FILE *f = fopen(filename, "rb");
  assert(f);
  static Color pixels[MAX_SIZE * MAX_SIZE];
  Header h = readPixels(f, pixels);
  fclose(f);

  static uint8_t encoded[MAX_SIZE * MAX_SIZE];
  int len = encodeImage(h, pixels, palette, pallen, encoded);

  for (int i = 0; i < len; i++) {
    printf("%d,", encoded[i]);
  }
  putchar('\n');
}

int main(int argc, char **argv) {
  assert(argc > 1);
  FILE *pf = fopen(argv[1], "rb");
  assert(pf);
  static Color palette[MAX_SIZE * MAX_SIZE];
  Header ph = readPixels(pf, palette);
  int pallen = ph.width * ph.height;
  fclose(pf);

  for (int i = 2; i < argc; i += 2) {
    dumpImage(argv[i], argv[i + 1], (i - 2) / 2, palette, pallen);
  }

  return 0;
}
