// SPDX-License-Identifier: CC0-1.0
/**
  A simple touch controls library for SDL3. It tracks which buttons are pressed
  and renders them using the given renderer.

  Written in 2025 by Marcin Serwin <marcin@serwin.dev>

  To the extent possible under law, the author(s) have dedicated all copyright
  and related and neighboring rights to this software to the public domain
  worldwide. This software is distributed without any warranty.

  You should have received a copy of the CC0 Public Domain Dedication along with
  this software. If not, see <http://creativecommons.org/publicdomain/zero/1.0/>
*/

#ifndef STC_MAIN_H
#define STC_MAIN_H

#include <SDL3/SDL_assert.h>
#include <SDL3/SDL_render.h>

#ifndef STC_PUBLIC_API
#define STC_PUBLIC_API
#endif

#define STC_ARROW_COUNT 4
#define STC_ARROW_MARGIN 2
#define STC_BUTTON_COUNT 3
#define STC_BUTTON_MARGIN 4
#define STC_SAFE_AREA_PADDING 4

#define FINGER_ID_NONE 0

struct STC_TouchData {
  SDL_Texture *arrowTexture;
  SDL_Texture *buttonTextures[STC_BUTTON_COUNT];

  SDL_FPoint dpadCenter;
  SDL_FPoint buttonCenters[STC_BUTTON_COUNT];
  float scale;

  SDL_FingerID pressedBy[STC_ARROW_COUNT + STC_BUTTON_COUNT];
};

enum STC_Button {
  STC_BUTTON_UP,
  STC_BUTTON_RIGHT,
  STC_BUTTON_DOWN,
  STC_BUTTON_LEFT,

  STC_BUTTON_A,
  STC_BUTTON_B,
  STC_BUTTON_C,

  STC_BUTTON_NONE = 0xff
};

#define FIRST_BUTTON STC_BUTTON_A

STC_PUBLIC_API bool STC_init(struct STC_TouchData *t, SDL_Renderer *renderer);
STC_PUBLIC_API bool STC_isPressed(const struct STC_TouchData *t,
                                  enum STC_Button but);
STC_PUBLIC_API void STC_render(const struct STC_TouchData *t,
                               SDL_Renderer *renderer);
STC_PUBLIC_API void STC_handleTouchEvent(struct STC_TouchData *t,
                                         SDL_TouchFingerEvent event);
STC_PUBLIC_API void STC_updateSafeArea(struct STC_TouchData *t,
                                       SDL_Renderer *renderer);
STC_PUBLIC_API void STC_destroy(struct STC_TouchData *t);

#ifdef STC_IMPLEMENTATION

#include "touchcontrols_data.h"

static float sq(float x) { return x * x; }

STC_PUBLIC_API
bool STC_isPressed(const struct STC_TouchData *t, enum STC_Button but) {
  return t->pressedBy[but] != 0;
}

static SDL_Texture *initTexture(SDL_Renderer *renderer, int width, int height,
                                void *pixels, int pitch) {
  SDL_Texture *texture = NULL;
  SDL_Surface *surf = SDL_CreateSurfaceFrom(
      width, height, SDL_PIXELFORMAT_INDEX1LSB, pixels, pitch);
  if (!surf) {
    return NULL;
  }

  SDL_Palette *p = SDL_CreateSurfacePalette(surf);
  if (!p) {
    goto end;
  }
  p->colors[0].a = p->colors[0].r = p->colors[0].g = p->colors[0].b = 0x00;
  p->colors[1].a = p->colors[1].r = p->colors[1].g = p->colors[1].b = 0x4f;

  texture = SDL_CreateTextureFromSurface(renderer, surf);
  if (!texture) {
    goto end;
  }
  SDL_SetTextureScaleMode(texture, SDL_SCALEMODE_NEAREST);
  SDL_SetTextureBlendMode(texture, SDL_BLENDMODE_BLEND_PREMULTIPLIED);

end:
  SDL_DestroySurface(surf);
  return texture;
}

STC_PUBLIC_API
bool STC_init(struct STC_TouchData *t, SDL_Renderer *renderer) {
  t->arrowTexture = initTexture(renderer, ARROW_WIDTH, ARROW_HEIGHT,
                                (void *)ARROW_PIXELS, ARROW_PITCH);
  t->buttonTextures[0] = initTexture(renderer, BUTTON_A_WIDTH, BUTTON_A_HEIGHT,
                                     (void *)BUTTON_A_PIXELS, BUTTON_A_PITCH);
  t->buttonTextures[1] = initTexture(renderer, BUTTON_B_WIDTH, BUTTON_B_HEIGHT,
                                     (void *)BUTTON_B_PIXELS, BUTTON_B_PITCH);
  t->buttonTextures[2] = initTexture(renderer, BUTTON_C_WIDTH, BUTTON_C_HEIGHT,
                                     (void *)BUTTON_C_PIXELS, BUTTON_C_PITCH);
  STC_updateSafeArea(t, renderer);
  return true;
}

STC_PUBLIC_API
void STC_updateSafeArea(struct STC_TouchData *t, SDL_Renderer *renderer) {
  SDL_Rect safeArea;
  SDL_GetRenderSafeArea(renderer, &safeArea);
  float dpadSize = (safeArea.h > safeArea.w ? safeArea.w : safeArea.h) / 2.2;

  t->scale = dpadSize / (2.0f * (ARROW_HEIGHT + STC_ARROW_MARGIN));
  t->dpadCenter.x =
      safeArea.x + (dpadSize / 2.0f + t->scale * STC_SAFE_AREA_PADDING);
  t->dpadCenter.y = safeArea.y + safeArea.h -
                    (dpadSize / 2.0f + t->scale * STC_SAFE_AREA_PADDING);

  t->buttonCenters[2].x =
      safeArea.x + safeArea.w -
      t->scale * (BUTTON_A_WIDTH / 2.0f + STC_SAFE_AREA_PADDING);
  t->buttonCenters[2].y =
      safeArea.y + safeArea.h -
      t->scale * (BUTTON_A_HEIGHT / 2.0f + STC_SAFE_AREA_PADDING);

  t->buttonCenters[1].x =
      t->buttonCenters[2].x - t->scale * (BUTTON_A_WIDTH + STC_BUTTON_MARGIN);
  t->buttonCenters[1].y = t->buttonCenters[2].y;

  t->buttonCenters[0].x = t->buttonCenters[2].x -
                          t->scale * (BUTTON_A_WIDTH + STC_BUTTON_MARGIN) / 2;
  t->buttonCenters[0].y =
      t->buttonCenters[2].y -
      t->scale * (BUTTON_A_WIDTH + STC_BUTTON_MARGIN) * (SDL_sqrtf(3) / 2);
}

STC_PUBLIC_API
void STC_render(const struct STC_TouchData *t, SDL_Renderer *renderer) {
  SDL_FRect dstrect;
  dstrect.x = t->dpadCenter.x - t->scale * ARROW_WIDTH / 2.0f;
  dstrect.y = t->dpadCenter.y - t->scale * (ARROW_HEIGHT + STC_ARROW_MARGIN);
  dstrect.w = t->scale * ARROW_WIDTH;
  dstrect.h = t->scale * ARROW_HEIGHT;

  SDL_FPoint rotationCenter;
  rotationCenter.x = dstrect.w / 2;
  rotationCenter.y = dstrect.h + t->scale * STC_ARROW_MARGIN;

#define activeGray(b) STC_isPressed(t, (enum STC_Button)(b)) ? 0xff : 0x8f

  for (int i = 0; i < STC_ARROW_COUNT; i++) {
    uint8_t gray = activeGray(i);
    SDL_SetTextureColorMod(t->arrowTexture, gray, gray, gray);

    SDL_RenderTextureRotated(renderer, t->arrowTexture, NULL, &dstrect, 90 * i,
                             &rotationCenter, SDL_FLIP_NONE);
  }

  for (int i = 0; i < STC_BUTTON_COUNT; i++) {
    dstrect.x = t->buttonCenters[i].x - t->scale * BUTTON_A_WIDTH / 2.0f;
    dstrect.y = t->buttonCenters[i].y - t->scale * BUTTON_A_HEIGHT / 2.0f;
    dstrect.w = t->scale * BUTTON_A_WIDTH;
    dstrect.h = t->scale * BUTTON_A_HEIGHT;

    uint8_t gray = activeGray(FIRST_BUTTON + i);
    SDL_SetTextureColorMod(t->buttonTextures[i], gray, gray, gray);

    SDL_RenderTexture(renderer, t->buttonTextures[i], NULL, &dstrect);
  }
#undef activeGray
}

static enum STC_Button getPressedButton(const struct STC_TouchData *t, float x,
                                        float y) {
  float dx = x - t->dpadCenter.x;
  float dy = y - t->dpadCenter.y;
  if (sq(dx) + sq(dy) <= sq(t->scale * (STC_ARROW_MARGIN + ARROW_HEIGHT))) {
    if (SDL_fabsf(dy) < SDL_fabsf(dx)) {
      return dx > 0 ? STC_BUTTON_RIGHT : STC_BUTTON_LEFT;
    } else {
      return dy > 0 ? STC_BUTTON_DOWN : STC_BUTTON_UP;
    }
  }

  for (int i = 0; i < STC_BUTTON_COUNT; i++) {
    if (sq(x - t->buttonCenters[i].x) + sq(y - t->buttonCenters[i].y) <=
        sq(t->scale * BUTTON_A_WIDTH / 2.0f)) {
      return (enum STC_Button)(FIRST_BUTTON + i);
    }
  }
  return STC_BUTTON_NONE;
}

STC_PUBLIC_API
void STC_handleTouchEvent(struct STC_TouchData *t, SDL_TouchFingerEvent event) {
  SDL_assert(event.fingerID != FINGER_ID_NONE);

  if (event.type == SDL_EVENT_FINGER_UP ||
      event.type == SDL_EVENT_FINGER_MOTION) {
    for (int i = 0; i < STC_ARROW_COUNT + STC_BUTTON_COUNT; i++) {
      if (event.fingerID == t->pressedBy[i]) {
        t->pressedBy[i] = FINGER_ID_NONE;
      }
    }
  }

  enum STC_Button curr =
      getPressedButton(t, event.x + event.dx, event.y + event.dy);
  if ((event.type == SDL_EVENT_FINGER_DOWN ||
       event.type == SDL_EVENT_FINGER_MOTION) &&
      curr != STC_BUTTON_NONE) {
    t->pressedBy[curr] = event.fingerID;
  }
}

STC_PUBLIC_API void STC_destroy(struct STC_TouchData *t) {
  SDL_DestroyTexture(t->arrowTexture);
  for (int i = 0; i < STC_BUTTON_COUNT; i++) {
    SDL_DestroyTexture(t->buttonTextures[i]);
  }
}

#endif // implementation
#endif // STC_MAIN_H
