import 'package:flutter/material.dart';
import 'package:labplus_for_gitlab/models/models.dart';
import 'package:labplus_for_gitlab/shared/shared.dart';
import 'package:intl/intl.dart';
import 'package:timeago/timeago.dart' as timeago;

import 'package:get/get.dart';
import 'package:grouped_list/grouped_list.dart';

import 'commits.dart';

class CommitsScreen extends GetView<CommitsController> {
  const CommitsScreen({super.key});

  @override
  Widget build(BuildContext context) {
    return Obx(() => _buildWidget());
  }

  Widget _buildWidget() {
    // ignore: unused_local_variable
    var l = controller.commits.length;
    return Scaffold(
      appBar: AppBar(
        title: Text("Commits".tr),
      ),
      body: _buildList(),
    );
  }

  Widget _buildList() {
    return RefreshIndicator(
      onRefresh: () => controller.listCommits(),
      child: HttpFutureBuilder(
        state: controller.state.value,
        child: Scrollbar(
          controller: controller.scrollControllerCommits,
          child: GroupedListView<Commit, DateTime>(
              // ignore: invalid_use_of_protected_member
              elements: controller.commits.value,
              controller: controller.scrollControllerCommits,
              physics: const AlwaysScrollableScrollPhysics(),
              order: GroupedListOrder.DESC,
              groupBy: (Commit element) => DateTime(
                    element.createdAt!.year,
                    element.createdAt!.month,
                    element.createdAt!.day,
                  ),
              groupSeparatorBuilder: (DateTime element) =>
                  _buildGroupSeparator(element),
              itemComparator: (element1, element2) =>
                  element1.createdAt!.compareTo(element2.createdAt!),
              itemBuilder: (context, element) {
                return _buildListItem(element, context);
              }),
        ),
      ),
    );
  }

  Widget _buildListItem(Commit item, BuildContext context) {
    // ignore:avoid_init_to_null
    var statusIcon = null;

    if (item.lastPipeline != null) {
      statusIcon = switch (item.status) {
        "success" => const Icon(
          Icons.check,
          color: Colors.green,
        ),
        "failed" => const Icon(
          Icons.close,
          color: Colors.red,
        ),
        "created" => const Icon(
          Icons.schedule_outlined,
          color: Colors.yellow,
        ),
        "waiting_for_resource" => const Icon(
          Icons.schedule_outlined,
          color: Colors.yellow,
        ),
        "preparing" => const Icon(
          Icons.schedule_outlined,
          color: Colors.yellow,
        ),
        "pending" => const Icon(
          Icons.schedule_outlined,
          color: Colors.yellow,
        ),
        "scheduled" => const Icon(
          Icons.schedule_outlined,
          color: Colors.yellow,
        ),
        "skipped" => const Icon(
          Icons.remove,
          color: Colors.grey,
        ),
        "canceled" => const Icon(
          Icons.remove,
          color: Colors.grey,
        ),
        "running" => const Icon(
          Icons.cached,
          color: Colors.blue,
        ),
        "manual" => const Icon(
          Icons.cached,
          color: Colors.blue,
        ),
        null => const Icon(
          Icons.question_mark,
        ),
        String() => const Icon(
          Icons.question_mark,
        ),
      };
    }

    var statusString = switch (item.status) {
      "success" => "Success",
      "failed" => "Failed",
      "created" => "Created",
      "waiting_for_resource" => "Waiting for Resource",
      "preparing" => "Preparing",
      "pending" => "Pending",
      "scheduled" => "Scheduled",
      "skipped" => "Skipped",
      "canceled" => "Canceled",
      "running" => "Running",
      "manual" => "Manual",
      null => "",
      String() => throw UnimplementedError(),
    };

    return Column(
      children: [
        ListTile(
          // contentPadding: CommonConstants.contentPaddingLitTileLarge,
          title: Text(
            item.title!,
            style:
                const TextStyle(fontWeight: CommonConstants.fontWeightListTile),
          ),
          trailing: const Icon(Icons.keyboard_arrow_right),
          subtitle: Column(
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              const SizedBox(height: 5),
              // Text(item.shortId!),
              // const SizedBox(height: 5),
              Text.rich(
                TextSpan(
                  children: [
                    TextSpan(
                        text: "${item.authorName!} ",
                        style: const TextStyle(
                            fontWeight: CommonConstants.fontWeightListTile)),
                    TextSpan(
                        text: "authored ${timeago.format(item.createdAt!)}",
                        style: const TextStyle(fontSize: 14)),
                  ],
                ),
              ),
              Row(children: [
                if(statusIcon != null)
                  statusIcon,
                const SizedBox(width: 10.0,),
                if(statusIcon != null)
                  Text(statusString)
              ],)
              
            ],
          ),
          onTap: () {
            controller.onCommitSelected(item);
          },
        ),
        const Divider(),
      ],
    );
  }

  Widget _buildGroupSeparator(DateTime element) {
    final String formatted = DateFormat('dd MMMM, yyyy').format(element);

    return Column(
      children: [
        ListTile(
            title: Text(
          formatted.toString(),
          style: const TextStyle(fontSize: 18, fontWeight: FontWeight.bold),
        )),
        const Divider(),
      ],
    );
  }
}
