package dev.dimension.flare.common

import dev.dimension.flare.common.macos.MacosBridge
import dev.dimension.flare.common.windows.WindowsBridge
import dev.dimension.flare.ui.model.UiTimeline
import dev.dimension.flare.ui.model.takeSuccess
import dev.dimension.flare.ui.presenter.status.StatusPresenter
import dev.dimension.flare.ui.route.Route
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.distinctUntilChanged
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.mapNotNull
import kotlinx.coroutines.launch
import org.apache.commons.lang3.SystemUtils

internal class NativeWindowBridge(
    private val scope: CoroutineScope,
    private val windowsBridge: WindowsBridge,
) {
    fun openImageImageViewer(url: String) {
        if (SystemUtils.IS_OS_MAC_OSX) {
            MacosBridge.openImageViewer(url)
        } else if (SystemUtils.IS_OS_WINDOWS) {
            windowsBridge.openImageViewer(url)
        }
    }

    fun openStatusImageViewer(route: Route.StatusMedia) {
        scope.launch {
            val status =
                StatusPresenter(
                    accountType = route.accountType,
                    statusKey = route.statusKey,
                ).moleculeFlow
                    .map {
                        it.status.takeSuccess()?.content as? UiTimeline.ItemContent.Status
                    }.mapNotNull {
                        it
                    }.distinctUntilChanged()
                    .firstOrNull()
            if (status != null) {
                val medias = status.images
                val statusKey = status.statusKey.toString()
                val userHandle = status.user?.handle ?: "unknown"
                if (SystemUtils.IS_OS_MAC_OSX) {
                    MacosBridge.openStatusImageViewer(
                        data = medias,
                        selectedIndex = route.index,
                        statusKey = statusKey,
                        userHandle = userHandle,
                    )
                } else if (SystemUtils.IS_OS_WINDOWS) {
                    windowsBridge.openStatusImageViewer(
                        data = medias,
                        selectedIndex = route.index,
                        statusKey = statusKey,
                        userHandle = userHandle,
                    )
                }
            }
        }
    }
}
