package dev.dimension.flare.ui.component.status

import androidx.compose.animation.AnimatedContent
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.aspectRatio
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.widthIn
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.alpha
import androidx.compose.ui.draw.blur
import androidx.compose.ui.draw.clip
import androidx.compose.ui.draw.clipToBounds
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.input.pointer.PointerIcon
import androidx.compose.ui.input.pointer.pointerHoverIcon
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.platform.LocalUriHandler
import androidx.compose.ui.unit.dp
import androidx.compose.ui.util.fastForEach
import compose.icons.FontAwesomeIcons
import compose.icons.fontawesomeicons.Solid
import compose.icons.fontawesomeicons.solid.CirclePlay
import compose.icons.fontawesomeicons.solid.EyeSlash
import dev.dimension.flare.common.SystemUtils
import dev.dimension.flare.compose.ui.Res
import dev.dimension.flare.compose.ui.status_sensitive_media
import dev.dimension.flare.ui.component.AdaptiveGrid
import dev.dimension.flare.ui.component.AudioPlayer
import dev.dimension.flare.ui.component.ComponentAppearance
import dev.dimension.flare.ui.component.FAIcon
import dev.dimension.flare.ui.component.LocalComponentAppearance
import dev.dimension.flare.ui.component.NetworkImage
import dev.dimension.flare.ui.component.platform.LocalWifiState
import dev.dimension.flare.ui.component.platform.PlatformCircularProgressIndicator
import dev.dimension.flare.ui.component.platform.PlatformFlyoutContainer
import dev.dimension.flare.ui.component.platform.PlatformIconButton
import dev.dimension.flare.ui.component.platform.PlatformText
import dev.dimension.flare.ui.component.platform.PlatformVideoPlayer
import dev.dimension.flare.ui.humanizer.humanize
import dev.dimension.flare.ui.model.UiMedia
import dev.dimension.flare.ui.route.DeeplinkRoute
import dev.dimension.flare.ui.route.toUri
import dev.dimension.flare.ui.theme.PlatformTheme
import kotlinx.collections.immutable.ImmutableList
import org.jetbrains.compose.resources.stringResource
import kotlin.time.Duration.Companion.milliseconds

@Composable
internal fun StatusMediaComponent(
    data: ImmutableList<UiMedia>,
    onMediaClick: (UiMedia) -> Unit,
    sensitive: Boolean,
    modifier: Modifier = Modifier,
) {
    val uriHandler = LocalUriHandler.current
    val appearanceSettings = LocalComponentAppearance.current
    var hideSensitive by remember(appearanceSettings.showSensitiveContent) {
        mutableStateOf(sensitive && !appearanceSettings.showSensitiveContent)
    }
    val showSensitiveButton = sensitive && !appearanceSettings.showSensitiveContent
    Box(
        modifier =
            modifier
                .clip(PlatformTheme.shapes.medium),
    ) {
        AdaptiveGrid(
            content = {
                data.fastForEach { media ->
                    Box {
                        CompositionLocalProvider(
                            LocalComponentAppearance provides
                                appearanceSettings
                                    .copy(
                                        videoAutoplay =
                                            if (hideSensitive) {
                                                ComponentAppearance.VideoAutoplay.NEVER
                                            } else {
                                                appearanceSettings.videoAutoplay
                                            },
                                    ),
                        ) {
                            MediaItem(
                                media = media,
                                modifier =
                                    Modifier
                                        .clipToBounds()
//                                .sharedElement(
//                                    rememberSharedContentState(
//                                        when (media) {
//                                            is UiMedia.Image -> media.previewUrl
//                                            is UiMedia.Video -> media.thumbnailUrl
//                                            is UiMedia.Audio -> media.previewUrl ?: media.url
//                                            is UiMedia.Gif -> media.previewUrl
//                                        },
//                                    ),
//                                    animatedVisibilityScope = this@AnimatedVisibilityScope,
//                                )
                                        .pointerHoverIcon(PointerIcon.Hand)
                                        .clickable {
                                            onMediaClick(media)
                                        },
                                keepAspectRatio = data.size == 1 && appearanceSettings.expandMediaSize,
                            )
                        }
                        if (!media.description.isNullOrEmpty()) {
                            PlatformFlyoutContainer(
                                modifier =
                                    Modifier
                                        .align(Alignment.BottomEnd),
                                content = { requestShowFlyout ->
                                    PlatformText(
                                        text = "ALT",
                                        modifier =
                                            Modifier
                                                .pointerHoverIcon(PointerIcon.Hand)
                                                .padding(16.dp)
                                                .background(
                                                    color = Color.Black.copy(alpha = 0.75f),
                                                    shape = PlatformTheme.shapes.medium,
                                                ).padding(
                                                    horizontal = 8.dp,
                                                    vertical = 2.dp,
                                                ).clickable {
                                                    if (!requestShowFlyout.invoke()) {
                                                        media.description?.let {
                                                            uriHandler.openUri(DeeplinkRoute.Status.AltText(it).toUri())
                                                        }
                                                    }
                                                },
                                        color = Color.White,
                                    )
                                },
                                flyout = {
                                    PlatformText(
                                        text = media.description ?: "",
                                        modifier =
                                            Modifier
                                                .padding(8.dp)
                                                .widthIn(max = 240.dp),
                                    )
                                },
                            )
                        }
                    }
                }
            },
            modifier =
                Modifier
                    .clip(PlatformTheme.shapes.medium)
                    .let {
//                        if (hideSensitive) {
//                            it.blur(32.dp)
//                        } else {
//                            it
//                        }
                        if (hideSensitive && SystemUtils.isBlurSupported) {
                            it.blur(32.dp)
                        } else {
                            it
                        }
                    },
            expandedSize = appearanceSettings.expandMediaSize,
        )
        if (showSensitiveButton) {
            Box(
                modifier =
                    Modifier
                        .matchParentSize()
                        .let {
                            if (hideSensitive && !SystemUtils.isBlurSupported) {
                                it.background(PlatformTheme.colorScheme.outline)
                            } else {
                                it
                            }
                        }.let {
                            if (hideSensitive) {
                                it.clickable {
                                    hideSensitive = false
                                }
                            } else {
                                it
                            }
                        }.padding(16.dp),
            ) {
                AnimatedContent(
                    hideSensitive,
                    modifier =
                        Modifier
                            .matchParentSize(),
                    label = "StatusMediaComponent",
                ) {
                    Box {
                        if (it) {
                            Box(
                                modifier =
                                    Modifier
                                        .align(Alignment.Center)
                                        .clip(PlatformTheme.shapes.medium)
                                        .background(PlatformTheme.colorScheme.card)
                                        .padding(16.dp),
                            ) {
                                PlatformText(
                                    text = stringResource(Res.string.status_sensitive_media),
                                )
                            }
                        } else {
                            PlatformIconButton(
                                onClick = {
                                    hideSensitive = true
                                },
                                modifier =
                                    Modifier
                                        .align(Alignment.TopStart)
                                        .alpha(0.5f)
                                        .clip(PlatformTheme.shapes.medium)
                                        .background(PlatformTheme.colorScheme.card),
                            ) {
                                FAIcon(
                                    imageVector = FontAwesomeIcons.Solid.EyeSlash,
                                    contentDescription = null,
                                )
                            }
                        }
                    }
                }
            }
        }
    }
}

@Composable
public fun MediaItem(
    media: UiMedia,
    modifier: Modifier = Modifier,
    keepAspectRatio: Boolean = true,
    showCountdown: Boolean = true,
    contentScale: ContentScale = ContentScale.Crop,
) {
    val appearanceSettings = LocalComponentAppearance.current
    when (media) {
        is UiMedia.Image -> {
            NetworkImage(
                model = media.previewUrl,
                contentDescription = media.description,
                contentScale = contentScale,
                customHeaders = media.customHeaders,
                modifier =
                    modifier
                        .fillMaxWidth()
                        .let {
                            if (keepAspectRatio) {
                                it.aspectRatio(
                                    media.aspectRatio,
                                    matchHeightConstraintsFirst = media.aspectRatio > 1f,
                                )
                            } else {
                                it
                            }
                        },
            )
        }

        is UiMedia.Video -> {
            val wifiState = LocalWifiState.current
            val shouldPlay =
                remember(appearanceSettings.videoAutoplay, wifiState) {
                    appearanceSettings.videoAutoplay == ComponentAppearance.VideoAutoplay.ALWAYS ||
                        (appearanceSettings.videoAutoplay == ComponentAppearance.VideoAutoplay.WIFI && wifiState)
                }
            if (shouldPlay) {
                PlatformVideoPlayer(
                    contentScale = contentScale,
                    uri = media.url,
                    muted = true,
                    previewUri = media.thumbnailUrl,
                    contentDescription = media.description,
                    modifier =
                        modifier
                            .fillMaxWidth()
                            .let {
                                if (keepAspectRatio) {
                                    it.aspectRatio(
                                        media.aspectRatio,
                                        matchHeightConstraintsFirst = media.aspectRatio > 1f,
                                    )
                                } else {
                                    it
                                }
                            },
                    loadingPlaceholder = {
                        Box(
                            modifier =
                                Modifier
                                    .fillMaxWidth(),
                            contentAlignment = Alignment.Center,
                        ) {
                            NetworkImage(
                                contentScale = contentScale,
                                model = media.thumbnailUrl,
                                contentDescription = media.description,
                                modifier =
                                    Modifier
                                        .fillMaxWidth(),
                            )
                        }
                        Box(
                            modifier =
                                Modifier
                                    .padding(16.dp)
                                    .background(
                                        Color.Black.copy(alpha = 0.5f),
                                        shape = PlatformTheme.shapes.medium,
                                    ).padding(horizontal = 8.dp, vertical = 4.dp)
                                    .align(Alignment.BottomStart),
                            contentAlignment = Alignment.Center,
                        ) {
                            PlatformCircularProgressIndicator(
                                modifier =
                                    Modifier
                                        .align(Alignment.BottomStart)
                                        .size(16.dp),
                                color = Color.White,
                            )
                        }
                    },
                    errorContent = {
                        Box(
                            modifier =
                                Modifier
                                    .fillMaxWidth(),
                            contentAlignment = Alignment.Center,
                        ) {
                            NetworkImage(
                                contentScale = contentScale,
                                model = media.thumbnailUrl,
                                contentDescription = media.description,
                                modifier =
                                    Modifier
                                        .fillMaxWidth(),
                            )
                        }
                        Box(
                            modifier =
                                Modifier
                                    .padding(16.dp)
                                    .background(
                                        Color.Black.copy(alpha = 0.5f),
                                        shape = PlatformTheme.shapes.medium,
                                    ).padding(horizontal = 8.dp, vertical = 4.dp)
                                    .align(Alignment.BottomStart),
                            contentAlignment = Alignment.Center,
                        ) {
                            FAIcon(
                                FontAwesomeIcons.Solid.CirclePlay,
                                contentDescription = null,
                                modifier =
                                    Modifier
                                        .size(16.dp),
                                tint = Color.White,
                            )
                        }
                    },
                    remainingTimeContent =
                        if (showCountdown) {
                            {
                                Box(
                                    modifier =
                                        Modifier
                                            .padding(16.dp)
                                            .background(
                                                Color.Black.copy(alpha = 0.5f),
                                                shape = PlatformTheme.shapes.medium,
                                            ).padding(horizontal = 8.dp, vertical = 4.dp)
                                            .align(Alignment.BottomStart),
                                    contentAlignment = Alignment.Center,
                                ) {
                                    PlatformText(
                                        text =
                                            remember(it) {
                                                it.milliseconds.humanize()
                                            },
                                        color = Color.White,
                                        style = PlatformTheme.typography.caption,
                                    )
                                }
                            }
                        } else {
                            null
                        },
                )
            } else {
                Box(
                    modifier = modifier,
                ) {
                    NetworkImage(
                        contentScale = contentScale,
                        model = media.thumbnailUrl,
                        customHeaders = media.customHeaders,
                        contentDescription = media.description,
                        modifier =
                            Modifier
                                .fillMaxWidth()
                                .let {
                                    if (keepAspectRatio) {
                                        it.aspectRatio(
                                            media.aspectRatio,
                                            matchHeightConstraintsFirst = media.aspectRatio > 1f,
                                        )
                                    } else {
                                        it
                                    }
                                },
                    )
                    Box(
                        modifier =
                            Modifier
                                .padding(16.dp)
                                .background(
                                    Color.Black.copy(alpha = 0.5f),
                                    shape = PlatformTheme.shapes.medium,
                                ).padding(horizontal = 8.dp, vertical = 4.dp)
                                .align(Alignment.BottomStart),
                        contentAlignment = Alignment.Center,
                    ) {
                        FAIcon(
                            FontAwesomeIcons.Solid.CirclePlay,
                            contentDescription = null,
                            modifier =
                                Modifier
                                    .size(16.dp),
                            tint = Color.White,
                        )
                    }
                }
            }
        }

        is UiMedia.Audio -> {
            AudioPlayer(
                uri = media.url,
                previewUri = media.previewUrl,
                contentDescription = media.description,
                modifier = modifier,
            )
        }

        is UiMedia.Gif ->
            NetworkImage(
                model = media.url,
                contentDescription = media.description,
                contentScale = contentScale,
                customHeaders = media.customHeaders,
                modifier =
                    modifier
                        .fillMaxWidth()
                        .let {
                            if (keepAspectRatio) {
                                it.aspectRatio(
                                    media.aspectRatio,
                                    matchHeightConstraintsFirst = media.aspectRatio > 1f,
                                )
                            } else {
                                it
                            }
                        },
            )
    }
}
