package dev.dimension.flare.ui.component.status

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.LocalIndication
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyRow
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.text.selection.SelectionContainer
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.draw.clipToBounds
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.vector.ImageVector
import androidx.compose.ui.hapticfeedback.HapticFeedbackType
import androidx.compose.ui.input.pointer.PointerIcon
import androidx.compose.ui.input.pointer.pointerHoverIcon
import androidx.compose.ui.platform.LocalHapticFeedback
import androidx.compose.ui.platform.LocalUriHandler
import androidx.compose.ui.platform.UriHandler
import androidx.compose.ui.text.intl.Locale
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.compose.ui.util.fastForEach
import compose.icons.FontAwesomeIcons
import compose.icons.fontawesomeicons.Brands
import compose.icons.fontawesomeicons.Regular
import compose.icons.fontawesomeicons.Solid
import compose.icons.fontawesomeicons.brands.Bluesky
import compose.icons.fontawesomeicons.brands.Mastodon
import compose.icons.fontawesomeicons.brands.Weibo
import compose.icons.fontawesomeicons.brands.XTwitter
import compose.icons.fontawesomeicons.regular.Bookmark
import compose.icons.fontawesomeicons.regular.CommentDots
import compose.icons.fontawesomeicons.regular.Heart
import compose.icons.fontawesomeicons.solid.At
import compose.icons.fontawesomeicons.solid.Bookmark
import compose.icons.fontawesomeicons.solid.CircleInfo
import compose.icons.fontawesomeicons.solid.Ellipsis
import compose.icons.fontawesomeicons.solid.EllipsisVertical
import compose.icons.fontawesomeicons.solid.Globe
import compose.icons.fontawesomeicons.solid.Heart
import compose.icons.fontawesomeicons.solid.Image
import compose.icons.fontawesomeicons.solid.List
import compose.icons.fontawesomeicons.solid.Lock
import compose.icons.fontawesomeicons.solid.LockOpen
import compose.icons.fontawesomeicons.solid.Message
import compose.icons.fontawesomeicons.solid.Minus
import compose.icons.fontawesomeicons.solid.Plus
import compose.icons.fontawesomeicons.solid.QuoteLeft
import compose.icons.fontawesomeicons.solid.Reply
import compose.icons.fontawesomeicons.solid.Retweet
import compose.icons.fontawesomeicons.solid.ShareNodes
import compose.icons.fontawesomeicons.solid.Trash
import compose.icons.fontawesomeicons.solid.UserSlash
import compose.icons.fontawesomeicons.solid.VolumeXmark
import dev.dimension.flare.compose.ui.Res
import dev.dimension.flare.compose.ui.bookmark_add
import dev.dimension.flare.compose.ui.bookmark_remove
import dev.dimension.flare.compose.ui.comment
import dev.dimension.flare.compose.ui.delete
import dev.dimension.flare.compose.ui.fx_share
import dev.dimension.flare.compose.ui.hide_media
import dev.dimension.flare.compose.ui.like
import dev.dimension.flare.compose.ui.mastodon_item_show_less
import dev.dimension.flare.compose.ui.mastodon_item_show_more
import dev.dimension.flare.compose.ui.mastodon_visibility_direct
import dev.dimension.flare.compose.ui.mastodon_visibility_private
import dev.dimension.flare.compose.ui.mastodon_visibility_public
import dev.dimension.flare.compose.ui.mastodon_visibility_unlisted
import dev.dimension.flare.compose.ui.more
import dev.dimension.flare.compose.ui.poll_expired
import dev.dimension.flare.compose.ui.poll_expired_at
import dev.dimension.flare.compose.ui.post_show_full_text
import dev.dimension.flare.compose.ui.quote
import dev.dimension.flare.compose.ui.reaction_add
import dev.dimension.flare.compose.ui.reaction_remove
import dev.dimension.flare.compose.ui.reply
import dev.dimension.flare.compose.ui.reply_to
import dev.dimension.flare.compose.ui.report
import dev.dimension.flare.compose.ui.retweet
import dev.dimension.flare.compose.ui.retweet_remove
import dev.dimension.flare.compose.ui.share
import dev.dimension.flare.compose.ui.show_media
import dev.dimension.flare.compose.ui.status_detail_tldr
import dev.dimension.flare.compose.ui.status_detail_translate
import dev.dimension.flare.compose.ui.unlike
import dev.dimension.flare.compose.ui.user_block
import dev.dimension.flare.compose.ui.user_block_with_parameter
import dev.dimension.flare.compose.ui.user_follow_edit_list
import dev.dimension.flare.compose.ui.user_mute
import dev.dimension.flare.compose.ui.user_mute_with_parameter
import dev.dimension.flare.compose.ui.user_send_message
import dev.dimension.flare.compose.ui.user_unblock
import dev.dimension.flare.compose.ui.user_unmute
import dev.dimension.flare.compose.ui.vote
import dev.dimension.flare.data.datasource.microblog.ActionMenu
import dev.dimension.flare.data.model.PostActionStyle
import dev.dimension.flare.model.MicroBlogKey
import dev.dimension.flare.model.PlatformType
import dev.dimension.flare.ui.component.AdaptiveGrid
import dev.dimension.flare.ui.component.AvatarComponent
import dev.dimension.flare.ui.component.DateTimeText
import dev.dimension.flare.ui.component.EmojiImage
import dev.dimension.flare.ui.component.FAIcon
import dev.dimension.flare.ui.component.FlareDividerDefaults
import dev.dimension.flare.ui.component.HorizontalDivider
import dev.dimension.flare.ui.component.LocalComponentAppearance
import dev.dimension.flare.ui.component.RichText
import dev.dimension.flare.ui.component.placeholder
import dev.dimension.flare.ui.component.platform.PlatformCard
import dev.dimension.flare.ui.component.platform.PlatformCheckbox
import dev.dimension.flare.ui.component.platform.PlatformDropdownMenuDivider
import dev.dimension.flare.ui.component.platform.PlatformDropdownMenuItem
import dev.dimension.flare.ui.component.platform.PlatformDropdownMenuScope
import dev.dimension.flare.ui.component.platform.PlatformFilledTonalButton
import dev.dimension.flare.ui.component.platform.PlatformRadioButton
import dev.dimension.flare.ui.component.platform.PlatformText
import dev.dimension.flare.ui.component.platform.PlatformTextButton
import dev.dimension.flare.ui.component.platform.PlatformTextStyle
import dev.dimension.flare.ui.icons.Misskey
import dev.dimension.flare.ui.model.ClickContext
import dev.dimension.flare.ui.model.UiCard
import dev.dimension.flare.ui.model.UiMedia
import dev.dimension.flare.ui.model.UiPoll
import dev.dimension.flare.ui.model.UiTimeline
import dev.dimension.flare.ui.model.collectAsUiState
import dev.dimension.flare.ui.model.onError
import dev.dimension.flare.ui.model.onLoading
import dev.dimension.flare.ui.model.onSuccess
import dev.dimension.flare.ui.render.UiRichText
import dev.dimension.flare.ui.theme.PlatformContentColor
import dev.dimension.flare.ui.theme.PlatformTheme
import kotlinx.collections.immutable.ImmutableList
import kotlinx.collections.immutable.toImmutableList
import moe.tlaster.precompose.molecule.producePresenter
import org.jetbrains.compose.resources.stringResource

@Composable
public fun CommonStatusComponent(
    item: UiTimeline.ItemContent.Status,
    modifier: Modifier = Modifier,
    isDetail: Boolean = false,
    isQuote: Boolean = false,
    showMedia: Boolean = true,
    maxLines: Int? = null,
    showExpandButton: Boolean = true,
) {
    val uriHandler = LocalUriHandler.current
    val appearanceSettings = LocalComponentAppearance.current
    val showAsFullWidth = !appearanceSettings.fullWidthPost && !isQuote && !isDetail
    Row(
        modifier =
            Modifier
                .let {
                    if (isDetail) {
                        it
                    } else {
                        it.clickable {
                            item.onClicked.invoke(
                                ClickContext(
                                    launcher = { url ->
                                        uriHandler.openUri(url)
                                    },
                                ),
                            )
                        }
                    }
                }.then(modifier),
    ) {
        if (showAsFullWidth) {
            item.user?.let {
                AvatarComponent(
                    it.avatar,
                    modifier =
                        Modifier.clickable {
                            it.onClicked.invoke(
                                ClickContext(
                                    launcher = {
                                        uriHandler.openUri(it)
                                    },
                                ),
                            )
                        },
                )
                Spacer(modifier = Modifier.width(8.dp))
            }
        }
        Column {
            item.user?.let { user ->
                val dateContent = @Composable {
                    Row(
                        verticalAlignment = Alignment.CenterVertically,
                        horizontalArrangement = Arrangement.spacedBy(4.dp),
                    ) {
                        when (val content = item.topEndContent) {
                            is UiTimeline.ItemContent.Status.TopEndContent.Visibility -> {
                                StatusVisibilityComponent(
                                    visibility = content.visibility,
                                    modifier =
                                        Modifier
                                            .size(PlatformTheme.typography.caption.fontSize.value.dp),
                                    tint = PlatformTheme.colorScheme.caption,
                                )
                            }

                            null -> Unit
                        }
                        if (appearanceSettings.showPlatformLogo) {
                            val icon =
                                when (item.platformType) {
                                    PlatformType.Mastodon ->
                                        FontAwesomeIcons.Brands.Mastodon
                                    PlatformType.Misskey ->
                                        FontAwesomeIcons.Brands.Misskey
                                    PlatformType.Bluesky ->
                                        FontAwesomeIcons.Brands.Bluesky
                                    PlatformType.xQt ->
                                        FontAwesomeIcons.Brands.XTwitter
                                    PlatformType.VVo ->
                                        FontAwesomeIcons.Brands.Weibo
                                }
                            FAIcon(
                                imageVector = icon,
                                contentDescription = null,
                                modifier =
                                    Modifier
                                        .size(PlatformTheme.typography.caption.fontSize.value.dp),
                                tint = PlatformTheme.colorScheme.caption,
                            )
                        }
                        if (!isDetail) {
                            DateTimeText(
                                item.createdAt,
                                style = PlatformTheme.typography.caption,
                                color = PlatformTheme.colorScheme.caption,
                            )
                        }
                    }
                }
                if (showAsFullWidth) {
                    UserCompat(
                        user = user,
                        onUserClick = {
                            user.onClicked.invoke(
                                ClickContext(
                                    launcher = {
                                        uriHandler.openUri(it)
                                    },
                                ),
                            )
                        },
                        leading = null,
                    ) {
                        dateContent.invoke()
                    }
                } else if (isQuote) {
                    UserCompat(
                        user,
                        onUserClick = {
                            user.onClicked.invoke(
                                ClickContext(
                                    launcher = {
                                        uriHandler.openUri(it)
                                    },
                                ),
                            )
                        },
                    ) {
                        dateContent.invoke()
                    }
                } else {
                    CommonStatusHeaderComponent(
                        data = user,
                        onUserClick = {
                            user.onClicked.invoke(
                                ClickContext(
                                    launcher = {
                                        uriHandler.openUri(it)
                                    },
                                ),
                            )
                        },
                    ) {
                        dateContent.invoke()
                    }
                }
            }
            when (val content = item.aboveTextContent) {
                is UiTimeline.ItemContent.Status.AboveTextContent.ReplyTo -> {
                    Spacer(modifier = Modifier.height(4.dp))
                    StatusReplyComponent(
                        replyHandle = content.handle,
                    )
                }

                null -> Unit
            }
            if (isDetail) {
                SelectionContainer {
                    StatusContentComponent(
                        content = item.content,
                        contentWarning = item.contentWarning,
                        poll = item.poll,
                        maxLines = Int.MAX_VALUE,
                        showExpandButton = showExpandButton,
                    )
                }
            } else {
                StatusContentComponent(
                    content = item.content,
                    contentWarning = item.contentWarning,
                    poll = item.poll,
                    maxLines =
                        maxLines
                            ?: if (item.shouldExpandTextByDefault) {
                                Int.MAX_VALUE
                            } else {
                                appearanceSettings.lineLimit
                            },
                    showExpandButton = showExpandButton,
                )
            }

            if (isDetail && !item.content.isEmpty && appearanceSettings.showTranslateButton) {
                TranslationComponent(
                    statusKey = item.statusKey,
                    contentWarning = item.contentWarning,
                    rawContent = item.content.innerText,
                    content = item.content,
                )
            }

            if (item.images.isNotEmpty() && showMedia) {
                Spacer(modifier = Modifier.height(8.dp))
                StatusMediasComponent(
                    item,
                    onMediaClick = { media ->
                        item.onMediaClicked.invoke(
                            ClickContext(
                                launcher = {
                                    uriHandler.openUri(it)
                                },
                            ),
                            media,
                            item.images.indexOf(media),
                        )
                    },
                )
            }
            item.card?.let { card ->
                if (appearanceSettings.showLinkPreview && item.images.isEmpty() && item.quote.isEmpty()) {
                    Spacer(modifier = Modifier.height(8.dp))
                    StatusCardComponent(
                        card = card,
                        modifier =
                            Modifier
                                .pointerHoverIcon(PointerIcon.Hand)
                                .clickable {
                                    uriHandler.openUri(card.url)
                                }.fillMaxWidth(),
                    )
                }
            }
            if (item.quote.isNotEmpty() && !isQuote) {
                Spacer(modifier = Modifier.height(4.dp))
                StatusQuoteComponent(
                    quotes = item.quote,
                )
            }

            when (val content = item.bottomContent) {
                is UiTimeline.ItemContent.Status.BottomContent.Reaction -> {
                    if (content.emojiReactions.isNotEmpty()) {
                        Spacer(modifier = Modifier.height(4.dp))
                        StatusReactionComponent(
                            data = content,
                        )
                    }
                }

                null -> Unit
            }

            if (isDetail) {
                Spacer(modifier = Modifier.height(8.dp))
                DateTimeText(
                    item.createdAt,
                    style = PlatformTheme.typography.caption,
                    color = PlatformTheme.colorScheme.caption,
                    fullTime = true,
                )
            }
            if (appearanceSettings.postActionStyle != PostActionStyle.Hidden || isDetail) {
                Spacer(modifier = Modifier.height(8.dp))
                if (isDetail) {
                    CompositionLocalProvider(
                        PlatformContentColor provides PlatformTheme.colorScheme.text,
                        PlatformTextStyle provides PlatformTheme.typography.body,
                    ) {
                        StatusActions(
                            item.actions,
                            modifier =
                                Modifier
                                    .fillMaxWidth(),
                        )
                    }
                } else {
                    CompositionLocalProvider(
                        PlatformContentColor provides PlatformTheme.colorScheme.caption,
                        PlatformTextStyle provides PlatformTheme.typography.caption,
                    ) {
                        StatusActions(
                            item.actions,
                            modifier =
                                Modifier
                                    .fillMaxWidth(),
                        )
                    }
                }
            }
        }
    }
}

@Composable
internal fun StatusMediasComponent(
    item: UiTimeline.ItemContent.Status,
    onMediaClick: (UiMedia) -> Unit,
) {
    val appearanceSettings = LocalComponentAppearance.current
    var showMedia by remember { mutableStateOf(false) }
    if (appearanceSettings.showMedia || showMedia) {
        if (!appearanceSettings.showMedia) {
            Row(
                modifier =
                    Modifier
                        .fillMaxWidth()
                        .clickable {
                            showMedia = false
                        },
                verticalAlignment = Alignment.CenterVertically,
            ) {
                FAIcon(
                    imageVector = FontAwesomeIcons.Solid.Image,
                    contentDescription = stringResource(resource = Res.string.hide_media),
                    modifier =
                        Modifier
                            .size(12.dp),
                    tint = PlatformTheme.colorScheme.caption,
                )
                Spacer(modifier = Modifier.width(4.dp))
                PlatformText(
                    text = stringResource(resource = Res.string.hide_media),
                    style = PlatformTheme.typography.caption,
                    color = PlatformTheme.colorScheme.caption,
                )
            }
        }
        StatusMediaComponent(
            data = item.images,
            onMediaClick = onMediaClick,
            sensitive = item.sensitive,
        )
    } else {
        Row(
            modifier =
                Modifier
                    .fillMaxWidth()
                    .clickable {
                        showMedia = true
                    },
            verticalAlignment = Alignment.CenterVertically,
        ) {
            FAIcon(
                imageVector = FontAwesomeIcons.Solid.Image,
                contentDescription = stringResource(resource = Res.string.show_media),
                modifier =
                    Modifier
                        .size(12.dp),
                tint = PlatformTheme.colorScheme.caption,
            )
            Spacer(modifier = Modifier.width(4.dp))
            PlatformText(
                text = stringResource(resource = Res.string.show_media),
                style = PlatformTheme.typography.caption,
                color = PlatformTheme.colorScheme.caption,
            )
        }
    }
}

@Composable
private fun StatusQuoteComponent(
    quotes: ImmutableList<UiTimeline.ItemContent.Status>,
    modifier: Modifier = Modifier,
) {
    Box(
        modifier =
            modifier
                .border(
                    FlareDividerDefaults.thickness,
                    color = FlareDividerDefaults.color,
                    shape = PlatformTheme.shapes.medium,
                ).clip(
                    shape = PlatformTheme.shapes.medium,
                ),
    ) {
        Column {
            CompositionLocalProvider(
                LocalComponentAppearance provides
                    LocalComponentAppearance.current.copy(
                        postActionStyle = PostActionStyle.Hidden,
                    ),
            ) {
                quotes.forEachIndexed { index, quote ->
                    CommonStatusComponent(
                        quote,
                        isQuote = true,
                        modifier =
                            Modifier
                                .padding(8.dp),
                    )
                    if (index != quotes.lastIndex && quotes.size > 1) {
                        HorizontalDivider()
                    }
                }
            }
        }
    }
}

@Composable
private fun StatusReactionComponent(
    data: UiTimeline.ItemContent.Status.BottomContent.Reaction,
    modifier: Modifier = Modifier,
) {
    LazyRow(
        horizontalArrangement = Arrangement.spacedBy(8.dp),
        verticalAlignment = Alignment.CenterVertically,
        modifier = modifier,
    ) {
        items(data.emojiReactions) { reaction ->
            val color =
                if (reaction.me) {
                    PlatformTheme.colorScheme.primaryContainer
                } else {
                    PlatformTheme.colorScheme.cardAlt
                }
            val borderColor =
                if (reaction.me) {
                    PlatformTheme.colorScheme.primary
                } else {
                    Color.Transparent
                }
            PlatformCard(
                shape = RoundedCornerShape(100),
                containerColor = color,
                modifier =
                    Modifier
                        .border(
                            FlareDividerDefaults.thickness,
                            color = borderColor,
                            shape = RoundedCornerShape(100),
                        ),
            ) {
                Row(
                    verticalAlignment = Alignment.CenterVertically,
                    modifier =
                        Modifier
                            .clickable {
                                reaction.onClicked.invoke()
                            }.padding(horizontal = 8.dp, vertical = 4.dp),
                ) {
                    if (reaction.isUnicode) {
                        PlatformText(reaction.name)
                    } else {
                        EmojiImage(
                            uri = reaction.url,
                            modifier = Modifier.height(16.dp),
                        )
                    }
                    Spacer(modifier = Modifier.width(4.dp))
                    PlatformText(
                        text = reaction.count.humanized,
                    )
                }
            }
        }
    }
}

@Composable
private fun TranslationComponent(
    statusKey: MicroBlogKey,
    contentWarning: UiRichText?,
    rawContent: String,
    content: UiRichText,
) {
    val componentAppearance = LocalComponentAppearance.current
    var enabledTranslate by rememberSaveable("translate-$statusKey") {
        mutableStateOf(false)
    }
    var enabledTldr by rememberSaveable("tldr-$statusKey") {
        mutableStateOf(false)
    }
    Row {
        PlatformTextButton(
            onClick = {
                if (!enabledTranslate) {
                    enabledTranslate = true
                }
            },
        ) {
            PlatformText(
                text =
                    stringResource(
                        resource = Res.string.status_detail_translate,
                    ),
            )
        }
        if (componentAppearance.aiConfig.tldr && content.isLongText) {
            PlatformTextButton(
                onClick = {
                    if (!enabledTldr) {
                        enabledTldr = true
                    }
                },
            ) {
                PlatformText(
                    text =
                        stringResource(
                            resource = Res.string.status_detail_tldr,
                        ),
                )
            }
        }
    }
    if (enabledTldr) {
        Spacer(modifier = Modifier.height(4.dp))
        val state by producePresenter(
            "tldr_${contentWarning}_${rawContent}_${Locale.current.language}",
        ) {
            statusTldrPresenter(
                contentWarning = contentWarning,
                content = content,
                targetLanguage = Locale.current.language,
            )
        }
        state
            .onSuccess {
                PlatformText(text = it)
            }.onLoading {
                PlatformText(
                    text = "Lores ipsum dolor sit amet",
                    modifier = Modifier.placeholder(true),
                )
            }.onError {
                PlatformText(text = it.message ?: "Error")
            }
    }
    if (enabledTranslate) {
        Spacer(modifier = Modifier.height(4.dp))
        val state by producePresenter(
            "translate_${contentWarning}_${rawContent}_${Locale.current.language}_${componentAppearance.aiConfig.translation}",
        ) {
            statusTranslatePresenter(
                contentWarning = contentWarning,
                content = content,
                targetLanguage = Locale.current.language,
                useAi = componentAppearance.aiConfig.translation,
            )
        }
        state.contentWarning
            ?.onSuccess {
                PlatformText(text = it)
            }?.onLoading {
                PlatformText(
                    text = "Lores ipsum dolor sit amet",
                    modifier = Modifier.placeholder(true),
                )
            }?.onError {
                PlatformText(text = it.message ?: "Error")
            }
        state.text
            .onSuccess {
                PlatformText(text = it)
            }.onLoading {
                PlatformText(
                    text = "Lores ipsum dolor sit amet",
                    modifier = Modifier.placeholder(true),
                )
            }.onError {
                PlatformText(text = it.message ?: "Error")
            }
    }
}

@Composable
public fun StatusVisibilityComponent(
    visibility: UiTimeline.ItemContent.Status.TopEndContent.Visibility.Type,
    tint: Color = PlatformContentColor.current,
    modifier: Modifier = Modifier,
) {
    when (visibility) {
        UiTimeline.ItemContent.Status.TopEndContent.Visibility.Type.Public ->
            FAIcon(
                imageVector = FontAwesomeIcons.Solid.Globe,
                contentDescription = stringResource(resource = Res.string.mastodon_visibility_public),
                modifier = modifier,
                tint = tint,
            )

        UiTimeline.ItemContent.Status.TopEndContent.Visibility.Type.Home ->
            FAIcon(
                imageVector = FontAwesomeIcons.Solid.LockOpen,
                contentDescription = stringResource(resource = Res.string.mastodon_visibility_unlisted),
                modifier = modifier,
                tint = tint,
            )

        UiTimeline.ItemContent.Status.TopEndContent.Visibility.Type.Followers ->
            FAIcon(
                imageVector = FontAwesomeIcons.Solid.Lock,
                contentDescription = stringResource(resource = Res.string.mastodon_visibility_private),
                modifier = modifier,
                tint = tint,
            )

        UiTimeline.ItemContent.Status.TopEndContent.Visibility.Type.Specified ->
            FAIcon(
                imageVector = FontAwesomeIcons.Solid.At,
                contentDescription = stringResource(resource = Res.string.mastodon_visibility_direct),
                modifier = modifier,
                tint = tint,
            )
    }
}

@Composable
internal fun StatusActions(
    items: ImmutableList<ActionMenu>,
    modifier: Modifier = Modifier,
) {
    val appearanceSettings = LocalComponentAppearance.current
    val haptics = LocalHapticFeedback.current
    val launcher = LocalUriHandler.current
    Row(
        modifier = modifier,
        verticalAlignment = Alignment.Bottom,
        horizontalArrangement =
            when (appearanceSettings.postActionStyle) {
                PostActionStyle.Hidden -> Arrangement.Start
                PostActionStyle.LeftAligned -> Arrangement.Start
                PostActionStyle.RightAligned -> Arrangement.End
                PostActionStyle.Stretch -> Arrangement.SpaceBetween
            },
//        horizontalArrangement = Arrangement.spacedBy(4.dp),
    ) {
        items.forEachIndexed { index, action ->
            if (index == items.lastIndex && appearanceSettings.postActionStyle == PostActionStyle.LeftAligned) {
                Spacer(modifier = Modifier.weight(1f))
            }
            when (action) {
                is ActionMenu.Group -> {
                    StatusActionGroup(
                        icon = action.displayItem.icon?.toImageVector() ?: FontAwesomeIcons.Solid.Ellipsis,
                        number = action.displayItem.count,
                        color = action.displayItem.color?.toComposeColor() ?: PlatformContentColor.current,
                        withTextMinWidth = index != items.lastIndex,
                    ) { closeMenu, isMenuShown ->
                        action.actions.fastForEach { subActions ->
                            when (subActions) {
                                is ActionMenu.Item -> {
                                    StatusActionItemMenu(subActions, closeMenu, launcher)
                                }

                                is ActionMenu.AsyncActionMenuItem -> {
                                    if (isMenuShown) {
                                        val state by subActions.flow.collectAsUiState()
                                        state
                                            .onSuccess {
                                                StatusActionItemMenu(it, closeMenu, launcher)
                                            }.onLoading {
                                                PlatformDropdownMenuItem(
                                                    text = {
                                                        PlatformText(
                                                            text = "Loading",
                                                            modifier =
                                                                Modifier.placeholder(
                                                                    true,
                                                                    color = PlatformTheme.colorScheme.cardAlt,
                                                                ),
                                                        )
                                                    },
                                                    leadingIcon = {
                                                        FAIcon(
                                                            imageVector = FontAwesomeIcons.Solid.Ellipsis,
                                                            contentDescription = "Loading",
                                                            modifier =
                                                                Modifier
                                                                    .size(PlatformTextStyle.current.fontSize.value.dp + 2.dp)
                                                                    .placeholder(
                                                                        true,
                                                                        color = PlatformTheme.colorScheme.cardAlt,
                                                                    ),
                                                        )
                                                    },
                                                    onClick = {
                                                    },
                                                )
                                            }
                                    }
                                }

                                // nested group is not supported
                                is ActionMenu.Group -> Unit
                                ActionMenu.Divider -> PlatformDropdownMenuDivider()
                            }
                        }
                    }
                }

                is ActionMenu.Item -> {
                    StatusActionButton(
                        icon = action.icon?.toImageVector() ?: FontAwesomeIcons.Solid.Ellipsis, // Fallback or handle null
                        number = action.count,
                        color = action.color?.toComposeColor() ?: PlatformContentColor.current,
                        withTextMinWidth = index != items.lastIndex,
                        onClicked = {
                            action.onClicked?.let { onClick ->
                                haptics.performHapticFeedback(HapticFeedbackType.ContextClick)
                                onClick.invoke(
                                    ClickContext(
                                        launcher = {
                                            launcher.openUri(it)
                                        },
                                    ),
                                )
                            }
                        },
                    )
                }

                // async action item is only supported in group
                is ActionMenu.AsyncActionMenuItem -> Unit
                // divider is only supported in group
                ActionMenu.Divider -> Unit
            }
        }
    }
}

@Composable
private fun PlatformDropdownMenuScope.StatusActionItemMenu(
    subActions: ActionMenu.Item,
    closeMenu: () -> Unit,
    launcher: UriHandler,
) {
    val color = subActions.color?.toComposeColor() ?: PlatformContentColor.current
    PlatformDropdownMenuItem(
        leadingIcon = {
            subActions.icon?.let {
                FAIcon(
                    imageVector = it.toImageVector(),
                    contentDescription = null,
                    tint = color,
                )
            }
        },
        text = {
            subActions.text?.let {
                PlatformText(
                    text = it.asString(),
                    color = color,
                )
            }
        },
        onClick = {
            closeMenu.invoke()
            subActions.onClicked?.invoke(
                ClickContext(
                    launcher = {
                        launcher.openUri(it)
                    },
                ),
            )
        },
    )
}

@Composable
private fun ActionMenu.Item.Text.asString(): String =
    when (this) {
        is ActionMenu.Item.Text.Raw -> text
        is ActionMenu.Item.Text.Localized -> {
            val resource =
                when (type) {
                    ActionMenu.Item.Text.Localized.Type.Like -> Res.string.like
                    ActionMenu.Item.Text.Localized.Type.Unlike -> Res.string.unlike
                    ActionMenu.Item.Text.Localized.Type.Retweet -> Res.string.retweet
                    ActionMenu.Item.Text.Localized.Type.Unretweet -> Res.string.retweet_remove
                    ActionMenu.Item.Text.Localized.Type.Reply -> Res.string.reply
                    ActionMenu.Item.Text.Localized.Type.Comment -> Res.string.comment
                    ActionMenu.Item.Text.Localized.Type.Quote -> Res.string.quote
                    ActionMenu.Item.Text.Localized.Type.Bookmark -> Res.string.bookmark_add
                    ActionMenu.Item.Text.Localized.Type.Unbookmark -> Res.string.bookmark_remove
                    ActionMenu.Item.Text.Localized.Type.More -> Res.string.more
                    ActionMenu.Item.Text.Localized.Type.Delete -> Res.string.delete
                    ActionMenu.Item.Text.Localized.Type.Report -> Res.string.report
                    ActionMenu.Item.Text.Localized.Type.React -> Res.string.reaction_add
                    ActionMenu.Item.Text.Localized.Type.UnReact -> Res.string.reaction_remove
                    ActionMenu.Item.Text.Localized.Type.Share -> Res.string.share
                    ActionMenu.Item.Text.Localized.Type.FxShare -> Res.string.fx_share
                    ActionMenu.Item.Text.Localized.Type.EditUserList -> Res.string.user_follow_edit_list
                    ActionMenu.Item.Text.Localized.Type.SendMessage -> Res.string.user_send_message
                    ActionMenu.Item.Text.Localized.Type.Mute -> Res.string.user_mute
                    ActionMenu.Item.Text.Localized.Type.UnMute -> Res.string.user_unmute
                    ActionMenu.Item.Text.Localized.Type.Block -> Res.string.user_block
                    ActionMenu.Item.Text.Localized.Type.UnBlock -> Res.string.user_unblock
                    ActionMenu.Item.Text.Localized.Type.BlockWithHandleParameter -> Res.string.user_block_with_parameter
                    ActionMenu.Item.Text.Localized.Type.MuteWithHandleParameter -> Res.string.user_mute_with_parameter
                }
            stringResource(resource, *parameters.toTypedArray())
        }
    }

@Composable
private fun ActionMenu.Item.Color.toComposeColor(): Color =
    when (this) {
        ActionMenu.Item.Color.Red -> PlatformTheme.colorScheme.error
        ActionMenu.Item.Color.ContentColor -> PlatformContentColor.current
        ActionMenu.Item.Color.PrimaryColor -> PlatformTheme.colorScheme.retweetColor
    }

private fun ActionMenu.Item.Icon.toImageVector(): ImageVector =
    when (this) {
        ActionMenu.Item.Icon.Like -> FontAwesomeIcons.Regular.Heart
        ActionMenu.Item.Icon.Unlike -> FontAwesomeIcons.Solid.Heart
        ActionMenu.Item.Icon.Retweet -> FontAwesomeIcons.Solid.Retweet
        ActionMenu.Item.Icon.Unretweet -> FontAwesomeIcons.Solid.Retweet
        ActionMenu.Item.Icon.Reply -> FontAwesomeIcons.Solid.Reply
        ActionMenu.Item.Icon.Comment -> FontAwesomeIcons.Regular.CommentDots
        ActionMenu.Item.Icon.Quote -> FontAwesomeIcons.Solid.QuoteLeft
        ActionMenu.Item.Icon.Bookmark -> FontAwesomeIcons.Regular.Bookmark
        ActionMenu.Item.Icon.Unbookmark -> FontAwesomeIcons.Solid.Bookmark
        ActionMenu.Item.Icon.More -> FontAwesomeIcons.Solid.Ellipsis
        ActionMenu.Item.Icon.Delete -> FontAwesomeIcons.Solid.Trash
        ActionMenu.Item.Icon.Report -> FontAwesomeIcons.Solid.CircleInfo
        ActionMenu.Item.Icon.React -> FontAwesomeIcons.Solid.Plus
        ActionMenu.Item.Icon.UnReact -> FontAwesomeIcons.Solid.Minus
        ActionMenu.Item.Icon.Share -> FontAwesomeIcons.Solid.ShareNodes
        ActionMenu.Item.Icon.MoreVerticel -> FontAwesomeIcons.Solid.EllipsisVertical
        ActionMenu.Item.Icon.List -> FontAwesomeIcons.Solid.List
        ActionMenu.Item.Icon.ChatMessage -> FontAwesomeIcons.Solid.Message
        ActionMenu.Item.Icon.Mute -> FontAwesomeIcons.Solid.VolumeXmark
        ActionMenu.Item.Icon.UnMute -> FontAwesomeIcons.Solid.VolumeXmark
        ActionMenu.Item.Icon.Block -> FontAwesomeIcons.Solid.UserSlash
        ActionMenu.Item.Icon.UnBlock -> FontAwesomeIcons.Solid.UserSlash
    }

@Composable
private fun StatusReplyComponent(
    replyHandle: String,
    modifier: Modifier = Modifier,
) {
    Row(
        verticalAlignment = Alignment.CenterVertically,
        horizontalArrangement = Arrangement.spacedBy(4.dp),
    ) {
        FAIcon(
            imageVector = FontAwesomeIcons.Solid.Reply,
            contentDescription = stringResource(resource = Res.string.reply_to),
            tint = PlatformTheme.colorScheme.caption,
            modifier =
                Modifier
                    .size(12.dp),
        )
        PlatformText(
            text = stringResource(resource = Res.string.reply_to, replyHandle),
            style = PlatformTheme.typography.caption,
            color = PlatformTheme.colorScheme.caption,
            maxLines = 1,
        )
    }
}

@Composable
private fun StatusContentComponent(
    content: UiRichText,
    contentWarning: UiRichText?,
    poll: UiPoll?,
    maxLines: Int,
    showExpandButton: Boolean,
    modifier: Modifier = Modifier,
) {
    var expanded by rememberSaveable {
        mutableStateOf(false)
    }
    var showSoftExpand by rememberSaveable {
        mutableStateOf(false)
    }
    Column(
        modifier = modifier,
    ) {
        contentWarning?.let {
            if (it.raw.isNotEmpty()) {
                Column(
                    modifier =
                        Modifier
                            .fillMaxWidth()
                            .padding(vertical = 4.dp),
                    verticalArrangement = Arrangement.spacedBy(4.dp),
                ) {
                    RichText(
                        text = it,
                    )
                    PlatformFilledTonalButton(
                        modifier =
                            Modifier
                                .fillMaxWidth(),
                        onClick = {
                            expanded = !expanded
                        },
                    ) {
                        if (expanded) {
                            PlatformText(stringResource(Res.string.mastodon_item_show_less))
                        } else {
                            PlatformText(stringResource(Res.string.mastodon_item_show_more))
                        }
                    }
                }
            }
        }
        AnimatedVisibility(visible = expanded || contentWarning?.raw.isNullOrEmpty()) {
            Column {
                if (!content.isEmpty) {
                    RichText(
                        text = content,
                        modifier = Modifier.fillMaxWidth(),
                        maxLines =
                            if (expanded || maxLines == Int.MAX_VALUE) {
                                Int.MAX_VALUE
                            } else {
                                maxLines
                            },
                        onTextLayout = {
                            showSoftExpand =
                                it.hasVisualOverflow &&
                                !expanded &&
                                maxLines != Int.MAX_VALUE &&
                                showExpandButton
                        },
                    )
                    if (showSoftExpand) {
                        PlatformTextButton(
                            onClick = {
                                expanded = true
                            },
                        ) {
                            PlatformText(
                                stringResource(Res.string.post_show_full_text),
                                color = PlatformTheme.colorScheme.primary,
                            )
                        }
                    }
                }
                poll?.let {
                    Spacer(modifier = Modifier.height(8.dp))
                    StatusPollComponent(
                        poll = it,
                    )
                }
            }
        }
    }
}

@Composable
private fun StatusPollComponent(
    poll: UiPoll,
    modifier: Modifier = Modifier,
) {
    val selectedOptions =
        remember {
            mutableStateListOf(*poll.ownVotes.toTypedArray())
        }
    Column(
        modifier = modifier,
        verticalArrangement = Arrangement.spacedBy(8.dp),
    ) {
        poll.options.forEachIndexed { index, option ->
            PollOption(
                option = option,
                modifier = Modifier.fillMaxWidth(),
                canVote = poll.canVote,
                multiple = poll.multiple,
                selected = selectedOptions.contains(index),
                onClick = {
                    if (poll.multiple) {
                        if (selectedOptions.contains(index)) {
                            selectedOptions.remove(index)
                        } else {
                            selectedOptions.add(index)
                        }
                    } else {
                        selectedOptions.clear()
                        selectedOptions.add(index)
                    }
                },
            )
        }
        if (poll.expired) {
            PlatformText(
                text = stringResource(resource = Res.string.poll_expired),
                modifier =
                    Modifier
                        .align(Alignment.End),
                style = PlatformTheme.typography.caption,
                color = PlatformTheme.colorScheme.caption,
            )
        } else {
            poll.expiredAt?.let { expiredAt ->
                PlatformText(
                    text =
                        stringResource(
                            resource = Res.string.poll_expired_at,
                            expiredAt.full,
                        ),
                    modifier =
                        Modifier
                            .align(Alignment.End),
                    style = PlatformTheme.typography.caption,
                    color = PlatformTheme.colorScheme.caption,
                )
            }
        }
        if (poll.canVote) {
            PlatformFilledTonalButton(
                modifier = Modifier.fillMaxWidth(),
                onClick = {
                    poll.onVote.invoke(selectedOptions.toImmutableList())
                },
            ) {
                PlatformText(
                    text = stringResource(resource = Res.string.vote),
                )
            }
        }
    }
}

@Composable
private fun PollOption(
    canVote: Boolean,
    multiple: Boolean,
    option: UiPoll.Option,
    selected: Boolean,
    onClick: () -> Unit,
    modifier: Modifier = Modifier,
) {
    Box(
        modifier =
            modifier
                .height(IntrinsicSize.Min)
                .border(
                    width = FlareDividerDefaults.thickness,
                    color = FlareDividerDefaults.color,
                    shape = PlatformTheme.shapes.small,
                )
//            .background(
//                color = PlatformTheme.colorScheme.secondaryContainer,
//                shape = PlatformTheme.shapes.small,
//            )
                .clip(
                    shape = PlatformTheme.shapes.small,
                ),
    ) {
        Box(
            modifier =
                Modifier
                    .fillMaxHeight()
                    .fillMaxWidth(option.percentage)
                    .background(
                        color = PlatformTheme.colorScheme.cardAlt,
                        shape = PlatformTheme.shapes.small,
                    ),
        )
        val mutableInteractionSource =
            remember {
                MutableInteractionSource()
            }
        ListComponent(
            modifier =
                Modifier.clickable(
                    onClick = onClick,
                    interactionSource = mutableInteractionSource,
                    indication = LocalIndication.current,
                    enabled = canVote,
                ),
            headlineContent = {
                PlatformText(
                    text = option.title,
                    color = PlatformTheme.colorScheme.caption,
                    modifier =
                        Modifier
                            .padding(8.dp),
                )
            },
            trailingContent = {
                if (canVote || selected) {
                    if (multiple) {
                        PlatformCheckbox(
                            checked = selected,
                            onCheckedChange = {
                                onClick.invoke()
                            },
                            interactionSource = mutableInteractionSource,
                            enabled = canVote,
                        )
                    } else {
                        PlatformRadioButton(
                            selected = selected,
                            onClick = onClick,
                            interactionSource = mutableInteractionSource,
                            enabled = canVote,
                        )
                    }
                } else {
                    // keep the height consist
//                    PlatformRadioButton(
//                        selected = false,
//                        onClick = {},
//                        enabled = false,
//                        modifier = Modifier.alpha(0f),
//                    )
                }
            },
        )
    }
}

@Composable
private fun StatusCardComponent(
    card: UiCard,
    modifier: Modifier = Modifier,
) {
    val appearanceSettings = LocalComponentAppearance.current
    if (appearanceSettings.compatLinkPreview) {
        CompatCard(
            card = card,
            modifier = modifier,
        )
    } else {
        ExpandedCard(
            card = card,
            modifier = modifier,
        )
    }
}

@Composable
private fun ExpandedCard(
    card: UiCard,
    modifier: Modifier = Modifier,
) {
    val appearanceSettings = LocalComponentAppearance.current
    Column(
        modifier =
            Modifier
                .border(
                    FlareDividerDefaults.thickness,
                    color = FlareDividerDefaults.color,
                    shape = PlatformTheme.shapes.medium,
                ).clip(
                    shape = PlatformTheme.shapes.medium,
                ).then(modifier),
    ) {
        card.media?.let {
            AdaptiveGrid(
                content = {
                    MediaItem(
                        media = it,
                        keepAspectRatio = appearanceSettings.expandMediaSize,
                        modifier =
                            Modifier
                                .clipToBounds(),
                    )
                },
                expandedSize = appearanceSettings.expandMediaSize,
                modifier =
                    Modifier
                        .clipToBounds(),
            )
        }
        Column(
            modifier =
                Modifier
                    .padding(8.dp),
        ) {
            PlatformText(
                text = card.title,
                maxLines = 2,
            )
            card.description?.let {
                PlatformText(
                    text = it,
                    style = PlatformTheme.typography.caption,
                    color = PlatformTheme.colorScheme.caption,
                    maxLines = 2,
                )
            }
        }
    }
}

@Composable
private fun CompatCard(
    card: UiCard,
    modifier: Modifier = Modifier,
) {
    Row(
        modifier =
            Modifier
                .border(
                    FlareDividerDefaults.thickness,
                    color = FlareDividerDefaults.color,
                    shape = PlatformTheme.shapes.medium,
                ).clip(
                    shape = PlatformTheme.shapes.medium,
                ).then(modifier),
    ) {
        card.media?.let {
            MediaItem(
                media = it,
                modifier =
                    Modifier
                        .size(72.dp)
                        .clipToBounds(),
            )
        }
        Column(
            modifier =
                Modifier
                    .padding(8.dp),
        ) {
            PlatformText(
                text = card.title,
                maxLines = 1,
                overflow = TextOverflow.Ellipsis,
            )
            card.description?.let {
                PlatformText(
                    text = it,
                    style = PlatformTheme.typography.caption,
                    color = PlatformTheme.colorScheme.caption,
                    maxLines = 2,
                    overflow = TextOverflow.Ellipsis,
                )
            }
        }
    }
}
