import com.google.gms.googleservices.GoogleServicesPlugin
import java.util.Properties

plugins {
    id("com.android.application")
    alias(libs.plugins.kotlin.android)
    alias(libs.plugins.ksp)
    alias(libs.plugins.kotlin.serialization)
    alias(libs.plugins.ktorfit)
    alias(libs.plugins.ktlint)
    alias(libs.plugins.google.services) apply false
    alias(libs.plugins.compose.compiler)
    id("kotlin-parcelize")
    alias(libs.plugins.stability.analyzer)
}


android {
    namespace = "dev.dimension.flare"
    compileSdk = libs.versions.compileSdk.get().toInt()
    val fdroid = rootProject.file("fdroid.properties")
    val fdroidProp = Properties()
    fdroidProp.load(fdroid.inputStream())

    defaultConfig {
        applicationId = "dev.dimension.flare"
        minSdk = libs.versions.minSdk.get().toInt()
        targetSdk = libs.versions.compileSdk.get().toInt()
        versionCode =
            System.getenv("BUILD_NUMBER")?.toIntOrNull() ?: fdroidProp.getProperty("versionCode")
                ?.toIntOrNull() ?: 1
        versionName =
            System.getenv("BUILD_VERSION")?.toString() ?: fdroidProp.getProperty("versionName")
                ?.toString() ?: "0.0.0"

        testInstrumentationRunner = "androidx.test.runner.AndroidJUnitRunner"
        vectorDrawables {
            useSupportLibrary = true
        }

        packaging {
            resources {
                excludes.add("/META-INF/{AL2.0,LGPL2.1}")
                excludes.add("DebugProbesKt.bin")
            }
        }
    }

    val file = rootProject.file("signing.properties")
    val hasSigningProps = file.exists()


    buildTypes {
        debug {
            if (hasSigningProps) {
            }
        }
        release {
            isMinifyEnabled = true
            proguardFiles(
                getDefaultProguardFile("proguard-android-optimize.txt"),
                "proguard-rules.pro",
            )
            if (hasSigningProps) {
            } else {
            }
        }
    }
    compileOptions {
        isCoreLibraryDesugaringEnabled = true
        sourceCompatibility = JavaVersion.toVersion(libs.versions.java.get())
        targetCompatibility = JavaVersion.toVersion(libs.versions.java.get())
    }
    buildFeatures {
        compose = true
        buildConfig = true
    }
//    composeOptions {
//        kotlinCompilerExtensionVersion = libs.versions.compose.compiler.get()
//    }
    packaging {
        resources {
            excludes += "/META-INF/{AL2.0,LGPL2.1}"
        }
    }
    lint {
        disable.add("MissingTranslation")
    }
    androidResources {
        generateLocaleConfig = true
    }
}

dependencies {
    implementation(libs.core.ktx)
    implementation(libs.lifecycle.runtime.ktx)
    implementation(libs.lifecycle.runtime.compose)
    implementation(libs.activity.compose)
    implementation(platform(libs.compose.bom))
    implementation(libs.bundles.compose)
    implementation(libs.bundles.kotlinx)
    implementation(platform(libs.koin.bom))
    implementation(libs.bundles.koin)
    implementation(libs.ktorfit.lib)
    ksp(libs.ktorfit.ksp)
    implementation(libs.bundles.coil3)
    implementation(libs.bundles.coil3.extensions)
    implementation(libs.bundles.ktor)
    implementation(libs.molecule.runtime)
    implementation(libs.ksoup)
    implementation(libs.bundles.accompanist)
    lintChecks(libs.compose.lint.checks)
    implementation(libs.composeIcons.fontAwesome)
    implementation(libs.datastore)
    implementation(libs.kotlinx.serialization.protobuf)
    implementation(libs.bundles.media3)
    implementation(libs.swiper)
    implementation(libs.reorderable)
    implementation(libs.androidx.window)
    coreLibraryDesugaring(libs.desugar.jdk.libs)
    implementation(libs.compose.webview)
    implementation(projects.shared)
    implementation(projects.composeUi)
    implementation(libs.androidx.splash)
    implementation(libs.materialKolor)
    implementation(libs.colorpicker.compose)
    implementation(libs.material.motion.compose)
    implementation(libs.nestedScrollView)
    implementation(libs.precompose.molecule)
    implementation(libs.webkit)
    implementation(libs.bundles.navigation3)
    implementation(libs.richtext.ui.material3)
    implementation(libs.androidx.browser)


    testImplementation(libs.junit)
    androidTestImplementation(libs.androidx.test.ext.junit)
    androidTestImplementation(libs.espresso.core)
    androidTestImplementation(platform(libs.compose.bom))
    androidTestImplementation(libs.ui.test.junit4)
    debugImplementation(libs.ui.tooling)
    debugImplementation(libs.ui.test.manifest)
}

ktlint {
    version.set(libs.versions.ktlint)
    filter {
        exclude { element -> element.file.path.contains("build", ignoreCase = true) }
    }
}

if (project.file("google-services.json").exists()) {
    afterEvaluate {
        val uploadCrashlyticsMappingFileRelease by tasks
        val processDebugGoogleServices by tasks
        uploadCrashlyticsMappingFileRelease.dependsOn(processDebugGoogleServices)
    }
}


abstract class GenerateDeepLinkManifestTask : DefaultTask() {
    @get:InputFile
    @get:PathSensitive(PathSensitivity.RELATIVE)
    abstract val hostsFile: RegularFileProperty
    @get:OutputFile
    abstract val manifest: RegularFileProperty

    @TaskAction
    fun run() {
        val hosts = hostsFile.get().asFile.readLines()
            .map { it.trim() }
            .filter { it.isNotEmpty() && !it.startsWith("#") }
            .distinct()
        val dataTags = hosts.joinToString("\n") { host ->
            """<data android:host="$host" />"""
        }

        manifest.get().asFile.writeText(
            """
<?xml version="1.0" encoding="utf-8"?>
<manifest xmlns:android="http://schemas.android.com/apk/res/android">
  <application>
    <activity android:name="dev.dimension.flare.MainActivity" android:exported="true">
      <intent-filter android:autoVerify="false">
        <action android:name="android.intent.action.VIEW"/>
        <category android:name="android.intent.category.DEFAULT"/>
        <category android:name="android.intent.category.BROWSABLE"/>
         <data android:scheme="https" />
         $dataTags
      </intent-filter>
    </activity>
  </application>
</manifest>
      """.trimIndent()
        )
    }
}

extensions.getByType(com.android.build.api.variant.AndroidComponentsExtension::class.java)
    .onVariants { variant: com.android.build.api.variant.Variant ->
        val t = tasks.register(
            "generate${variant.name.replaceFirstChar { it.uppercase() }}DeepLinkManifest",
            GenerateDeepLinkManifestTask::class.java
        ) {
            hostsFile = project.layout.projectDirectory.file("deeplink.txt")
        }

        variant.sources.manifests.addGeneratedManifestFile(
            t,
            GenerateDeepLinkManifestTask::manifest
        )
    }

kotlin {
    sourceSets {
        all {
            languageSettings {
                optIn("androidx.compose.material3.ExperimentalMaterial3ExpressiveApi")
            }
        }
    }
}

ktorfit {
    compilerPluginVersion.set("2.3.3")
}