import Cocoa
import Foundation
import Kingfisher
import SwiftUI
import AVKit
import AppKit

class ClosableWindow: NSWindow {
    override func cancelOperation(_ sender: Any?) {
        self.close()
    }
}

@_cdecl("open_img_viewer")
public func open_img_viewer(urlCString: UnsafePointer<CChar>?) {
    let urlStr = urlCString.flatMap { String(cString: $0) } ?? "about:blank"
    let targetURL = URL(string: urlStr)!
    DispatchQueue.main.async {
        if NSApp == nil { _ = NSApplication.shared }

        let win = ClosableWindow(contentRect: NSRect(x: 200, y: 200, width: 1000, height: 700),
                           styleMask: [.resizable, .titled, .closable, .miniaturizable, .fullSizeContentView],
                           backing: .buffered, defer: false)
        win.titlebarAppearsTransparent = true
        win.titleVisibility = .hidden
        win.isMovableByWindowBackground = true
        win.appearance = .init(named: .darkAqua)
        win.setFrameOriginToPositionWindowInCenterOfScreen()

        win.contentView = makeZoomingScrollView(targetURL: targetURL)
        win.isReleasedWhenClosed = false

        win.makeKeyAndOrderFront(nil)
        NSApp.activate(ignoringOtherApps: true)
    }
}

@_cdecl("open_video_viewer")
public func open_video_viewer(urlCString: UnsafePointer<CChar>?) {
    let urlStr = urlCString.flatMap { String(cString: $0) } ?? "about:blank"
    let targetURL = URL(string: urlStr)!
    DispatchQueue.main.async {
        if NSApp == nil { _ = NSApplication.shared }

        let win = ClosableWindow(contentRect: NSRect(x: 200, y: 200, width: 1000, height: 700),
                           styleMask: [.resizable, .titled, .closable, .miniaturizable, .fullSizeContentView],
                           backing: .buffered, defer: false)
        win.titlebarAppearsTransparent = true
        win.titleVisibility = .hidden
        win.isMovableByWindowBackground = true
        win.appearance = .init(named: .darkAqua)
        win.setFrameOriginToPositionWindowInCenterOfScreen()
        let playerView = AVPlayerView()
        let videoPlayer = AVPlayer(url: targetURL)
        playerView.player = videoPlayer
        videoPlayer.play()
        win.contentView = playerView
        win.isReleasedWhenClosed = false

        win.makeKeyAndOrderFront(nil)
        NSApp.activate(ignoringOtherApps: true)
    }
}

struct OpenStatusImageModel: Decodable {
    let index: Int
    let medias: [StatusMediaItem]
}

struct StatusMediaItem: Decodable {
    let url: String
    let type: String
    let placeholder: String?
    let fileName: String
}

@_cdecl("open_status_image_viewer")
public func open_status_image_viewer(_ json: UnsafePointer<CChar>?) {
    guard let json = json else { return }
    let s = String(cString: json)
    if let data = s.data(using: .utf8),
       let model = try? JSONDecoder().decode(OpenStatusImageModel.self, from: data) {
        DispatchQueue.main.sync {
            if NSApp == nil { _ = NSApplication.shared }
            let win = ClosableWindow(contentRect: NSRect(x: 200, y: 200, width: 1000, height: 700),
                               styleMask: [.resizable, .titled, .closable, .miniaturizable, .fullSizeContentView],
                               backing: .buffered, defer: false)
            win.titlebarAppearsTransparent = true
            win.titleVisibility = .hidden
            win.isMovableByWindowBackground = true
            win.appearance = .init(named: .darkAqua)
            win.setFrameOriginToPositionWindowInCenterOfScreen()

            win.contentView = NSHostingView(rootView: StatusMediaView(medias: model.medias, page: model.index))
            win.isReleasedWhenClosed = false

            win.makeKeyAndOrderFront(nil)
            NSApp.activate(ignoringOtherApps: true)

        }
    }
}

struct StatusMediaView: View {
    let medias: [StatusMediaItem]
    @State var page: Int? = 0
    var body: some View {
        ScrollView(.horizontal) {
            HStack(spacing: 0) {
                ForEach(0..<medias.count, id: \.self) { i in
                    let media = medias[i]
                    ZStack {
                        if media.type == "gif" || media.type == "image" {
                            KFImage(.init(string: media.url)!)
                                .placeholder({
                                    if let placeholder = media.placeholder {
                                        KFImage
                                            .url(.init(string: placeholder)!)
                                            .resizable()
                                            .scaledToFit()
                                    } else {
                                        ProgressView()
                                    }
                                })
                                .resizable()
                                .scaledToFit()
                                .contextMenu {
                                    Button("Save Image") {
                                        let panel = NSSavePanel()
                                        panel.nameFieldStringValue = media.fileName
                                        panel.allowedContentTypes = [.png, .jpeg, .gif, .webP]
                                        panel.begin { response in
                                            if response == .OK, let url = panel.url {
                                                KingfisherManager.shared.retrieveImage(with: URL(string: media.url)!, options: nil) { result in
                                                    switch result {
                                                    case .success(let value):
                                                        if let data = value.image.kf.pngRepresentation() {
                                                            try? data.write(to: url)
                                                        }
                                                    case .failure(let error):
                                                        NSLog("Failed to save image: \(error)")
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                        } else if media.type == "video" {
                            if page == i {
                                let player = AVPlayer(url: .init(string: media.url)!)
                                VideoPlayer(player: player)
                                    .onAppear {
                                        player.play()
                                    }
                            } else if let placeholder = media.placeholder {
                                KFImage(.init(string: placeholder)!)
                            } else {
                                EmptyView()
                            }
                        }
                    }
                    .containerRelativeFrame(.horizontal, count: 1, spacing: 0)
                    .id(i)
                }
            }
            .scrollTargetLayout()
        }
        .scrollIndicators(.hidden)
        .scrollTargetBehavior(.paging)
        .scrollPosition(id: $page)
        .ignoresSafeArea()
    }
}

@MainActor
func makeZoomingScrollView(targetURL: URL) -> NSScrollView {
    let imageView = NSImageView()
    imageView.imageScaling = .scaleNone
    imageView.imageAlignment = .alignCenter
    imageView.animates = true

    let scrollView = NSScrollView()
    scrollView.contentView = CenteringClipView()
    scrollView.documentView = imageView
    scrollView.drawsBackground = false
    scrollView.hasVerticalScroller = true
    scrollView.hasHorizontalScroller = true
    scrollView.allowsMagnification = true
    scrollView.minMagnification = 0.1
    scrollView.maxMagnification = 8
    scrollView.automaticallyAdjustsContentInsets = false
    scrollView.contentInsets = .init()

    imageView.kf
        .setImage(with: targetURL) { result in
        switch result {
        case .success(let value):
            let size = value.image.size
            guard size.width > 0, size.height > 0 else { return }

            imageView.frame = CGRect(origin: .zero, size: size)

            scrollView.layoutSubtreeIfNeeded()
            scrollView.magnify(toFit: imageView.bounds)
            (scrollView.contentView as? NSClipView).map {
                $0.scroll(to: $0.bounds.origin)
                scrollView.reflectScrolledClipView($0)
            }
        case .failure(let error):
            NSLog("Kingfisher load failed: \(error)")
        }
    }

    return scrollView
}

final class CenteringClipView: NSClipView {
    override func constrainBoundsRect(_ proposedBounds: NSRect) -> NSRect {
        var rect = super.constrainBoundsRect(proposedBounds)
        guard let doc = documentView else { return rect }

        let docSize = doc.frame.size
        let clipSize = bounds.size

        if docSize.width < clipSize.width {
            rect.origin.x = -(clipSize.width - docSize.width) / 2
        }
        if docSize.height < clipSize.height {
            rect.origin.y = -(clipSize.height - docSize.height) / 2
        }
        return rect
    }
}

extension NSWindow {
    public func setFrameOriginToPositionWindowInCenterOfScreen() {
        if let screenSize = screen?.frame.size {
            self.setFrameOrigin(NSPoint(x: (screenSize.width-frame.size.width)/2, y: (screenSize.height-frame.size.height)/2))
        }
    }
}
