plugins {
  alias(libs.plugins.android.application)
  alias(libs.plugins.kotlin.android)
  alias(libs.plugins.kotlin.compose)
  alias(libs.plugins.ktlint)
  alias(libs.plugins.detekt)
}

android {
  namespace = "dev.davidv.translator"
  compileSdk = 34
  ndkVersion = "28.0.12674087"
  buildToolsVersion = "34.0.0"

  sourceSets {
    getByName("androidTest") {
      assets {
        srcDirs("src/androidTest/assets")
      }
    }
  }
  defaultConfig {
    applicationId = "dev.davidv.translator"
    minSdk = 28 // iconv functions need 28?
    targetSdk = 34
    versionCode = 6
    versionName = "0.2.1"

    testInstrumentationRunner = "androidx.test.runner.AndroidJUnitRunner"
  }

  buildTypes {
    release {
      isMinifyEnabled = true
      isShrinkResources = true
      // when building in F-Droid CI, the `cargo` binary is not in path
      // so there's a prebuild step to modify this file and replace "cargo"
      // with the full path to cargo (/home/vagrant/.cargo/bin/..)
      // however, modifying this file leaves the repo in a dirty state
      // which means that the revision in `META-INF/version-control-info.textproto`
      // does not match with the _actual_ commit.
      // Disabling this until I figure out how to put `cargo` in PATH
      // in F-Droid CI
      vcsInfo {
        include = false
      }
      proguardFiles(
        getDefaultProguardFile("proguard-android-optimize.txt"),
        "proguard-rules.pro",
      )
    }

    flavorDimensions += listOf("architecture")
    productFlavors {
      create("x86_64") {
        ndk {
          abiFilters += listOf("x86_64") // armeabi-v7a arm64-v8a
        }
        dimension = "architecture"
      }
      create("x86") {
        ndk {
          abiFilters += listOf("x86")
        }
        dimension = "architecture"
      }
      create("aarch64") {
        ndk {
          abiFilters += listOf("arm64-v8a")
        }
        dimension = "architecture"
      }
    }
  }
  compileOptions {
    sourceCompatibility = JavaVersion.VERSION_11
    targetCompatibility = JavaVersion.VERSION_11
  }
  kotlinOptions {
    jvmTarget = "11"
  }
  buildFeatures {
    compose = true
  }
}

val bindingsRootDir = "src/main/bindings"
val jniLibsDir = "../jniLibs"
val ndk = "${System.getenv("ANDROID_SDK_ROOT")}/ndk/28.0.12674087"
tasks.register("buildBindingsX86_64") {
  group = "build"
  description = "Build Rust bindings library for x86_64"

  doLast {
    exec {
      workingDir = file(bindingsRootDir)
      environment("ANDROID_NDK_ROOT", ndk)
      environment("ANDROID_NDK_HOME", ndk)
      commandLine(
        "/home/vagrant/.cargo/bin/cargo",
        "ndk",
        "build",
        "--target",
        "x86_64",
        "--release",
        "--platform",
        "28",
        "--link-libcxx-shared",
        "--output-dir",
        "../jniLibs",
      )
    }
  }
}

tasks.register("buildBindingsAarch64") {
  group = "build"
  description = "Build Rust bindings library for aarch64"

  doLast {
    exec {
      workingDir = file(bindingsRootDir)
      environment("ANDROID_NDK_ROOT", ndk)
      environment("ANDROID_NDK_HOME", ndk)
      commandLine(
        "/home/vagrant/.cargo/bin/cargo",
        "ndk",
        "build",
        "--target",
        "arm64-v8a",
        "--release",
        "--platform",
        "28",
        "--link-libcxx-shared",
        "--output-dir",
        "../jniLibs",
      )
    }
  }
}

tasks.register("buildBindingsAll") {
  group = "build"
  description = "Build Rust bindings library for all architectures"
  dependsOn("buildBindingsX86_64", "buildBindingsAarch64")
}

tasks.whenTaskAdded {
  if (name.contains("preAarch64") && name.contains("Build")) {
    dependsOn("buildBindingsAarch64")
  }
  if (name.contains("preX86_64") && name.contains("Build")) {
    dependsOn("buildBindingsX86_64")
  }
}

dependencies {

  implementation(libs.androidx.core.ktx)
  implementation(libs.androidx.lifecycle.runtime.ktx)
  implementation(libs.androidx.activity.compose)
  implementation(platform(libs.androidx.compose.bom))
  implementation(libs.androidx.ui)
  implementation(libs.androidx.ui.graphics)
  implementation(libs.androidx.ui.tooling.preview)
  implementation(libs.androidx.material3)
  implementation(libs.androidx.navigation.compose)
  implementation(libs.androidx.exifinterface)
  implementation(libs.androidx.uiautomator)
  testImplementation(libs.junit)
  androidTestImplementation(libs.androidx.junit)
  androidTestImplementation(libs.androidx.espresso.core)
  androidTestImplementation(platform(libs.androidx.compose.bom))
  androidTestImplementation(libs.androidx.ui.test.junit4)
  debugImplementation(libs.androidx.ui.tooling)
  debugImplementation(libs.androidx.ui.test.manifest)
  implementation(libs.kotlinx.coroutines.android)
  implementation(libs.kotlinx.coroutines.core)
  implementation(project(":app:bergamot"))
  // OpenMP - Multi-threaded. Provides better performance on multi-core processors when using only single instance of Tesseract.
//  implementation(("cz.adaptech.tesseract4android:tesseract4android-openmp:4.9.0"))
}

ktlint {
  android.set(true)
  ignoreFailures.set(false)
  reporters {
    reporter(org.jlleitschuh.gradle.ktlint.reporter.ReporterType.PLAIN)
    reporter(org.jlleitschuh.gradle.ktlint.reporter.ReporterType.CHECKSTYLE)
  }
  filter {
    exclude { element -> element.file.path.contains("generated/") }
  }
}

detekt {
  toolVersion = "1.23.4"
  config.setFrom(file("$projectDir/detekt-config.yml"))
  buildUponDefaultConfig = true
  allRules = false
}

tasks.register("lintAll") {
  dependsOn("ktlintCheck", "detekt")
  description = "Run all lint checks (ktlint and detekt)"
  group = "verification"
}

tasks.register("formatAll") {
  dependsOn("ktlintFormat")
  description = "Format all code using ktlint"
  group = "formatting"
}
