package dev.cognitivity.chronal.ui.metronome

import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.foundation.layout.BoxScope
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.offset
import androidx.compose.foundation.layout.padding
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.KeyboardArrowDown
import androidx.compose.material.icons.outlined.KeyboardArrowUp
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import dev.cognitivity.chronal.ChronalApp
import dev.cognitivity.chronal.ChronalApp.Companion.context
import dev.cognitivity.chronal.R
import dev.cognitivity.chronal.ui.metronome.windows.metronome
import dev.cognitivity.chronal.ui.metronome.windows.setBPM
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch

@Composable
fun BoxScope.TempoChanger() {
    Column(
        modifier = Modifier.align(Alignment.Center)
    ) {
        Row {
            val scope = rememberCoroutineScope()

            Column(
                modifier = Modifier.align(Alignment.CenterVertically)
                    .padding(2.dp, 0.dp, 2.dp, 0.dp)
            ) {
                Icon(
                    imageVector = Icons.Outlined.KeyboardArrowUp,
                    contentDescription = context.getString(R.string.metronome_increase_tempo),
                    modifier = Modifier.pointerInput(Unit) {
                            detectTapGestures(
                                onPress = {
                                    var isHeld = true
                                    scope.launch {
                                        setBPM(metronome.bpm + 1)
                                        delay(500)
                                        while (isHeld) {
                                            setBPM(metronome.bpm + 1)
                                            delay(50)
                                        }
                                    }
                                    tryAwaitRelease()
                                    isHeld = false
                                }
                            )
                        },
                    tint = MaterialTheme.colorScheme.tertiary
                )
                Icon(
                    imageVector = Icons.Outlined.KeyboardArrowDown,
                    contentDescription = "Decrease tempo",
                    modifier = Modifier.pointerInput(Unit) {
                            detectTapGestures(
                                onPress = {
                                    var isHeld = true
                                    scope.launch {
                                        setBPM(metronome.bpm - 1)

                                        delay(500)
                                        while (isHeld) {
                                            setBPM(metronome.bpm - 1)
                                            delay(50)
                                        }
                                    }
                                    tryAwaitRelease()
                                    isHeld = false
                                }
                            )
                        },
                    tint = MaterialTheme.colorScheme.tertiary
                )
            }
            Text(
                modifier = Modifier.align(Alignment.CenterVertically),
                text = metronome.bpm.toString(),
                style = MaterialTheme.typography.displayLarge,
                color = MaterialTheme.colorScheme.onSurface,
                maxLines = 1,
                overflow = TextOverflow.Visible
            )
            Text(
                modifier = Modifier.offset(y = (-4).dp)
                    .align(Alignment.Bottom),
                text = context.getString(R.string.metronome_bpm),
                style = MaterialTheme.typography.headlineSmall,
                color = MaterialTheme.colorScheme.onSurfaceVariant,
                maxLines = 1,
                overflow = TextOverflow.Visible
            )
        }

        val markings = ChronalApp.getInstance().settings.tempoMarkings.value.reversed()
        val marking = markings.firstOrNull { it.range.contains(metronome.bpm) }
        val string = marking?.name ?: context.getString(R.string.metronome_tempo_unknown)

        Text(
            text = string,
            style = MaterialTheme.typography.bodyLarge,
            color = MaterialTheme.colorScheme.tertiary,
            fontStyle = FontStyle.Italic,
            modifier = Modifier.align(Alignment.CenterHorizontally)
                .offset(y = (-8).dp)
        )
    }
}
