import 'package:drift/drift.dart';
import 'package:drift/internal/migrations.dart';
import 'package:drift/wasm.dart';
import 'package:drift_dev/src/services/schema/verifier_web.dart' as impl;
import 'package:drift_dev/src/services/schema/verifier_common.dart';
import 'package:sqlite3/wasm.dart';

import 'migrations_common.dart' as common;

export 'migrations_common.dart'
    show SchemaMismatch, SchemaInstantiationHelper, MissingSchemaException;

abstract class WebSchemaVerifier
    implements common.SchemaVerifier<CommonDatabase> {
  /// Creates a schema verifier for the drift-generated [helper].
  ///
  /// See [tests] for more information.
  /// The optional [setup] parameter is used internally by the verifier for
  /// every database connection it opens. This can be used to, for instance,
  /// register custom functions expected by your database.
  ///
  /// [tests]: https://drift.simonbinder.eu/docs/migrations/tests/
  factory WebSchemaVerifier(
    CommonSqlite3 sqlite3,
    SchemaInstantiationHelper helper, {
    void Function(CommonDatabase raw)? setup,
  }) = impl.WebSchemaVerifier;
}

/// Utilities verifying that the current schema of the database matches what
/// the generated code expects.
extension VerifySelf on GeneratedDatabase {
  /// Compares and validates the schema of the current database with what the
  /// generated code expects.
  ///
  /// When changing tables or other elements of your database schema, you need
  /// to increate your [GeneratedDatabase.schemaVersion] and write a migration
  /// to transform your existing tables to the new structure.
  ///
  /// For queries, drift always assumes that your database schema matches the
  /// structure of your defined tables. This isn't the case when you forget to
  /// write a schema migration, which can cause all kinds of problems later.
  ///
  /// For this reason, the [validateDatabaseSchema] method can be used in your
  /// database, (perhaps in a [MigrationStrategy.beforeOpen] callback) to verify
  /// that your database schema is what drift expects.
  ///
  /// The [common.ValidationOptions] can be used to make the schema validation
  /// more strict (e.g. by enabling [common.ValidationOptions.validateDropped]
  /// to ensure that no old tables continue to exist if they're not referenced
  /// in the new schema) or more lenient (e.g. by disabling
  /// [common.ValidationOptions.validateColumnConstraints]).
  ///
  /// This variant of [validateDatabaseSchema] is only supported on the web as
  /// a platform.
  Future<void> validateDatabaseSchema({
    required CommonSqlite3 sqlite3,
    common.ValidationOptions options = const common.ValidationOptions(),
    @Deprecated('Use field in ValidationOptions instead') bool? validateDropped,
  }) async {
    await verifyDatabase(
      this,
      options.applyDeprecatedValidateDroppedParam(validateDropped),
      () => WasmDatabase.inMemory(sqlite3),
    );
  }
}
