import org.jetbrains.kotlin.gradle.dsl.JvmTarget
import java.io.ByteArrayOutputStream
import java.nio.charset.Charset

buildscript {
    repositories {
        google()
        mavenCentral()
        gradlePluginPortal()
    }
    dependencies {
        classpath(libs.gradle)
    }
}

plugins {
    alias(libs.plugins.android.application)
    alias(libs.plugins.kotlin.android)
    alias(libs.plugins.android.junit5)
}

abstract class GitVersionValueSource : ValueSource<Int, ValueSourceParameters.None> {
    @get:Inject
    abstract val execOperations: ExecOperations

    override fun obtain(): Int {
        val output = ByteArrayOutputStream()
        execOperations.exec {
            commandLine("git", "rev-list", "HEAD", "--count")
            standardOutput = output
        }
        return Integer.valueOf(String(output.toByteArray(), Charset.defaultCharset()).trim())
    }
}

val gitVersionProvider = providers.of(GitVersionValueSource::class) {}
val gitVersion = gitVersionProvider.get()

android {
    defaultConfig {
        compileSdk = 35
        minSdk = 26
        targetSdk = 36
        versionCode = 53
        versionName = "6.4.1"
        applicationId = "de.storchp.opentracks.osmplugin"

        testInstrumentationRunnerArguments += mapOf("runnerBuilder" to "de.mannodermaus.junit5.AndroidJUnit5Builder")
        testInstrumentationRunner = "androidx.test.runner.AndroidJUnitRunner"
    }

    dependenciesInfo {
        // Disables dependency metadata when building APKs.
        includeInApk = false
        // Disables dependency metadata when building Android App Bundles.
        includeInBundle = false
    }

    bundle {
        language {
            // Specifies that the app bundle should not support
            // configuration APKs for language resources. These
            // resources are instead packaged with each base and
            // dynamic feature APK.
            enableSplit = false
        }
    }

    compileOptions {
        // Sets Java compatibility to Java 17
        sourceCompatibility = JavaVersion.VERSION_17
        targetCompatibility = JavaVersion.VERSION_17
    }

    kotlin {
        compilerOptions {
            jvmTarget.set(JvmTarget.JVM_17)
        }
    }

    flavorDimensions += "default"
    productFlavors {
        create("full") {
            applicationId = "de.storchp.opentracks.osmplugin"
            buildConfigField("boolean", "offline", "false")
            dimension = "default"
        }
        create("offline") {
            applicationId = "de.storchp.opentracks.osmplugin.offline"
            buildConfigField("boolean", "offline", "true")
            dimension = "default"
        }
    }

    sourceSets.getByName("full") {
        manifest.srcFile("src/full/AndroidManifest.xml")
    }
    sourceSets.getByName("offline") {
        manifest.srcFile("src/offline/AndroidManifest.xml")
    }

    buildTypes {
        getByName("debug") {
            applicationIdSuffix = ".debug"
            versionNameSuffix = "-debug"
        }
        register("nightly") {
            applicationIdSuffix = ".nightly"
        }
        getByName("release") {
            isMinifyEnabled = false
            proguardFiles(
                getDefaultProguardFile("proguard-android.txt"),
                "proguard-rules.pro"
            )
        }
    }
    buildFeatures {
        viewBinding = true
        buildConfig = true
    }
    lint {
        disable.add("MissingTranslation")
    }
    androidResources {
        @Suppress("UnstableApiUsage")
        generateLocaleConfig = true
    }

    namespace = "de.storchp.opentracks.osmplugin"

    applicationVariants.all {
        resValue("string", "applicationId", applicationId)

        if (buildType.name == "nightly" || buildType.name == "debug") {
            outputs.forEach { output ->
                output as com.android.build.gradle.internal.api.ApkVariantOutputImpl
                output.versionCodeOverride = gitVersion
                output.versionNameOverride = "${applicationId}_${output.versionCode}"
                output.outputFileName = "${applicationId}_${versionCode}.apk"
            }
        } else {
            outputs.forEach { output ->
                output as com.android.build.gradle.internal.api.ApkVariantOutputImpl
                output.outputFileName = "${applicationId}_${versionName}.apk"
            }
        }
    }
}

dependencies {

    // Desugaring
    coreLibraryDesugaring(libs.desugar.jdk.libs)

    // AndroidX
    implementation(libs.lifecycle.viewmodel.ktx)
    implementation(libs.appcompat)
    implementation(libs.documentfile)
    implementation(libs.preference.ktx)
    implementation(libs.constraintlayout)
    implementation(libs.material)
    implementation(libs.activity)
    implementation(libs.activity.ktx)
    implementation(libs.colorpicker)

    // VTM
    implementation(libs.vtm)
    implementation(libs.vtm.themes)
    implementation(libs.vtm.http)
    implementation(libs.slf4j.api)
    runtimeOnly(libs.vtm.android) {
        artifact {
            classifier = "natives-armeabi-v7a"
        }
    }
    runtimeOnly(libs.vtm.android) {
        artifact {
            classifier = "natives-arm64-v8a"
        }
    }
    runtimeOnly(libs.vtm.android) {
        artifact {
            classifier = "natives-x86"
        }
    }
    runtimeOnly(libs.vtm.android) {
        artifact {
            classifier = "natives-x86_64"
        }
    }
    implementation(libs.vtm.android)
    implementation(libs.androidsvg)
    implementation(libs.jsoup.jsoup)

    testImplementation(libs.junit.jupiter.api)
    testImplementation(libs.junit.jupiter.params)
    testImplementation(libs.assertj.core)
    testImplementation(libs.mockk)

    testRuntimeOnly(libs.junit.jupiter.engine)

    androidTestImplementation(libs.runner)
    androidTestImplementation(libs.junit.jupiter.api)
    androidTestImplementation(libs.junit.jupiter.params)
    androidTestImplementation(libs.android.test.core)

    androidTestRuntimeOnly(libs.android.test.runner)
}
