package de.schliweb.makeacopy.ui.ocr.review.model;

import android.graphics.RectF;
import de.schliweb.makeacopy.ui.ocr.OCRViewModel;
import de.schliweb.makeacopy.utils.RecognizedWord;

import java.util.List;

/**
 * Maps the current OCR UI state into the editable OCR JSON schema (OcrDoc, schema=1).
 * Boxes are expected to be in pixels of the OCR destination image (scaled bitmap).
 */
public final class OcrDocMapper {
    private OcrDocMapper() {
    }

    public static OcrDoc fromState(OCRViewModel.OcrUiState state) {
        OcrDoc doc = new OcrDoc();
        if (state == null) return doc;
        OCRViewModel.OcrTransform tx = state.transform();
        if (tx != null) {
            doc.imageSize.w = tx.dstW();
            doc.imageSize.h = tx.dstH();
        }
        // Use effective words (reviewed if available, otherwise original OCR)
        List<RecognizedWord> words = state.getEffectiveWords();
        if (words == null) return doc;
        int id = 1;
        for (RecognizedWord rw : words) {
            if (rw == null) continue;
            RectF b = rw.getBoundingBox();
            if (b == null) continue;
            OcrDoc.Word w = new OcrDoc.Word();
            w.id = id++;
            w.t = rw.getText() != null ? rw.getText() : "";
            // convert to [x,y,w,h] ints; clamp to non-negative
            int x = Math.max(0, Math.round(b.left));
            int y = Math.max(0, Math.round(b.top));
            int width = Math.max(0, Math.round(b.right - b.left));
            int height = Math.max(0, Math.round(b.bottom - b.top));
            w.b[0] = x;
            w.b[1] = y;
            w.b[2] = width;
            w.b[3] = height;
            float c = rw.getConfidence();
            if (c > 1.0f) c = c / 100f; // normalize tess 0..100 → 0..1
            if (c < 0f) c = 0f;
            if (c > 1f) c = 1f;
            w.c = c;
            w.e = false; // not edited initially
            w.l = 0; // unknown line id for now (MVP stub)
            w.k = 0; // unknown block id for now (MVP stub)
            w.lang = rw.getLang(); // preserve language from reviewed words
            doc.words.add(w);
        }
        return doc;
    }
}
