package de.schliweb.makeacopy.data.library;

import androidx.room.*;

import java.util.List;

/**
 * Data Access Object (DAO) for managing operations on the "collections" table
 * within the Room database. Provides methods for inserting, updating,
 * retrieving, and deleting records, as well as performing specific queries
 * related to collections and their relationships.
 * <p>
 * Methods included:
 * <p>
 * - insert(CollectionEntity collection): Adds a new collection to the database.
 * If a conflict occurs (e.g., duplicate ID), the operation is aborted.
 * <p>
 * - update(CollectionEntity collection): Updates an existing collection's details
 * in the database.
 * <p>
 * - getAll(): Retrieves all collections from the database, sorted by their sort
 * order (ascending) and creation timestamp (ascending).
 * <p>
 * - getById(String id): Fetches a collection by its unique ID. Returns a single
 * collection or null if no match is found.
 * <p>
 * - deleteById(String id): Deletes the collection with the specified ID from the
 * database.
 * <p>
 * - countItems(String collectionId): Counts the number of related items (e.g., scans)
 * linked to a specific collection ID via the scan-collection join table.
 */
@Dao
public interface CollectionsDao {
    @Insert(onConflict = OnConflictStrategy.ABORT)
    void insert(CollectionEntity collection);

    @Update
    void update(CollectionEntity collection);

    @Query("SELECT * FROM collections ORDER BY sortOrder ASC, createdAt ASC")
    List<CollectionEntity> getAll();

    @Query("SELECT * FROM collections WHERE id = :id LIMIT 1")
    CollectionEntity getById(String id);

    @Query("SELECT * FROM collections WHERE name = :name LIMIT 1")
    CollectionEntity getByName(String name);

    @Query("DELETE FROM collections WHERE id = :id")
    void deleteById(String id);

    @Query("SELECT COUNT(*) FROM scan_collection_join WHERE collectionId = :collectionId")
    int countItems(String collectionId);
}