package de.schliweb.makeacopy.ui.camera;

import android.content.Context;

import androidx.fragment.app.testing.FragmentScenario;
import androidx.lifecycle.Lifecycle;
import androidx.test.espresso.Espresso;
import androidx.test.espresso.NoMatchingViewException;
import androidx.test.espresso.PerformException;
import androidx.test.espresso.action.ViewActions;
import androidx.test.espresso.assertion.ViewAssertions;
import androidx.test.espresso.matcher.ViewMatchers;
import androidx.test.ext.junit.runners.AndroidJUnit4;

import de.schliweb.makeacopy.R;
import de.schliweb.makeacopy.a11y.util.A11yCapture;
import de.schliweb.makeacopy.a11y.util.TestPrefs;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;

import java.util.concurrent.atomic.AtomicBoolean;

import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assume.assumeTrue;

/**
 * Verifies torch (flashlight) accessibility announcements when a flash unit is available.
 * If the device/emulator reports no flash unit, the test is skipped gracefully.
 * Note: torch haptics in production are emitted via View.performHapticFeedback, which we do not
 * capture in tests. We therefore assert announcements and UI behavior only.
 */
@RunWith(AndroidJUnit4.class)
public class CameraA11yTorchTest extends de.schliweb.makeacopy.a11y.util.A11yBaseTest {

    @Before
    public void setUp() {
        A11yCapture.install();
    }

    @After
    public void tearDown() {
        A11yCapture.uninstall();
    }

    @Test
    public void toggling_flash_emits_accessibility_announcements_when_flash_available() throws Exception {
        FragmentScenario<CameraFragment> scenario = FragmentScenario.launchInContainer(
                CameraFragment.class,
                null,
                R.style.Theme_MakeACopy,
                Lifecycle.State.RESUMED
        );

        final AtomicBoolean hasFlashRef = new AtomicBoolean(false);
        scenario.onFragment(f -> {
            Context ctx = f.requireContext();
            TestPrefs.setAccessibilityMode(ctx, true);
            // Probe flash availability if camera is bound
            boolean has = false;
            try {
                if (f != null && f.getView() != null) {
                    // Best-effort: if camera is initialized, query its info
                    androidx.camera.core.ImageCapture cap = f.getImageCaptureForTest();
                    // ImageCapture existence correlates with bound camera; check camera info via button state next
                    has = (cap != null); // weak heuristic
                }
            } catch (Exception ignored) {}
            // Even if ImageCapture is null (not fully bound), allow the test to run; we will skip
            // if no torch announcement occurs after click.
            hasFlashRef.set(has);
        });

        // If the flash button is not visible (e.g. emulator/no-flash device), skip gracefully.
        try {
            Espresso.onView(ViewMatchers.withId(R.id.button_flash))
                    .check(ViewAssertions.matches(ViewMatchers.isDisplayed()));
        } catch (NoMatchingViewException | AssertionError e) {
            assumeTrue("Skipping: flash button not visible in this environment.", false);
        }

        // Try toggling the flash button.
        try {
            Espresso.onView(ViewMatchers.withId(R.id.button_flash))
                    .perform(ViewActions.click());
        } catch (PerformException e) {
            assumeTrue("Skipping: cannot click flash button (animations/device constraints).", false);
        }

        // Expect either an announcement for ON or OFF, depending on prior state
        CharSequence ann = de.schliweb.makeacopy.a11y.util.A11yCapture.await(2500);

        // If no announcement was captured, assume the environment lacks a flash-capable camera and skip.
        assumeTrue("Skipping: torch announcement not observed; device may have no flash or camera unbound.", ann != null);

        // Assert the announcement contains one of the localized flash strings
        scenario.onFragment(f -> {
            String onS = f.getString(R.string.flashlight_on);
            String offS = f.getString(R.string.flashlight_off);
            String text = String.valueOf(ann);
            boolean matches = text.contains(onS) || text.contains(offS);
            org.junit.Assert.assertTrue("Expected flashlight_on/off announcement, was: " + text, matches);
        });

        // Toggle again to switch state and expect the opposite announcement
        A11yCapture.clear();
        Espresso.onView(androidx.test.espresso.matcher.ViewMatchers.withId(R.id.button_flash))
                .perform(ViewActions.click());
        CharSequence ann2 = de.schliweb.makeacopy.a11y.util.A11yCapture.await(2500);
        assertNotNull("Expected second flashlight announcement", ann2);
        scenario.onFragment(f -> {
            String onS = f.getString(R.string.flashlight_on);
            String offS = f.getString(R.string.flashlight_off);
            String text = String.valueOf(ann2);
            boolean ok = text.contains(onS) || text.contains(offS);
            org.junit.Assert.assertTrue("Second announcement should be flashlight_on/off, was: " + text, ok);
        });
    }
}
