package de.schliweb.makeacopy.ocr;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Typeface;
import androidx.test.ext.junit.runners.AndroidJUnit4;
import androidx.test.platform.app.InstrumentationRegistry;
import de.schliweb.makeacopy.utils.OCRHelper;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.runner.RunWith;

import static org.junit.Assert.*;

/**
 * Instrumented tests for multi-language OCR recognition.
 * Tests verify that mixed-script documents (e.g., Persian + English)
 * are correctly recognized when using combined language codes like "fas+eng".
 * <p>
 * Test scenarios:
 * - Persian + English mixed text recognition
 * - Arabic + French mixed text recognition
 * - Verification that both scripts are detected in the output
 */
@RunWith(AndroidJUnit4.class)
public class MultiLanguageOcrTest {

    private static Context context;

    @BeforeClass
    public static void setup() {
        context = InstrumentationRegistry.getInstrumentation().getTargetContext();
    }

    /**
     * Creates a test bitmap with both English and Persian text.
     * The bitmap contains clear, high-contrast text suitable for OCR.
     *
     * @return Bitmap with mixed English and Persian text
     */
    private Bitmap createEnglishPersianTestBitmap() {
        // Create a white bitmap (300 DPI equivalent for A6 size: ~1240x1748)
        int width = 800;
        int height = 400;
        Bitmap bitmap = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);
        Canvas canvas = new Canvas(bitmap);

        // White background
        canvas.drawColor(Color.WHITE);

        // Paint for English text
        Paint englishPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        englishPaint.setColor(Color.BLACK);
        englishPaint.setTextSize(48f);
        englishPaint.setTypeface(Typeface.SERIF);

        // Paint for Persian text (uses default system font which should support Arabic script)
        Paint persianPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        persianPaint.setColor(Color.BLACK);
        persianPaint.setTextSize(48f);
        // Use default typeface which typically supports Arabic/Persian on Android

        // Draw English text at the top
        String englishText = "Hello World";
        canvas.drawText(englishText, 50, 80, englishPaint);

        // Draw more English text
        String englishText2 = "Document Scanner";
        canvas.drawText(englishText2, 50, 150, englishPaint);

        // Draw Persian text in the middle
        // "سلام دنیا" = "Hello World" in Persian
        String persianText = "سلام دنیا";
        canvas.drawText(persianText, 50, 250, persianPaint);

        // Draw more Persian text
        // "اسکنر اسناد" = "Document Scanner" in Persian
        String persianText2 = "اسکنر اسناد";
        canvas.drawText(persianText2, 50, 320, persianPaint);

        return bitmap;
    }

    /**
     * Creates a test bitmap with both English and Arabic text.
     *
     * @return Bitmap with mixed English and Arabic text
     */
    private Bitmap createEnglishArabicTestBitmap() {
        int width = 800;
        int height = 400;
        Bitmap bitmap = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);
        Canvas canvas = new Canvas(bitmap);

        canvas.drawColor(Color.WHITE);

        Paint textPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        textPaint.setColor(Color.BLACK);
        textPaint.setTextSize(48f);

        // English text
        canvas.drawText("Welcome", 50, 80, textPaint);
        canvas.drawText("Technology", 50, 150, textPaint);

        // Arabic text
        // "مرحبا" = "Welcome" in Arabic
        canvas.drawText("مرحبا", 50, 250, textPaint);
        // "تكنولوجيا" = "Technology" in Arabic
        canvas.drawText("تكنولوجيا", 50, 320, textPaint);

        return bitmap;
    }

    /**
     * Creates a test bitmap with both German and French text.
     *
     * @return Bitmap with mixed German and French text
     */
    private Bitmap createGermanFrenchTestBitmap() {
        int width = 800;
        int height = 400;
        Bitmap bitmap = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);
        Canvas canvas = new Canvas(bitmap);

        canvas.drawColor(Color.WHITE);

        Paint textPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        textPaint.setColor(Color.BLACK);
        textPaint.setTextSize(48f);
        textPaint.setTypeface(Typeface.SERIF);

        // German text with umlauts
        canvas.drawText("Guten Tag", 50, 80, textPaint);
        canvas.drawText("Überprüfung", 50, 150, textPaint);

        // French text with accents
        canvas.drawText("Bonjour", 50, 250, textPaint);
        canvas.drawText("Vérification", 50, 320, textPaint);

        return bitmap;
    }

    /**
     * Tests OCR recognition of mixed English and Persian text using "fas+eng" language code.
     * Verifies that both Latin and Arabic script characters are recognized.
     */
    @Test
    public void testPersianEnglishMixedOcr() throws Exception {
        Bitmap testBitmap = createEnglishPersianTestBitmap();
        assertNotNull("Test bitmap should be created", testBitmap);

        try {
            String ocrResult = performOcr(testBitmap, "fas+eng");
            assertNotNull("OCR result should not be null", ocrResult);
            assertFalse("OCR result should not be empty", ocrResult.trim().isEmpty());

            System.out.println("[DEBUG_LOG] Persian+English OCR result: " + ocrResult);

            // Check for Latin characters (English text)
            boolean hasLatinChars = ocrResult.codePoints()
                    .anyMatch(cp -> (cp >= 'A' && cp <= 'Z') || (cp >= 'a' && cp <= 'z'));

            // Check for Arabic/Persian characters (Unicode range 0x0600-0x06FF)
            boolean hasArabicPersianChars = ocrResult.codePoints()
                    .anyMatch(cp -> cp >= 0x0600 && cp <= 0x06FF);

            System.out.println("[DEBUG_LOG] Has Latin chars: " + hasLatinChars);
            System.out.println("[DEBUG_LOG] Has Arabic/Persian chars: " + hasArabicPersianChars);

            // At minimum, we expect some text to be recognized
            // Note: The quality of recognition depends on the system fonts available
            // for rendering Persian text on the test device
            assertTrue("OCR should recognize some text",
                    hasLatinChars || hasArabicPersianChars);

            // If both scripts are rendered correctly, both should be recognized
            if (hasLatinChars && hasArabicPersianChars) {
                System.out.println("[DEBUG_LOG] SUCCESS: Both English and Persian text recognized!");
            } else if (hasLatinChars) {
                System.out.println("[DEBUG_LOG] PARTIAL: Only English text recognized (Persian font may not be available on device)");
            } else if (hasArabicPersianChars) {
                System.out.println("[DEBUG_LOG] PARTIAL: Only Persian text recognized");
            }

        } finally {
            testBitmap.recycle();
        }
    }

    /**
     * Tests OCR recognition of mixed English and Arabic text using "ara+eng" language code.
     */
    @Test
    public void testArabicEnglishMixedOcr() throws Exception {
        Bitmap testBitmap = createEnglishArabicTestBitmap();
        assertNotNull("Test bitmap should be created", testBitmap);

        try {
            String ocrResult = performOcr(testBitmap, "ara+eng");
            assertNotNull("OCR result should not be null", ocrResult);
            assertFalse("OCR result should not be empty", ocrResult.trim().isEmpty());

            System.out.println("[DEBUG_LOG] Arabic+English OCR result: " + ocrResult);

            boolean hasLatinChars = ocrResult.codePoints()
                    .anyMatch(cp -> (cp >= 'A' && cp <= 'Z') || (cp >= 'a' && cp <= 'z'));

            boolean hasArabicChars = ocrResult.codePoints()
                    .anyMatch(cp -> cp >= 0x0600 && cp <= 0x06FF);

            System.out.println("[DEBUG_LOG] Has Latin chars: " + hasLatinChars);
            System.out.println("[DEBUG_LOG] Has Arabic chars: " + hasArabicChars);

            assertTrue("OCR should recognize some text",
                    hasLatinChars || hasArabicChars);

        } finally {
            testBitmap.recycle();
        }
    }

    /**
     * Tests OCR recognition of mixed German and French text using "deu+fra" language code.
     * This tests Latin script with different diacritics (umlauts vs accents).
     */
    @Test
    public void testGermanFrenchMixedOcr() throws Exception {
        Bitmap testBitmap = createGermanFrenchTestBitmap();
        assertNotNull("Test bitmap should be created", testBitmap);

        try {
            String ocrResult = performOcr(testBitmap, "deu+fra");
            assertNotNull("OCR result should not be null", ocrResult);
            assertFalse("OCR result should not be empty", ocrResult.trim().isEmpty());

            System.out.println("[DEBUG_LOG] German+French OCR result: " + ocrResult);

            // Check for Latin characters
            boolean hasLatinChars = ocrResult.codePoints()
                    .anyMatch(cp -> (cp >= 'A' && cp <= 'Z') || (cp >= 'a' && cp <= 'z'));

            assertTrue("OCR should recognize Latin text", hasLatinChars);

            // Check for specific words or characters
            String lowerResult = ocrResult.toLowerCase();
            boolean hasGermanContent = lowerResult.contains("guten") || 
                    lowerResult.contains("tag") || 
                    lowerResult.contains("ü") ||
                    lowerResult.contains("überprüfung");
            boolean hasFrenchContent = lowerResult.contains("bonjour") || 
                    lowerResult.contains("é") ||
                    lowerResult.contains("vérification");

            System.out.println("[DEBUG_LOG] Has German content: " + hasGermanContent);
            System.out.println("[DEBUG_LOG] Has French content: " + hasFrenchContent);

            // At least some content should be recognized
            assertTrue("OCR should recognize German or French text",
                    hasGermanContent || hasFrenchContent);

        } finally {
            testBitmap.recycle();
        }
    }

    /**
     * Tests that OCRHelper correctly initializes with multi-language code.
     */
    @Test
    public void testMultiLanguageInitialization() throws Exception {
        OCRHelper helper = new OCRHelper(context);
        try {
            // Test Persian + English
            helper.setLanguage("fas+eng");
            boolean initialized = helper.initTesseract();
            assertTrue("Tesseract should initialize with fas+eng", initialized);

            // PSM should be AUTO for RTL languages
            int psm = helper.getPageSegMode();
            System.out.println("[DEBUG_LOG] PSM for fas+eng: " + psm);
            assertEquals("PSM should be AUTO (3) for RTL language combination", 3, psm);

        } finally {
            helper.shutdown();
        }
    }

    /**
     * Tests that OCRHelper correctly initializes with Arabic + French.
     */
    @Test
    public void testArabicFrenchInitialization() throws Exception {
        OCRHelper helper = new OCRHelper(context);
        try {
            helper.setLanguage("ara+fra");
            boolean initialized = helper.initTesseract();
            assertTrue("Tesseract should initialize with ara+fra", initialized);

            int psm = helper.getPageSegMode();
            System.out.println("[DEBUG_LOG] PSM for ara+fra: " + psm);
            assertEquals("PSM should be AUTO (3) for RTL language combination", 3, psm);

        } finally {
            helper.shutdown();
        }
    }

    /**
     * Tests that OCRHelper correctly initializes with German + English (both LTR).
     */
    @Test
    public void testGermanEnglishInitialization() throws Exception {
        OCRHelper helper = new OCRHelper(context);
        try {
            helper.setLanguage("deu+eng");
            boolean initialized = helper.initTesseract();
            assertTrue("Tesseract should initialize with deu+eng", initialized);

            // For LTR-only combinations, PSM may not be AUTO
            int psm = helper.getPageSegMode();
            System.out.println("[DEBUG_LOG] PSM for deu+eng: " + psm);
            // PSM can be various values for LTR languages, just verify initialization worked

        } finally {
            helper.shutdown();
        }
    }

    /**
     * Performs OCR on a bitmap using the specified language code.
     *
     * @param bitmap   The bitmap to process
     * @param langCode The language code (e.g., "fas+eng", "ara+fra")
     * @return The recognized text
     */
    private String performOcr(Bitmap bitmap, String langCode) throws Exception {
        OCRHelper helper = new OCRHelper(context);
        try {
            helper.setLanguage(langCode);
            boolean initialized = helper.initTesseract();
            if (!initialized) {
                throw new RuntimeException("Failed to initialize Tesseract for language: " + langCode);
            }

            OCRHelper.OcrResultWords result = helper.runOcrWithRetry(bitmap);
            return result != null ? result.text : null;
        } finally {
            helper.shutdown();
        }
    }
}
