import 'package:flutter/cupertino.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:foss_warn/class/class_app_state.dart';
import 'package:foss_warn/class/class_fpas_place.dart';
import 'package:foss_warn/class/class_notification_service.dart';
import 'package:foss_warn/class/class_user_preferences.dart';
import 'package:foss_warn/class/class_warn_message.dart';
import 'package:foss_warn/enums/severity.dart';
import 'package:foss_warn/enums/sorting_categories.dart';
import 'package:foss_warn/extensions/context.dart';
import 'package:foss_warn/extensions/list.dart';
import 'package:foss_warn/services/alert_api/fpas.dart';
import 'package:foss_warn/services/api_handler.dart';
import 'package:foss_warn/services/list_handler.dart';
import 'package:foss_warn/services/update_loop.dart';

import '../class/class_error_logger.dart';

class AlertRetrievalError implements Exception {}

final processedAlertsProvider =
    StateNotifierProvider<WarningService, List<WarnMessage>>(
  (ref) {
    return WarningService(
      userPreferences: ref.watch(userPreferencesProvider),
      userPreferencesService: ref.watch(userPreferencesProvider.notifier),
      places: ref.watch(myPlacesProvider),
    );
  },
);

/// Fetches alerts for all subscriptions.
/// Any new alerts will be fetched completely, any we already know about
/// will be retrieved from cache instead.
final alertsFutureProvider = FutureProvider<List<WarnMessage>>((ref) async {
  var alertApi = ref.watch(alertApiProvider);
  var places = ref.watch(myPlacesProvider);

  if (places.isEmpty) return [];

  // Fetch all available alerts
  List<AlertApiResult> retrievedAlerts;

  /// fetch alerts for one place and catch invalid subscriptions errors
  Future<List<({String alertId, String subscriptionId})>> getAlertForOnePlace(
    Place place,
    AppState appState,
  ) async {
    if (place.isExpired) {
      // the places has expired, We can not fetch for alerts until we resubscribed again
      return [];
    }

    try {
      return await alertApi.getAlerts(
        subscriptionId: place.subscriptionId,
        appState: appState,
      );
    } on InvalidSubscriptionError {
      // set expired to true
      ref
          .read(myPlacesProvider.notifier)
          .set(places.updateEntry(place.copyWith(isExpired: true)));
      return [];
    }
  }

  try {
    List<List<({String alertId, String subscriptionId})>> alertsForPlaces =
        await Future.wait([
      for (var place in places) ...[
        getAlertForOnePlace(place, ref.read(appStateProvider)),
      ],
    ]);

    // Combine alerts for the individual places into a single list
    retrievedAlerts =
        alertsForPlaces.reduce((value, element) => value + element);
  } catch (e) {
    debugPrint("[warnings] Tried to get alerts forPlaces and failed with $e");
    throw AlertRetrievalError();
  }

  var previouslyCachedAlerts = ref.read(processedAlertsProvider);

  // Determine which alerts we don't already know about
  var newAlerts = <AlertApiResult>[];
  for (AlertApiResult alert in retrievedAlerts) {
    if (!previouslyCachedAlerts.any(
      (oldAlert) =>
          oldAlert.fpasId == alert.alertId &&
          oldAlert.placeSubscriptionId == alert.subscriptionId,
    )) {
      newAlerts.add(alert);
    }
  }

  // Only get detail for new results
  List<WarnMessage> newAlertsDetails = [];
  await Future.wait(
    [
      for (var alert in newAlerts) ...[
        alertApi.getAlertDetail(
          alertId: alert.alertId,
          placeSubscriptionId: alert.subscriptionId,
        ),
      ],
    ],
    // cleanup is called in case there was an error in one of the futures
    cleanUp: (value) {
      newAlertsDetails.add(value);
    },
  ).then(
    (value) {
      newAlertsDetails = value;
    },
    onError: (exception) {
      ErrorLogger.writeErrorLog(
        "warnings.dart",
        "Get new alert details",
        exception.toString(),
      );

      var appStateService = ref.read(appStateProvider.notifier);
      appStateService.setError(true);
      return [];
    },
  ).catchError((exception) {
    ErrorLogger.writeErrorLog(
      "warnings.dart",
      "Get new alert details",
      exception.toString(),
    );
    var appStateService = ref.read(appStateProvider.notifier);
    appStateService.setError(true);
  });

  var result = newAlertsDetails + previouslyCachedAlerts;

  // add new alert to the processed alerts
  for (WarnMessage alert in newAlertsDetails) {
    ref.read(processedAlertsProvider.notifier).updateAlert(alert);
  }

  var cachedAlerts = ref.read(processedAlertsProvider);
  for (WarnMessage alert in cachedAlerts) {
    if (!retrievedAlerts.any(
      (apiResult) =>
          alert.fpasId == apiResult.alertId &&
          alert.placeSubscriptionId == apiResult.subscriptionId,
    )) {
      // the alert is not in the server response anymore, remove cached alert
      ref.read(processedAlertsProvider.notifier).deleteAlert(alert);
    }
  }

  // we have once fetched alerts, we do not need to display the loading scree again.
  var appStateService = ref.read(appStateProvider.notifier);
  appStateService.setIsFirstFetch(false);

  return result;
});

/// Provides a complete list of all warnings for subscribed places.
///
/// It polls for new alerts and merges the result with any locally processed/modified alerts.
/// Any processing off alerts has to be done through [processedAlertsProvider].
final alertsProvider = Provider<List<WarnMessage>>((ref) {
  ref.listen(tickingChangeProvider(50), (_, event) {
    ref.invalidate(alertsFutureProvider);
  });

  var userPreferences = ref.watch(userPreferencesProvider);

  var alertsSnapshot = ref.watch(alertsFutureProvider);

  if (!alertsSnapshot.hasValue) return [];
  var alerts = alertsSnapshot.requireValue;

  List<WarnMessage> sortWarnings(List<WarnMessage> warnings) {
    var sortedWarnings = List<WarnMessage>.of(warnings);

    if (userPreferences.sortWarningsBy == SortingCategories.severity) {
      sortedWarnings.sort(
        (a, b) => Severity.getIndexFromSeverity(a.info[0].severity)
            .compareTo(Severity.getIndexFromSeverity(b.info[0].severity)),
      );
    } else if (userPreferences.sortWarningsBy == SortingCategories.data) {
      sortedWarnings.sort((a, b) => b.sent.compareTo(a.sent));
    }

    return sortedWarnings;
  }

  /// Check if the given alert is an update of a previous alert.
  /// Returns the notified status of the original alert if the severity hasn't increased
  bool isAlertAnUpdate({
    required List<WarnMessage> existingWarnings,
    required WarnMessage newWarning,
  }) {
    // check if there is a referenced warning
    if (newWarning.references != null) {
      // check if one of the referenced alerts is already in the warnings list
      for (var warning in existingWarnings) {
        if (newWarning.references!.identifier
            .any((identifier) => warning.identifier == identifier)) {
          // if there is a referenced alert, used the same value for notified.
          // use the notified value of the referenced warning, but only if the severity is still the same or lesser
          if (newWarning.info[0].severity.index >=
              warning.info[0].severity.index) {
            return warning.notified;
          }
        }
      }
    }
    return false;
  }

  // Determine which alert is an update of a previous one
  var updatedWarnings = <WarnMessage>[];
  for (var alert in alerts) {
    updatedWarnings.add(
      alert.copyWith(
        isUpdateOfAlreadyNotifiedWarning: isAlertAnUpdate(
          existingWarnings: alerts,
          newWarning: alert,
        ),
      ),
    );
  }
  for (var warning in updatedWarnings) {
    if (warning.references == null) continue;

    // The alert contains a reference, so it is an update of an previous alert
    for (String referenceId in warning.references!.identifier) {
      // Check all alerts for references
      var alWm =
          alerts.firstWhereOrNull((alert) => alert.identifier == referenceId);
      // if alert exist, set update flag to true
      if (alWm != null) {
        alerts.updateEntry(
          alWm.copyWith(hideWarningBecauseThereIsANewerVersion: true),
        );
      }
    }
  }

  return sortWarnings(alerts);
});

/// set the read status from all warnings to true
/// @ref to update view
void markAllWarningsAsRead(WidgetRef ref) {
  var alerts = ref.read(processedAlertsProvider);

  for (var alert in alerts) {
    ref
        .read(processedAlertsProvider.notifier)
        .updateAlert(alert.copyWith(read: true));
  }
}

/// show notifications for alerts
void showNotification(
  List<WarnMessage> alerts,
  List<Place> places,
  UserPreferences userPreferences,
  BuildContext context,
  WarningService alertService,
) {
  var localisations = context.localizations;

  for (WarnMessage warning in alerts) {
    var place = places.firstWhere(
      (place) => place.subscriptionId == warning.placeSubscriptionId,
    );

    if ((!warning.read &&
            !warning.notified &&
            !warning.isUpdateOfAlreadyNotifiedWarning) &&
        checkIfEventShouldBeNotified(
          warning.info[0].severity,
          userPreferences,
        )) {
      NotificationService.showNotification(
        // generate from the warning in the List the notification id
        // because the warning identifier is no int, we have to generate a hash code
        id: warning.identifier.hashCode,
        title: localisations.notification_alert_new_title(place.name),
        body: warning.info[0].headline,
        payload: place.name,
        channelId:
            "de.nucleus.foss_warn.notifications_${warning.info[0].severity.name}",
        channelName: warning.info[0].severity.getLocalizedName(context),
      );
      // update notification status for alert
      //@TODO(Nucleus): Can raise an "Tried to use WarningService after `dispose` was called. Consider checking `mounted`. error
      //@TODO(Nucleus): How can we avoid that the WarningService gets disposed?
      if (!alertService.mounted) return;
      alertService.updateAlert(warning.copyWith(notified: true));
    } else if (warning.isUpdateOfAlreadyNotifiedWarning &&
        !warning.notified &&
        !warning.read) {
      NotificationService.showNotification(
        // generate from the warning in the List the notification id
        // because the warning identifier is no int, we have to generate a hash code
        id: warning.identifier.hashCode,
        title: localisations.notification_alert_update_title(place.name),
        body: warning.info[0].headline,
        payload: place.name,
        channelId: "de.nucleus.foss_warn.notifications_update",
        channelName: "update",
      );
    }
  }
}

class WarningService extends StateNotifier<List<WarnMessage>> {
  WarningService({
    required this.userPreferences,
    required this.userPreferencesService,
    required this.places,
  }) : super([]) {
    _loadAlertsFromDisk();
  }

  final UserPreferences userPreferences;
  final UserPreferencesService userPreferencesService;
  final List<Place> places;

  Future<void> _loadAlertsFromDisk() async {
    state = userPreferences.cachedAlerts;
  }

  Future<void> _saveAlertsToDisk() async {
    userPreferencesService.setCachedAlerts(state);
  }

  bool hasWarningToNotify() =>
      state.isNotEmpty &&
      state.any(
        (element) =>
            !element.notified &&
            !element.hideWarningBecauseThereIsANewerVersion &&
            checkIfEventShouldBeNotified(
              element.info[0].severity,
              userPreferences,
            ),
      );

  void updateAlert(WarnMessage alert) {
    var alerts = List<WarnMessage>.from(state);

    if (alerts.contains(alert)) {
      // do not forget to use the return value as new state
      alerts = alerts.updateEntry(alert);
    } else {
      // New from polling
      alerts.add(alert);
    }
    state = alerts;
    _saveAlertsToDisk();
  }

  void deleteAlert(WarnMessage alert) {
    var alerts = List<WarnMessage>.from(state);

    if (alerts.contains(alert)) {
      alerts.remove(alert);
      state = alerts;
    }
    _saveAlertsToDisk();
  }

  /// set the read and notified status from all warnings to false
  /// used for debug purpose
  /// [@ref] to update view
  void resetReadAndNotificationStatusForAllWarnings() {
    state = [
      for (var alert in state) ...[
        alert.copyWith(
          read: false,
          notified: false,
        ),
      ],
    ];
  }
}

/// Return [true] if the user wants a notification - [false] if not.
///
/// The source should be listed in the List notificationSourceSettings.
/// check if the user wants to be notified for
/// the given source and the given severity
///
/// example:
///
/// Warning severity | Notification setting | notification?   <br>
/// Moderate (2)     | Minor (3)            | 3 >= 2 => true  <br>
/// Minor (3)        | Moderate (2)         | 2 >= 3 => false
bool checkIfEventShouldBeNotified(
  Severity severity,
  UserPreferences userPreferences,
) =>
    Severity.getIndexFromSeverity(
      userPreferences.notificationSourceSetting.notificationLevel,
    ) >=
    Severity.getIndexFromSeverity(severity);
