import 'dart:convert';

import 'package:flutter_map/flutter_map.dart';
import 'package:flutter_map_geojson/flutter_map_geojson.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:foss_warn/class/class_app_state.dart';
import 'package:latlong2/latlong.dart';
import 'package:flutter/material.dart';
import 'package:foss_warn/class/class_error_logger.dart';

class Area {
  String description; // general description of the area
  String geoJson; // polygons of the area stored as pure json file

  Area({
    required String areaDesc,
    required this.geoJson,
  }) : description = areaDesc;

  Area.withRegion({
    required String areaDesc,
    required this.geoJson,
  }) : description = areaDesc;

  Area.fromJson(Map<String, dynamic> json)
      : description = json['areaDesc'],
        geoJson = json['geoJson'] ??
            ""; //@TODO(Nucleus) geoJson should never be "" as this produces an error if we try to use the geoJson for the map

  Map<String, dynamic> toJson() => <String, dynamic>{
        'areaDesc': description,
        'geoJson': geoJson,
      };

  /// create a list of area from the stored json data
  static List<Area> areaListFromJson(List<Map<String, dynamic>>? data) {
    List<Area> result = [];
    if (data != null) {
      for (int i = 0; i < data.length; i++) {
        result.add(Area.fromJson(data[i]));
      }
    }
    return result;
  }

  /// create a list of area from CAP Data
  /// converts the CAP geo information into geo json data
  static List<Area> areaListFromJsonWithCAPData(
    List<Map<String, dynamic>>? data,
  ) {
    List<Area> result = [];
    if (data != null) {
      // there a multiple entries => multiple areas
      for (int i = 0; i < data.length; i++) {
        Map<String, dynamic> capToGeoJson = _convertCAPGeoInfoToGeoJson(
          data[i],
        );
        data[i].putIfAbsent("geoJson", () => jsonEncode(capToGeoJson));
        result.add(Area.fromJson(data[i]));
      }
    }
    return result;
  }

  /// create an area from the CAP data
  /// convert he CAP geo info into geo json data
  static Area areaFromJsonWithCAPData(
    Map<String, dynamic> data,
  ) {
    Map<String, dynamic> capToGeoJson = _convertCAPGeoInfoToGeoJson(data);
    data.putIfAbsent("geoJson", () => jsonEncode(capToGeoJson));
    return Area.fromJson(data);
  }

  /// converts tha CAP geo information to geo json
  /// { "type": "FeatureCollection",
  ///   "features": [
  ///     { "type": "Feature",
  ///       "geometry": {
  ///         "type": "Polygon",
  ///         "coordinates": [
  ///           [ [100.0, 0.0], [101.0, 0.0], [101.0, 1.0],
  ///             [100.0, 1.0], [100.0, 0.0] ]
  ///         ]
  ///       },
  ///      ]
  ///    }
  static Map<String, dynamic> _convertCAPGeoInfoToGeoJson(
    Map<String, dynamic> data,
  ) {
    Map<String, dynamic> featureCollection = {};
    featureCollection.putIfAbsent("type", () => "FeatureCollection");

    List<Map<String, dynamic>> features = [];

    // convert polygon information into geojson
    if (data.containsKey("polygon")) {
      Map<String, dynamic> feature = {};
      feature.putIfAbsent("type", () => "Feature");

      Map<String, dynamic> polygonFeature = {};

      if (data["polygon"] is List) {
        // the data contain more then one polygon. So we build a multipolygon
        List<String> polygons = List<String>.from(data["polygon"]);

        // multiple polygons
        // type MultiPolygon
        polygonFeature.putIfAbsent("type", () => "MultiPolygon");

        List<List<List<List<double>>>> coordinatesList = [];

        List<String> latLng = [];
        List<String> coordinates = [];

        for (int i = 0; i < polygons.length; i += 1) {
          coordinates.clear();
          coordinates = polygons[i].split(" ");
          List<List<List<double>>> oneRowCoordinatesOuterList = [];
          List<List<double>> oneRowCoordinates = [];

          for (int j = 0; j < coordinates.length; j += 1) {
            // split with ,
            latLng.clear();
            latLng = coordinates[j].split(",");

            List<double> onePairOfCoordinatesList = [];

            // geo json requires decimal aka double values therefore we
            // parse the value first as double
            onePairOfCoordinatesList.add(double.parse(latLng[1]));
            onePairOfCoordinatesList.add(double.parse(latLng[0]));

            latLng.clear();

            // add the pair to the row List
            oneRowCoordinates.add(onePairOfCoordinatesList);
          }

          oneRowCoordinatesOuterList.clear();
          oneRowCoordinatesOuterList.add(oneRowCoordinates);

          // add the row to the List
          coordinatesList.add(oneRowCoordinatesOuterList);
          // remove not needed storage
        }
        polygonFeature.putIfAbsent("coordinates", () => coordinatesList);

        feature.putIfAbsent("geometry", () => polygonFeature);

        Map<String, dynamic> properties = {};
        properties.putIfAbsent("prop0", () => "value0");
        feature.putIfAbsent("properties", () => properties);
        features.add(feature);
      } else {
        // add type information
        polygonFeature.putIfAbsent("type", () => "Polygon");
        List<List<List<double>>> coordinatesList = [];

        List<List<double>> oneRowCoordinates = [];
        // convert the blank coordinates in CAP format into geojson
        // e.g. "Polygon": "45,-179.99 45,179.99 89.99,179.99 89.99,-179.99 45,-179.99"
        // into
        // { "type": "Polygon",
        //     "coordinates": [
        //         [[45.0, -179.99], [45.0, 179.99], [89.99, 179.9], [89.99, -179.99], [45.0, -179.99]]
        //     ]
        // }
        // split with spaces and remove all [ ] which are maybe a result of .toString @todo
        var coordinates =
            (data["polygon"].toString().replaceAll("[", "").replaceAll("]", ""))
                .split(" ");

        for (var coordinate in coordinates) {
          // split with ,
          List<String> latLng = coordinate.split(",");
          List<double> onePairOfCoordinatesList = [];
          // geo json requires decimal aka double values therefore we
          // parse the value first as double and den convert it again as String
          onePairOfCoordinatesList.add(double.parse(latLng[1]));
          onePairOfCoordinatesList.add(double.parse(latLng[0]));

          // add the pair to the row List
          oneRowCoordinates.add(onePairOfCoordinatesList);
        }

        // add the row to the List
        coordinatesList.add(oneRowCoordinates);

        polygonFeature.putIfAbsent("coordinates", () => coordinatesList);

        feature.putIfAbsent("geometry", () => polygonFeature);

        Map<String, dynamic> properties = {};
        properties.putIfAbsent("prop0", () => "value0");
        feature.putIfAbsent("properties", () => properties);
        features.add(feature);
      }
    }
    featureCollection.putIfAbsent("features", () => features);

    return featureCollection;
  }

  /// create a list with all latLon for all geoJsonFeatures
  static List<LatLng> getListWithAllPolygons(List<Area> areas, WidgetRef ref) {
    List<LatLng> result = [];

    List<Polygon> polygons = createListOfPolygonsForAreas(areas, ref);

    for (Polygon i in polygons) {
      result.addAll(i.points);
    }

    return result;
  }

  /// create a list of polygons from a list of areas
  //  default color: 0xFFB01917
  //  default borderColor: 0xFFFB8C00
  static List<Polygon> createListOfPolygonsForAreas(
    List<Area> areas,
    WidgetRef ref,
  ) {
    List<Polygon> result = [];
    try {
      GeoJsonParser myGeoJson = GeoJsonParser(
        defaultPolygonFillColor: const Color(0xFFB01917).withValues(alpha: 0.2),
        defaultPolygonBorderColor: const Color(0xFFFB8C00),
        defaultPolylineStroke: 1,
      );
      for (Area area in areas) {
        myGeoJson.parseGeoJsonAsString(area.geoJson);
      }
      result.addAll(myGeoJson.polygons);
      return result;
    } catch (e) {
      ErrorLogger.writeErrorLog(
        "MapWidget",
        "Error while parsing geoJson",
        e.toString(),
      );
      var appStateService = ref.read(appStateProvider.notifier);
      appStateService.setError(true);
      return [];
    }
  }
}
