/*
 * Copyright (C) 2011 Andrew Karpow <andy@mail.tu-berlin.de>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.k4ever.k4android.utils;

import android.util.Base64;
import android.util.LruCache;
import com.android.volley.*;
import com.android.volley.toolbox.*;

import android.content.Context;
import android.content.SharedPreferences;
import android.graphics.Bitmap;
import android.preference.PreferenceManager;
import de.k4ever.k4android.R;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;

import java.io.InputStream;
import java.security.KeyStore;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.Map;

public class KassenHttpUtils {
    private static KassenHttpUtils mInstance;
    private static Context mCtx;
    private RequestQueue mRequestQueue;
    private ImageLoader mImageLoader;
    private static String mAddress;

	public static void destroyInstance() {
		mInstance = null;
	}

    public static String getUrl(String url, Context context) {
        return getInstance(context).mAddress + url;
    }

	public static String getMediaURL(Context context) {
		final SharedPreferences prefs = PreferenceManager.getDefaultSharedPreferences(context);
		return prefs.getString("pref_url", "") + KassenConfig.KASSEN_SUBDIR_MEDIA;
	}

    private KassenHttpUtils(Context context) {
        mCtx = context;
        mRequestQueue = getRequestQueue();
        final SharedPreferences prefs = PreferenceManager.getDefaultSharedPreferences(context);
        setAPIUrl(prefs.getString("pref_url", "") + KassenConfig.KASSEN_SUBDIR_API);

        mImageLoader = new ImageLoader(mRequestQueue,
                new ImageLoader.ImageCache() {
                    private final LruCache<String, Bitmap>
                            cache = new LruCache<String, Bitmap>(20);

                    @Override
                    public Bitmap getBitmap(String url) {
                        return cache.get(url);
                    }

                    @Override
                    public void putBitmap(String url, Bitmap bitmap) {
                        cache.put(url, bitmap);
                    }
                });
    }

    public static synchronized KassenHttpUtils getInstance(Context ctx) {
        if (mInstance == null) {
            mInstance = new KassenHttpUtils(ctx);
        }
        return mInstance;
	}

    public RequestQueue getRequestQueue() {
        if (mRequestQueue == null) {
			SSLContext ctx;

            try {
                // get default trust manager
                TrustManagerFactory tmf = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
                tmf.init((KeyStore) null);

                X509TrustManager defaultTrustManager = null;
                for (TrustManager tm : tmf.getTrustManagers()) {
                    if (tm instanceof X509TrustManager) {
                        defaultTrustManager = (X509TrustManager) tm;
                        break;
                    }
                }

                // load custom trust manager
                KeyStore customKeyStore = KeyStore.getInstance("BKS");
                final InputStream in = mCtx.getResources().openRawResource(R.raw.truststore);
                customKeyStore.load(in, "freitagsrunde".toCharArray());
                in.close();
                TrustManagerFactory customTrustManagerFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
                customTrustManagerFactory.init(customKeyStore);

                X509TrustManager customTrustManager = null;
                for (TrustManager tm : customTrustManagerFactory.getTrustManagers()) {
                    if (tm instanceof X509TrustManager) {
                        customTrustManager = (X509TrustManager) tm;
                        break;
                    }
                }

                // create new trust manager using custom and default TM
                final X509TrustManager finalDefaultTrustManager = defaultTrustManager;
                final X509TrustManager finalCustomTrustManager = customTrustManager;
                X509TrustManager customMergedTrustManager = new X509TrustManager() {
                    @Override
                    public void checkClientTrusted(X509Certificate[] x509Certificates, String s) throws CertificateException {
                        finalDefaultTrustManager.checkClientTrusted(x509Certificates, s);
                    }

                    @Override
                    public void checkServerTrusted(X509Certificate[] x509Certificates, String s) throws CertificateException {
                        try {
                            finalCustomTrustManager.checkServerTrusted(x509Certificates, s);
                        } catch (CertificateException e) {
                            finalDefaultTrustManager.checkServerTrusted(x509Certificates, s);
                        }
                    }

                    @Override
                    public X509Certificate[] getAcceptedIssuers() {
                        return finalDefaultTrustManager.getAcceptedIssuers();
                    }
                };

                ctx = SSLContext.getInstance("TLS");
                ctx.init(null, new TrustManager[] { customMergedTrustManager }, null);
            } catch (Exception e) {
                throw new AssertionError(e);
            }

			mRequestQueue = Volley.newRequestQueue(
					mCtx.getApplicationContext(),
					new HurlStack(
							null,
							ctx.getSocketFactory()
					)
			);
        }
        return mRequestQueue;
    }

    public <T> void addToRequestQueue(Request<T> req) {
        getRequestQueue().add(req);
    }

    public ImageLoader getImageLoader() {
        return mImageLoader;
    }

	public static void setAPIUrl(String address) {
		if(!address.endsWith("/"))
			address += "/";
		mAddress = address;
	}

	public static Map<String, String> getAuthHeader(Context ctx) {
		Map<String, String> headerMap = new HashMap<String, String>();
		SharedPreferences prefs = PreferenceManager.getDefaultSharedPreferences(ctx);
		String username = prefs.getString("pref_username", "");
		String password = prefs.getString("pref_password", "");

		String credentials = username + ":" + password;
		String base64EncodedCredentials =
				Base64.encodeToString(credentials.getBytes(), Base64.NO_WRAP);
		headerMap.put("Authorization", "Basic " + base64EncodedCredentials);

		return headerMap;
	}
}