package de.fff.ccgt.activity;

import android.annotation.SuppressLint;
import android.content.Intent;
import android.graphics.Color;
import android.os.Bundle;
import android.os.Handler;
import android.util.Log;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.WindowManager;
import android.widget.AdapterView;
import android.widget.SeekBar;
import android.widget.Spinner;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.appcompat.app.AppCompatActivity;

import java.lang.ref.WeakReference;
import java.util.Objects;

import be.tarsos.dsp.AudioEvent;
import be.tarsos.dsp.AudioProcessor;
import be.tarsos.dsp.pitch.PitchDetectionHandler;
import be.tarsos.dsp.pitch.PitchProcessor;
import be.tarsos.dsp.util.fft.FFT;
import de.fff.ccgt.R;
import de.fff.ccgt.service.AudioService;
import de.fff.ccgt.view.ConsoleBuffer;
import de.fff.ccgt.service.PitchService;
import de.fff.ccgt.service.PreferencesService;
import de.fff.ccgt.view.SpectrogramView;

/*
 * ccgt -  console curses guitar tuner
 *
 * JorenSix/TarsosDSP is licensed under the
 * GNU General Public License v3.0
 * Permissions of this strong copyleft license
 * are conditioned on making available complete
 * source code of licensed works and modifications,
 * which include larger works using a licensed work,
 * under the same license. Copyright and license
 * notices must be preserved. Contributors provide
 * an express grant of patent rights.
 */

public class MainActivity extends AppCompatActivity {

    private final static String TAG = MainActivity.class.getSimpleName();

    private AudioService audioService;
    private PitchService pitchService;
    private ConsoleBuffer consoleBuffer;
    private PreferencesService preferencesService;

    private float pitchInHz = 0;
    private double centsDeviation = 0;

    private SpectrogramView spectrogramView;
    private TextView pitchNameTV;
    private TextView octTV;
    private TextView freqTV;
    private TextView consoleTV;
    private Spinner calibSpinner;
    private SeekBar calibSeekBar;

    private final static Handler displayHandler = new Handler();
    private Thread displayUpdateThread = null;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);

        Objects.requireNonNull(getSupportActionBar()).setTitle(R.string.title_action_bar);

        spectrogramView = findViewById(R.id.spectrogram);
        pitchNameTV = findViewById(R.id.note);
        octTV = findViewById(R.id.octave);
        freqTV = findViewById(R.id.freq);
        freqTV.setTextColor(Color.WHITE);
        calibSpinner = findViewById(R.id.spinner);
        calibSeekBar = findViewById(R.id.calibrationSeekBar);
        consoleTV = findViewById(R.id.console);

        preferencesService = new PreferencesService(this.getApplicationContext());
        pitchService = new PitchService();
        audioService = new AudioService(this.getApplicationContext());
        consoleBuffer = new ConsoleBuffer();

        audioService.startAudio(preferencesService.getSampleRate(), preferencesService.getBufferSize(), preferencesService.getAlgorithm(), getPitchDetectionHandler(), getFftProcessor());

        initCalibration(true);
        startDisplay();
        handlePreferences();

    }



    private void handlePreferences() {
        if(preferencesService.isKeepScreenOn()) {
            getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
        } else {
            getWindow().clearFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
        }

        octTV.setTextColor(preferencesService.isShowOctave() ? Color.WHITE : Color.BLACK);

        spectrogramView.setSpectrogramLogarithmic(preferencesService.isSpectrogramLogarithmic());
        spectrogramView.setSamplerate(preferencesService.getSampleRate());
    }

    private void initCalibration(boolean setListener) {
        calibSpinner.setBackgroundColor(Color.DKGRAY);
        int index = preferencesService.getCalibrationFreqIndex();
        calibSpinner.setSelection(index);
        calibSeekBar.setProgress(index);

        if(setListener) {
            calibSeekBar.setOnSeekBarChangeListener(
                    new SeekBar.OnSeekBarChangeListener() {
                        int progress = 0;

                        @Override
                        public void onProgressChanged(SeekBar seekBar,
                                                      int progressValue, boolean fromUser) {
                            progress = progressValue;
                            calibSpinner.setSelection(progress);
                            preferencesService.setCalibrationFreqByIndex(progress);
                        }

                        @Override
                        public void onStartTrackingTouch(SeekBar seekBar) {
                        }

                        @Override
                        public void onStopTrackingTouch(SeekBar seekBar) {
                        }
                    });

            calibSpinner.setOnItemSelectedListener(
                    new AdapterView.OnItemSelectedListener() {
                        int selection = 0;

                        @Override
                        public void onItemSelected(AdapterView<?> parent, View view, int position, long id) {
                            selection = position;
                            calibSeekBar.setProgress(selection);
                            preferencesService.setCalibrationFreqByIndex(selection);
                        }

                        @Override
                        public void onNothingSelected(AdapterView<?> parent) {
                        }
                    }
            );
        }

    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        MenuInflater inflater = getMenuInflater();
        inflater.inflate(R.menu.ccgt_menu, menu);
        return true;
    }

    @Override
    public boolean onOptionsItemSelected(@NonNull MenuItem item) {
        if (item.getItemId() == R.id.settings) {
            startActivity(new Intent(this, SettingsActivity.class));
        }
        return super.onOptionsItemSelected(item);
    }

    private void startDisplay() {
        displayUpdateThread = new Thread(() -> {
            while (true) {
                try {
                    displayHandler.post(new UpdateConsoleRunnable(this));
                    Thread.sleep(preferencesService.getDisplayWaitTime());
                } catch (InterruptedException e) {
                    //e.printStackTrace();
                }
            }
        });
        displayUpdateThread.start();
    }

    @Override
    protected void onPause() {
        audioService.stopAudio();
        super.onPause();
    }

    @Override
    protected void onResume() {
        super.onResume();
        initCalibration(false);
        audioService.startAudio(preferencesService.getSampleRate(), preferencesService.getBufferSize(), preferencesService.getAlgorithm(), getPitchDetectionHandler(), getFftProcessor());
        handlePreferences();
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        if(displayUpdateThread != null) {
            displayUpdateThread.interrupt();
        }
        audioService.stopAudio();
    }

    public PitchDetectionHandler getPitchDetectionHandler() {
        @SuppressLint("DefaultLocale")
        PitchDetectionHandler pitchDetectionHandler = (pitchDetectionResult, audioEvent) -> {
            int referenceFrequency = preferencesService.getCalibrationFreq();
            runOnUiThread(() -> {
                pitchNameTV.setTextColor(pitchService.color(pitchDetectionResult, referenceFrequency));
                pitchNameTV.setText(pitchService.getNearestPitchClass(pitchDetectionResult, referenceFrequency));
                octTV.setText(pitchService.getOctave(pitchDetectionResult.getPitch(), referenceFrequency));
                freqTV.setText(String.format("%.02f", pitchDetectionResult.getPitch()));

                // TODO: 02.02.26 remove dependency on centsDeviation field
                centsDeviation = pitchService.isPitched(pitchDetectionResult) ? pitchService.getCentsDeviation(pitchDetectionResult.getPitch(), referenceFrequency) : Double.NaN;
            });
        };

        return pitchDetectionHandler;
    }

    public AudioProcessor getFftProcessor() {
        AudioProcessor fftProcessor = new AudioProcessor() {
            int buffersize = preferencesService.getBufferSize();
            final FFT fft = new FFT(buffersize);
            float[] amplitudes = new float[buffersize / 2];

            @Override
            public boolean process(AudioEvent audioEvent) {
                float[] audioFloatBuffer = audioEvent.getFloatBuffer().clone();
                fft.forwardTransform(audioFloatBuffer);
                //modulus: absolute value of complex fourier coefficient aka magnitude
                fft.modulus(audioFloatBuffer, amplitudes);
                runOnUiThread(() -> { // TODO: 02.02.26 pitchInHz currently not provided
                    spectrogramView.feedSpectrogramView(pitchInHz, amplitudes);
                });
                return true;
            }

            @Override
            public void processingFinished() {
                    Log.d(TAG, "processingFinished: fftProcessor");
            }
        };

        return fftProcessor;
    }

    private static class UpdateConsoleRunnable implements Runnable
    {
        private final WeakReference<MainActivity> mainActivityWeakReference;

        public UpdateConsoleRunnable(MainActivity myClassInstance)
        {
            mainActivityWeakReference = new WeakReference(myClassInstance);
        }

        @Override
        public void run()
        {
            MainActivity mainActivity = mainActivityWeakReference.get();
            if(mainActivity != null) {
                mainActivity.getConsoleTV().setText(mainActivity.getConsoleBuffer().push(mainActivity.getCentsDeviation()));
            }
        }
    }

    private ConsoleBuffer getConsoleBuffer() {
        return consoleBuffer;
    }

    private TextView getConsoleTV() {
        return consoleTV;
    }

    private double getCentsDeviation() {
        return centsDeviation;
    }

}
