package de.deftk.openww.android.fragments.feature.filestorage

import android.app.Activity
import android.app.AlertDialog
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.os.Bundle
import android.text.InputType
import android.view.*
import android.widget.EditText
import android.widget.SearchView
import android.widget.Toast
import androidx.activity.result.ActivityResult
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContract
import androidx.appcompat.view.ActionMode
import androidx.core.view.isVisible
import androidx.fragment.app.activityViewModels
import androidx.navigation.fragment.findNavController
import androidx.navigation.fragment.navArgs
import androidx.preference.PreferenceManager
import androidx.recyclerview.widget.DividerItemDecoration
import androidx.work.WorkInfo
import androidx.work.WorkManager
import de.deftk.openww.android.R
import de.deftk.openww.android.adapter.recycler.ActionModeAdapter
import de.deftk.openww.android.adapter.recycler.FileAdapter
import de.deftk.openww.android.api.Response
import de.deftk.openww.android.components.ContextMenuRecyclerView
import de.deftk.openww.android.databinding.FragmentFilesBinding
import de.deftk.openww.android.feature.AbstractNotifyingWorker
import de.deftk.openww.android.feature.LaunchMode
import de.deftk.openww.android.feature.filestorage.DownloadOpenWorker
import de.deftk.openww.android.feature.filestorage.FileCacheElement
import de.deftk.openww.android.feature.filestorage.NetworkTransfer
import de.deftk.openww.android.feature.filestorage.SessionFileUploadWorker
import de.deftk.openww.android.filter.FileStorageFileFilter
import de.deftk.openww.android.fragments.ActionModeFragment
import de.deftk.openww.android.utils.FileUtil
import de.deftk.openww.android.utils.ISearchProvider
import de.deftk.openww.android.utils.Reporter
import de.deftk.openww.android.viewmodel.FileStorageViewModel
import de.deftk.openww.android.viewmodel.UserViewModel
import de.deftk.openww.api.WebWeaverClient
import de.deftk.openww.api.implementation.feature.filestorage.session.SessionFile
import de.deftk.openww.api.model.Feature
import de.deftk.openww.api.model.IOperatingScope
import de.deftk.openww.api.model.feature.filestorage.FileType
import de.deftk.openww.api.model.feature.filestorage.IRemoteFile
import kotlinx.serialization.decodeFromString
import java.io.File
import kotlin.math.max

class FilesFragment : ActionModeFragment<IRemoteFile, FileAdapter.FileViewHolder>(R.menu.filestorage_actionmode_menu), ISearchProvider {

    //TODO cancel ongoing network transfers on account switch

    private val args: FilesFragmentArgs by navArgs()
    private val userViewModel: UserViewModel by activityViewModels()
    private val fileStorageViewModel: FileStorageViewModel by activityViewModels()
    private val preferences by lazy { PreferenceManager.getDefaultSharedPreferences(requireContext()) }
    private val workManager by lazy { WorkManager.getInstance(requireContext()) }
    private val navController by lazy { findNavController() }

    private lateinit var downloadSaveLauncher: ActivityResultLauncher<Pair<Intent, IRemoteFile>>
    private lateinit var uploadLauncher: ActivityResultLauncher<Array<String>>
    private lateinit var binding: FragmentFilesBinding
    private lateinit var searchView: SearchView

    private var scope: IOperatingScope? = null
    private var folderId: String? = null
    private var currentNetworkTransfers = emptyList<NetworkTransfer>()

    override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?): View {
        binding = FragmentFilesBinding.inflate(inflater, container, false)

        fileStorageViewModel.networkTransfers.observe(viewLifecycleOwner) { transfers ->
            for (i in 0 until max(transfers.size, currentNetworkTransfers.size)) {
                if (i < transfers.size && !currentNetworkTransfers.contains(transfers[i])) {
                    // handle new transfer
                    val transfer = transfers[i]
                    onNetworkTransferAdded(transfer)
                    continue
                }
                if (i < currentNetworkTransfers.size && !transfers.contains(currentNetworkTransfers[i])) {
                    // handle removed transfer
                    val transfer = currentNetworkTransfers[i]
                    onNetworkTransferRemoved(transfer)
                    continue
                }
            }
            currentNetworkTransfers = transfers
        }

        fileStorageViewModel.batchDeleteResponse.observe(viewLifecycleOwner) { response ->
            if (response != null)
                fileStorageViewModel.resetBatchDeleteResponse()
            enableUI(true)

            val failure = response?.filterIsInstance<Response.Failure>() ?: return@observe
            if (failure.isNotEmpty()) {
                Reporter.reportException(R.string.error_delete_failed, failure.first().exception, requireContext())
            } else {
                actionMode?.finish()
            }
        }

        fileStorageViewModel.importSessionFile.observe(viewLifecycleOwner) { response ->
            if (response != null)
                fileStorageViewModel.resetImportSessionFileResponse()
            enableUI(true)

            if (response is Response.Failure) {
                Reporter.reportException(R.string.error_import_session_file_failed, response.exception, requireContext())
            }
        }

        fileStorageViewModel.addFolderResponse.observe(viewLifecycleOwner) { response ->
            if (response != null)
                fileStorageViewModel.resetAddFolderResponse()
            enableUI(true)

            if (response is Response.Failure) {
                Reporter.reportException(R.string.error_add_folder_failed, response.exception, requireContext())
            }
        }

        binding.fileStorageSwipeRefresh.setOnRefreshListener {
            userViewModel.apiContext.value?.also { apiContext ->
                fileStorageViewModel.cleanCache(scope!!)
                fileStorageViewModel.loadChildrenTree(scope!!, folderId!!, true, apiContext)
                adapter.notifyDataSetChanged() // update previews
            }
        }

        userViewModel.apiContext.observe(viewLifecycleOwner) { apiContext ->
            if (apiContext != null) {
                val newScope = userViewModel.apiContext.value?.findOperatingScope(args.operatorId)
                if (newScope == null) {
                    Reporter.reportException(R.string.error_scope_not_found, args.operatorId, requireContext())
                    navController.popBackStack(R.id.fileStorageGroupFragment, false)
                    return@observe
                }
                if (!Feature.FILES.isAvailable(newScope.effectiveRights)) {
                    Reporter.reportFeatureNotAvailable(requireContext())
                    navController.popBackStack(R.id.fileStorageGroupFragment, true)
                    return@observe
                }

                if (scope != null) {
                    fileStorageViewModel.getFilteredFiles(scope!!).removeObservers(viewLifecycleOwner)
                    scope = newScope
                    (adapter as FileAdapter).scope = newScope
                } else {
                    scope = newScope
                }
                binding.fileList.adapter = adapter
                binding.fileList.addItemDecoration(DividerItemDecoration(requireContext(), DividerItemDecoration.VERTICAL))
                binding.fileList.recycledViewPool.setMaxRecycledViews(0, 0) // this is just a workaround (otherwise preview images disappear while scrolling, see https://github.com/square/picasso/issues/845#issuecomment-280626688) FIXME seems like an issue with recycling

                val filter = FileStorageFileFilter()
                if (args.folderNameId == null) {
                    folderId = args.folderId
                    filter.parentCriteria.value = folderId
                    fileStorageViewModel.fileFilter.value = filter
                }
                fileStorageViewModel.getFilteredFiles(scope!!).observe(viewLifecycleOwner) { response ->
                    if (response is Response.Success) {
                        if (args.folderNameId != null && folderId == null) {
                            folderId = fileStorageViewModel.resolveNameTree(scope!!, args.folderNameId!!)
                            if (folderId != null) {
                                filter.parentCriteria.value = folderId
                                fileStorageViewModel.fileFilter.value = filter
                                return@observe
                            }
                        }
                        adapter.submitList(response.value.map { it.file })
                        binding.fileEmpty.isVisible = response.value.isEmpty()
                        updateUploadFab()
                        requireActivity().invalidateOptionsMenu()

                        if (args.highlightFileId != null && response.value.isNotEmpty()) {
                            val highlightFileName = args.highlightFileId!!.substring(1)
                            val file = response.value.firstOrNull { it.file.name == highlightFileName }
                            if (file != null) {
                                val position = adapter.currentList.indexOf(file.file)
                                if (position != -1)
                                    binding.fileList.smoothScrollToPosition(position)
                            }
                        }
                    } else if (response is Response.Failure) {
                        Reporter.reportException(R.string.error_get_files_failed, response.exception, requireContext())
                    }
                    enableUI(true)
                    binding.fileStorageSwipeRefresh.isRefreshing = false
                }

                if (args.folderNameId != null) {
                    fileStorageViewModel.loadChildrenNameTree(scope!!, args.folderNameId!!, false, apiContext)
                    updateUploadFab()
                    enableUI(false)
                } else {
                    if (fileStorageViewModel.getAllFiles(scope!!).value?.valueOrNull()?.any { it.file.parentId == folderId } == true) {
                        fileStorageViewModel.loadChildren(scope!!, folderId!!, false, apiContext)
                    } else {
                        fileStorageViewModel.loadChildrenTree(scope!!, folderId!!, false, apiContext)
                    }
                    updateUploadFab()
                    if (fileStorageViewModel.getCachedChildren(scope!!, folderId!!).isEmpty())
                        enableUI(false)
                }
            } else {
                binding.fabUploadFile.isVisible = false
                binding.fileEmpty.isVisible = false
                binding.fileStorageSwipeRefresh.isRefreshing = false
                adapter.submitList(emptyList())
            }
        }

        downloadSaveLauncher = registerForActivityResult(SaveFileContract()) { (result, file) ->
            val uri = result.data?.data
            userViewModel.apiContext.value?.also { apiContext ->
                fileStorageViewModel.startSaveDownload(workManager, apiContext, file, scope!!, uri.toString())
            }
        }

        uploadLauncher = registerForActivityResult(OpenDocumentsContract()) { uris ->
            uris?.forEach { uri ->
                uploadFile(uri)
            }
        }

        binding.fabUploadFile.setOnClickListener {
            if (args.pasteMode) {
                //TODO don't rely on intent (functionality should be reused for internal copy-paste/cut operations)
                val intent = requireActivity().intent
                if (intent.action == Intent.ACTION_SEND) {
                    uploadFile(requireActivity().intent.getParcelableExtra(Intent.EXTRA_STREAM)!!)
                } else if (intent.action == Intent.ACTION_SEND_MULTIPLE) {
                    requireActivity().intent.getParcelableArrayListExtra<Uri>(Intent.EXTRA_STREAM)!!.forEach { uri ->
                        uploadFile(uri)
                    }
                }

                binding.fabUploadFile.isVisible = false
            } else {
                uploadLauncher.launch(arrayOf("*/*"))
            }
        }

        setHasOptionsMenu(true)
        registerForContextMenu(binding.fileList)
        return binding.root
    }

    private fun updateUploadFab() {
        binding.fabUploadFile.isVisible = getProviderFile()?.file?.effectiveCreate == true
        if (args.pasteMode) {
            binding.fabUploadFile.setImageResource(R.drawable.ic_content_paste_24)
        } else {
            binding.fabUploadFile.setImageResource(R.drawable.ic_add_24)
        }
    }

    private fun getProviderFile(): FileCacheElement? {
        return fileStorageViewModel.getAllFiles(scope!!).value?.valueOrNull()?.firstOrNull { it.file.id == folderId || (it.file.id == "" && folderId == "/") }
    }

    override fun createAdapter(): ActionModeAdapter<IRemoteFile, FileAdapter.FileViewHolder> {
        return FileAdapter(scope!!, this, fileStorageViewModel)
    }

    private fun onNetworkTransferAdded(transfer: NetworkTransfer) {
        val liveData = workManager.getWorkInfoByIdLiveData(transfer.workerId)
        when (transfer) {
            is NetworkTransfer.DownloadOpen -> {
                liveData.observe(viewLifecycleOwner) { workInfo ->
                    val adapterIndex = adapter.currentList.indexOfFirst { it.id == transfer.id }
                    var progress = workInfo.progress.getInt(AbstractNotifyingWorker.ARGUMENT_PROGRESS, 0)
                    when (workInfo.state) {
                        WorkInfo.State.SUCCEEDED -> {
                            progress = 100
                            val fileUri = Uri.parse(workInfo.outputData.getString(DownloadOpenWorker.DATA_FILE_URI))
                            val fileName = workInfo.outputData.getString(DownloadOpenWorker.DATA_FILE_NAME)!!
                            FileUtil.showFileOpenIntent(fileName, fileUri, preferences, requireContext())
                            fileStorageViewModel.hideNetworkTransfer(transfer, scope!!)
                        }
                        WorkInfo.State.CANCELLED -> {
                            //TODO remove notification
                            progress = -1
                            fileStorageViewModel.hideNetworkTransfer(transfer, scope!!)
                        }
                        WorkInfo.State.FAILED -> {
                            val message = workInfo.outputData.getString(AbstractNotifyingWorker.DATA_ERROR_MESSAGE) ?: "Unknown"
                            Reporter.reportException(R.string.error_download_worker_failed, message, requireContext())
                            progress = -1
                            fileStorageViewModel.hideNetworkTransfer(transfer, scope!!)
                        }
                        else -> { /* ignore */ }
                    }
                    transfer.progress = progress
                    val viewHolder = binding.fileList.findViewHolderForAdapterPosition(adapterIndex) as FileAdapter.FileViewHolder
                    viewHolder.setProgress(progress)
                }
            }
            is NetworkTransfer.DownloadSave -> {
                liveData.observe(viewLifecycleOwner) { workInfo ->
                    val adapterIndex = adapter.currentList.indexOfFirst { it.id == transfer.id }
                    var progress = workInfo.progress.getInt(AbstractNotifyingWorker.ARGUMENT_PROGRESS, 0)
                    when (workInfo.state) {
                        WorkInfo.State.SUCCEEDED -> {
                            progress = 100
                            Toast.makeText(requireContext(), R.string.download_finished, Toast.LENGTH_LONG).show()
                            fileStorageViewModel.hideNetworkTransfer(transfer, scope!!)
                        }
                        WorkInfo.State.CANCELLED -> {
                            //TODO remove notification
                            progress = -1
                            fileStorageViewModel.hideNetworkTransfer(transfer, scope!!)
                        }
                        WorkInfo.State.FAILED -> {
                            val message = workInfo.outputData.getString(AbstractNotifyingWorker.DATA_ERROR_MESSAGE) ?: "Unknown"
                            Reporter.reportException(R.string.error_download_worker_failed, message, requireContext())
                            progress = -1
                            fileStorageViewModel.hideNetworkTransfer(transfer, scope!!)
                        }
                        else -> { /* ignore */ }
                    }
                    transfer.progress = progress
                    val viewHolder = binding.fileList.findViewHolderForAdapterPosition(adapterIndex) as FileAdapter.FileViewHolder
                    viewHolder.setProgress(progress)
                }
            }
            is NetworkTransfer.Upload -> {
                liveData.observe(viewLifecycleOwner) { workInfo ->
                    val adapterIndex = adapter.currentList.indexOfFirst { it.id == transfer.id }
                    var progress = workInfo.progress.getInt(AbstractNotifyingWorker.ARGUMENT_PROGRESS, 0)
                    when (workInfo.state) {
                        WorkInfo.State.SUCCEEDED -> {
                            progress = 100
                            Toast.makeText(requireContext(), R.string.upload_finished, Toast.LENGTH_LONG).show()
                            fileStorageViewModel.hideNetworkTransfer(transfer, scope!!)
                            val sessionFile = WebWeaverClient.json.decodeFromString<SessionFile>(workInfo.outputData.getString(SessionFileUploadWorker.DATA_SESSION_FILE) ?: "")
                            userViewModel.apiContext.value?.also { apiContext ->
                                fileStorageViewModel.importSessionFile(sessionFile, scope!!, apiContext)
                            }
                        }
                        WorkInfo.State.CANCELLED -> {
                            //TODO remove notification
                            progress = -1
                            fileStorageViewModel.hideNetworkTransfer(transfer, scope!!)
                        }
                        WorkInfo.State.FAILED -> {
                            val message = workInfo.outputData.getString(AbstractNotifyingWorker.DATA_ERROR_MESSAGE) ?: "Unknown"
                            Reporter.reportException(R.string.error_download_worker_failed, message, requireContext())
                            progress = -1
                            fileStorageViewModel.hideNetworkTransfer(transfer, scope!!)
                        }
                        else -> { /* ignore */ }
                    }
                    transfer.progress = progress
                    val viewHolder = binding.fileList.findViewHolderForAdapterPosition(adapterIndex) as FileAdapter.FileViewHolder
                    viewHolder.setProgress(progress)
                }
            }
        }
    }

    private fun onNetworkTransferRemoved(transfer: NetworkTransfer) {
        if (LaunchMode.getLaunchMode(requireActivity().intent) == LaunchMode.FILE_UPLOAD && fileStorageViewModel.networkTransfers.value?.isEmpty() != false && transfer is NetworkTransfer.Upload) {
            requireActivity().finish()
        }
    }

    override fun onCreateOptionsMenu(menu: Menu, inflater: MenuInflater) {
        menu.clear()
        inflater.inflate(R.menu.filestorage_options_menu, menu)
        inflater.inflate(R.menu.list_options_menu, menu)
        val searchItem = menu.findItem(R.id.list_options_item_search)
        searchView = searchItem.actionView as SearchView
        searchView.setQuery(fileStorageViewModel.fileFilter.value?.smartSearchCriteria?.value, false) // restore recent search
        searchView.setOnQueryTextListener(object : SearchView.OnQueryTextListener {
            override fun onQueryTextSubmit(query: String?): Boolean {
                searchView.clearFocus()
                return true
            }

            override fun onQueryTextChange(newText: String?): Boolean {
                val filter = FileStorageFileFilter()
                filter.smartSearchCriteria.value = newText
                filter.parentCriteria.value = folderId
                fileStorageViewModel.fileFilter.value = filter
                return true
            }
        })
    }

    override fun onPrepareOptionsMenu(menu: Menu) {
        menu.findItem(R.id.filestorage_option_item_create_folder).isVisible = getProviderFile()?.file?.effectiveCreate == true
    }

    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        when (item.itemId) {
            R.id.filestorage_option_item_create_folder -> {
                val builder = AlertDialog.Builder(requireContext())
                builder.setTitle(R.string.create_new_folder)

                val input = EditText(requireContext())
                input.hint = getString(R.string.name)
                input.inputType = InputType.TYPE_CLASS_TEXT
                builder.setView(input)

                builder.setPositiveButton(R.string.confirm) { _, _ ->
                    userViewModel.apiContext.value?.apply {
                        fileStorageViewModel.addFolder(input.text.toString(), getProviderFile()!!.file, scope!!, this)
                        enableUI(false)
                    }
                }
                builder.setNegativeButton(R.string.cancel) { dialog, _ ->
                    dialog.cancel()
                }
                builder.create().show()
            }
            else -> return super.onOptionsItemSelected(item)
        }
        return true
    }

    override fun onSearchBackPressed(): Boolean {
        return if (searchView.isIconified) {
            false
        } else {
            searchView.isIconified = true
            searchView.setQuery(null, true)
            true
        }
    }

    override fun onCreateContextMenu(menu: ContextMenu, v: View, menuInfo: ContextMenu.ContextMenuInfo?) {
        requireActivity().menuInflater.inflate(R.menu.filestorage_context_menu, menu)
        if (menuInfo is ContextMenuRecyclerView.RecyclerViewContextMenuInfo) {
            val file = (binding.fileList.adapter as FileAdapter).getItem(menuInfo.position)
            val canRead = file.effectiveRead == true && file.type == FileType.FILE
            menu.findItem(R.id.filestorage_context_item_open).isVisible = canRead
            menu.findItem(R.id.filestorage_context_item_download).isVisible = canRead
            menu.findItem(R.id.filestorage_context_item_delete).isVisible = file.effectiveDelete == true
        }
    }

    override fun onContextItemSelected(item: MenuItem): Boolean {
        val menuInfo = item.menuInfo as ContextMenuRecyclerView.RecyclerViewContextMenuInfo
        val adapter = binding.fileList.adapter as FileAdapter
        return when (item.itemId) {
            R.id.filestorage_context_item_delete -> {
                val file = adapter.getItem(menuInfo.position)
                userViewModel.apiContext.value?.also { apiContext ->
                    fileStorageViewModel.batchDelete(listOf(file), scope!!, apiContext)
                }
                true
            }
            R.id.filestorage_context_item_info -> {
                val file = adapter.getItem(menuInfo.position)
                navController.navigate(FilesFragmentDirections.actionFilesFragmentToReadFileFragment(args.operatorId, file.id, folderId!!))
                true
            }
            R.id.filestorage_context_item_download -> {
                val file = adapter.getItem(menuInfo.position)
                val intent = Intent(Intent.ACTION_CREATE_DOCUMENT)
                intent.type = FileUtil.getMimeType(file.name)
                intent.putExtra(Intent.EXTRA_TITLE, file.name)
                downloadSaveLauncher.launch(intent to file)
                true
            }
            R.id.filestorage_context_item_open -> {
                val file = adapter.getItem(menuInfo.position)
                openFile(file)
                true
            }
            else -> super.onContextItemSelected(item)
        }
    }

    override fun onPrepareActionMode(mode: ActionMode, menu: Menu): Boolean {
        val canDelete = adapter.selectedItems.all { it.binding.file!!.effectiveDelete == true }
        menu.findItem(R.id.filestorage_action_item_delete).isEnabled = canDelete
        return super.onPrepareActionMode(mode, menu)
    }

    override fun onActionItemClicked(mode: ActionMode, item: MenuItem): Boolean {
        when (item.itemId) {
            R.id.filestorage_action_item_delete -> {
                userViewModel.apiContext.value?.also { apiContext ->
                    fileStorageViewModel.batchDelete(adapter.selectedItems.map { it.binding.file!! }, scope!!, apiContext)
                    enableUI(false)
                }
            }
            else -> return false
        }
        return true
    }

    override fun onItemClick(view: View, viewHolder: FileAdapter.FileViewHolder) {
        if (viewHolder.binding.file!!.type == FileType.FOLDER) {
            val action = FilesFragmentDirections.actionFilesFragmentSelf(viewHolder.binding.file!!.id, viewHolder.binding.scope!!.login, viewHolder.binding.file!!.name, pasteMode = args.pasteMode, folderNameId = null)
            navController.navigate(action)
        } else if (viewHolder.binding.file!!.type == FileType.FILE) {
            openFile(viewHolder.binding.file!!)
        }
    }

    private fun openFile(file: IRemoteFile) {
        if (file.type == FileType.FILE) {
            val tempDir = File(requireActivity().cacheDir, "filestorage")
            if (!tempDir.exists())
                tempDir.mkdir()
            val tempFile = File(tempDir, FileUtil.escapeFileName(file.name))
            userViewModel.apiContext.value?.also { apiContext ->
                fileStorageViewModel.startOpenDownload(workManager, apiContext, file, scope!!, tempFile.absolutePath)
            }
        }
    }

    private fun uploadFile(uri: Uri) {
        userViewModel.apiContext.value?.also { apiContext ->
            fileStorageViewModel.startUpload(workManager, scope!!, apiContext, uri, FileUtil.uriToFileName(uri, requireContext()), 0, folderId!!)
        }
    }

    override fun onUIStateChanged(enabled: Boolean) {
        binding.fileStorageSwipeRefresh.isEnabled = enabled
        binding.fileList.isEnabled = enabled
        binding.fabUploadFile.isEnabled = enabled
    }
}

class SaveFileContract : ActivityResultContract<Pair<Intent, IRemoteFile>, Pair<ActivityResult, IRemoteFile>>() {

    private lateinit var file: IRemoteFile

    override fun createIntent(context: Context, input: Pair<Intent, IRemoteFile>): Intent {
        file = input.second
        return input.first
    }

    override fun parseResult(resultCode: Int, intent: Intent?): Pair<ActivityResult, IRemoteFile> {
        return Pair(ActivityResult(resultCode, intent), file)
    }
}

class OpenDocumentsContract : ActivityResultContract<Array<String>, Array<Uri>?>() {

    override fun createIntent(context: Context, input: Array<String>?): Intent {
        return Intent(Intent.ACTION_OPEN_DOCUMENT)
            .putExtra(Intent.EXTRA_MIME_TYPES, input)
            .putExtra(Intent.EXTRA_ALLOW_MULTIPLE, true)
            .setType("*/*")
    }

    override fun parseResult(resultCode: Int, intent: Intent?): Array<Uri>? {
        if (intent == null || resultCode != Activity.RESULT_OK)
            return null

        return if (intent.clipData != null) {
            Array(intent.clipData!!.itemCount) { index -> intent.clipData!!.getItemAt(index).uri }
        } else {
            arrayOf(intent.data!!)
        }
    }
}