import 'dart:convert';
import 'dart:io';

import 'package:crypto_prices/constants.dart';
import 'package:crypto_prices/generated/l10n.dart';
import 'package:crypto_prices/util/import_export.dart';
import 'package:crypto_prices/util/manual_migration_handler.dart';
import 'package:crypto_prices/widgets/settingsScreens/migration_assistant.dart';
import 'package:flutter/material.dart';
import 'package:flutter_file_dialog/flutter_file_dialog.dart';
import 'package:path_provider/path_provider.dart';

///Displays the Import and Export settings
class ImportExportWidget extends StatefulWidget {

  @override
  State<ImportExportWidget> createState() => _ImportExportWidgetState();
}

class _ImportExportWidgetState extends State<ImportExportWidget> {

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
          title: Text(S.of(context).settingsImportExport)
      ),
      body: ListView(
        children: <Widget>[
          ListTile(
            title: Text(S.of(context).settingsImportExportImportFileTitle),
            subtitle: Text(S.of(context).settingsImportExportImportFileSubtitle),
            onTap: () async {
              _importData();
            },
          ),
          ListTile(
            title: Text(S.of(context).settingsImportExportExportFileTitle),
            subtitle: Text(S.of(context).settingsImportExportExportFileSubtitle),
            onTap: () {
              _exportData();
            },
          ),
        ],
      ),
    );
  }

  ///Displays an alert to confirm data can be overwritten. Returns true if data can be overwritten
  Future<bool> _showOverwriteAlert() async {
    bool overwriteData = false;

    await showDialog(
      context: context,
      builder: (BuildContext context) {
        return AlertDialog(
          title: Text(S.of(context).settingsImportExportOverwriteAlertTitle),
          content: Text(S.of(context).settingsImportExportOverwriteAlertMessage),
          actions: [
            TextButton(
              onPressed: () {
                Navigator.of(context).pop();
              },
              child: Text(S.of(context).alertDialogCancel)
            ),
            TextButton(
              onPressed: () {
                overwriteData = true;
                Navigator.pop(context); //pop dialog
              },
              child: Text(S.of(context).alertDialogContinue)
            ),
          ],
        );
      }
    );
    return overwriteData;
  }

  ///Opens a load file dialog and imports favorites, alerts, portfolios and transactions from a json file
  void _importData() async {
    final params = OpenFileDialogParams(
      dialogType: OpenFileDialogType.document,
      mimeTypesFilter: ["application/json"]
    );
    String? importFilePath = await FlutterFileDialog.pickFile(params: params);
    final importExport = ImportExport();

    if(importFilePath != null) {
      if (await importExport.willOverwriteData(importFilePath)) {
        final overwriteData = await _showOverwriteAlert();

        if (!overwriteData) {
          return;
        }
      }

      try {
        await importExport.importData(importFilePath);

        if (ManualMigrationHandler().isManualMigrationNeeded()) {
          await Navigator.of(context).push(MaterialPageRoute(builder: (context) => MigrationAssistant()));
        }

        final snackBar = SnackBar(
          content: Text(S.of(context).settingsImportExportImportSuccessful),
          duration: Duration(seconds: 1),
        );

        ScaffoldMessenger.of(context).showSnackBar(snackBar);
      } catch (e) {
        final snackBar = SnackBar(
          content: Text(S.of(context).settingsImportExportWrongFormat),
          duration: Duration(seconds: 1),
        );

        ScaffoldMessenger.of(context).showSnackBar(snackBar);
      }
    }
  }

  ///True if the file at the given path is from a previous database version
  Future<bool> _isLegacyImport(String importFilePath) async {
    File import = File(importFilePath);
    String jsonString = await import.readAsString();

    Map<String, dynamic> json = JsonDecoder().convert(jsonString);

    int importDatabaseVersion = json[Constants.DATABASEVERSIONKEY] ?? Constants.DATABASEVERSIONMULTIPLEPORTFOLIOS;

    if (importDatabaseVersion != Constants.DATABASEVERSION) {
      return true;
    } else {
      return false;
    }
  }

  ///Opens a save file dialog to export favorites, alerts, portfolios and transactions to a json file
  void _exportData() async {
    Directory appDocDir = await getApplicationDocumentsDirectory();
    String exportPath = appDocDir.path + "/${Constants.EXPORTFILENAME}.json";
    await ImportExport().exportData(exportPath);

    final params = SaveFileDialogParams(
      sourceFilePath: exportPath,
      fileName: Constants.EXPORTFILENAME,
      mimeTypesFilter: ["application/json"]
    );
    String? savedFilePath = await FlutterFileDialog.saveFile(params: params);
    File(exportPath).delete();

    if (savedFilePath != null) {
      final snackBar = SnackBar(
        content: Text(S.of(context).settingsImportExportExportSuccessful),
        duration: Duration(seconds: 1),
      );

      ScaffoldMessenger.of(context).showSnackBar(snackBar);
    }
  }
}