
import 'dart:io';

import 'package:crypto_prices/api/coingecko_api.dart';
import 'package:crypto_prices/database/available_coins_dao.dart';
import 'package:crypto_prices/database/explorer_transaction_url_dao.dart';
import 'package:crypto_prices/database/portfolio_dao.dart';
import 'package:crypto_prices/database/settings_dao.dart';
import 'package:crypto_prices/database/transaction_dao.dart';
import 'package:crypto_prices/database/wallet_dao.dart';
import 'package:crypto_prices/isar.g.dart';
import 'package:crypto_prices/models/coin.dart';
import 'package:crypto_prices/models/explorer_transaction_url.dart';
import 'package:crypto_prices/models/global_market_info.dart';
import 'package:crypto_prices/models/portfolio.dart';
import 'package:crypto_prices/models/wallet.dart';
import 'package:crypto_prices/models/screen_arguments.dart';
import 'package:crypto_prices/models/transaction.dart';
import 'package:crypto_prices/util/image_urls.dart';
import 'package:crypto_prices/util/import_export.dart';
import 'package:crypto_prices/util/manual_migration_handler.dart';
import 'package:crypto_prices/util/notification_handler.dart';
import 'package:crypto_prices/util/util.dart';
import 'package:crypto_prices/widgets/detailScreen/detail_screen_base.dart';
import 'package:crypto_prices/widgets/priceAlertsScreen/price_alert_edit_widget.dart';
import 'package:crypto_prices/widgets/settingsScreens/migration_assistant.dart';
import 'package:device_info_plus/device_info_plus.dart';
import 'package:flutter/material.dart';
import 'package:flutter_local_notifications/flutter_local_notifications.dart';
import 'package:flutter_localizations/flutter_localizations.dart';
import 'package:hive_flutter/hive_flutter.dart';
import 'package:isar/isar.dart';
import 'package:package_info_plus/package_info_plus.dart';
import 'package:path_provider/path_provider.dart';

import 'constants.dart';
import 'generated/l10n.dart';
import 'models/available_coin.dart';
import 'models/coin_detail.dart';
import 'models/date_price.dart';
import 'models/favorite_coin.dart';
import 'models/market_data.dart';
import 'util/workmanager_handler.dart';
import 'widgets/detailScreen/coin_detail_screen.dart';
import 'widgets/home_page.dart';


void main() async {
  WidgetsFlutterBinding.ensureInitialized();
  await Hive.initFlutter();

  Hive.registerAdapter(CoinDetailAdapter()); //register subclasses first
  Hive.registerAdapter(CoinAdapter());
  Hive.registerAdapter(MarketDataAdapter());
  Hive.registerAdapter(DatePriceAdapter());
  Hive.registerAdapter(AvailableCoinAdapter());
  Hive.registerAdapter(FavoriteCoinAdapter());
  Hive.registerAdapter(PortfolioAdapter());
  Hive.registerAdapter(WalletAdapter());
  Hive.registerAdapter(TransactionAdapter());
  Hive.registerAdapter(ExplorerTransactionUrlAdapter());
  Hive.registerAdapter(GlobalMarketInfoAdapter());

  //do before opening boxes to prevent incompatibility  errors
  int currentDatabaseVersion = await SettingsDAO().getDatabaseVersionPrefs();
  if (currentDatabaseVersion != Constants.DATABASEVERSION) {
    await migrateDatabase(currentDatabaseVersion);
    SettingsDAO().setDatabaseVersion(Constants.DATABASEVERSION);
  }

  await Hive.openBox(Constants.COINSBOXNAME);
  await Hive.openBox(Constants.FAVORITESBOXNAME);
  await Hive.openBox(Constants.MARKETDATABOXNAME);
  await Hive.openBox(Constants.COINDETAILBOXNAME);
  await Hive.openBox(Constants.AVAILABLECOINSBOXNAME);
  await Hive.openBox(Constants.SETTINGSBOXNAME);
  await Hive.openBox(Constants.CURRENCYCHANGEDBOXNAME);
  await Hive.openBox<Portfolio>(Constants.PORTFOLIOBOXNAME);
  await Hive.openBox<Wallet>(Constants.WALLETBOXNAME);
  await Hive.openBox<Transaction>(Constants.TRANSACTIONSBOXNAME);
  await Hive.openBox<ExplorerTransactionUrl>(Constants.EXPLORERSTRANSACTIONBOXNAME);
  await Hive.openBox(Constants.GLOBALMARKETINFOBOXNAME);
  await Hive.openBox(Constants.COINTHUMBNAILURLSBOXNAME);

  isar = await openIsar();

  SettingsDAO settingsDAO = SettingsDAO();

  WorkmanagerHandler().initializeWorkmanager();

  PackageInfo packageInfo = await PackageInfo.fromPlatform();
  String currentVersion = packageInfo.version;

  //restart all alert background tasks after installing an app update
  if (settingsDAO.getCurrentAppVersionHive() != currentVersion) {
    WorkmanagerHandler().restartAllAlerts();
    settingsDAO.setCurrentAppVersion(currentVersion);
  }

  String initialRoute = HomePage.routeName;

  NotificationHandler notificationHandler = NotificationHandler();
  notificationHandler.initialize(onSelectNotification);

  NotificationAppLaunchDetails? notificationAppLaunchDetails =
    await notificationHandler.plugin.getNotificationAppLaunchDetails();

  if (notificationAppLaunchDetails?.didNotificationLaunchApp ?? false) {
    String? payload = notificationAppLaunchDetails!.payload;
    if (payload != null) {
      initialRoute = CoinDetailScreen.routeName;
      routeArguments = CoinDetailScreenArguments(payload.split("_").first, payload.split("_").last);
    }
  }

  ImageUrls().decodeThumbnails(); //parse image urls in background

  AvailableCoinsDao availableCoinsDao = AvailableCoinsDao();
  final oldSupportedCoins = AvailableCoinsDao().getAllAvailableCoins();

  DateTime? availableCoinsLastUpdated = settingsDAO.getAvailableCoinsLastUpdated();
  if (oldSupportedCoins.length == 0
      || availableCoinsLastUpdated == null
      || DateTime.now().difference(availableCoinsLastUpdated).inDays >= Constants.AVAILABLECOINSUPDATETIMERDAYS) {
    CoinGeckoAPI().getAvailableCoins().then((value) {
      availableCoinsDao.updateAllAvailableCoins(value);
      settingsDAO.setAvailableCoinsLastUpdated(DateTime.now());
    });
  }

  if (Hive.box<ExplorerTransactionUrl>(Constants.EXPLORERSTRANSACTIONBOXNAME).isEmpty
      || settingsDAO.getCurrentAppVersionHive() != currentVersion) {
    ExplorerTransactionUrlDao().decodeDefaultTransactionUrls();
  }

  if (ManualMigrationHandler().isManualMigrationNeeded()) {
    initialRoute = MigrationAssistant.routeName;
  }

  DeviceInfoPlugin deviceInfo = DeviceInfoPlugin();
  androidDeviceInfo = await deviceInfo.androidInfo;

  runApp(MyApp(initialRoute: initialRoute));
}

final navKey = new GlobalKey<NavigatorState>();

Object? routeArguments;

Isar? isar;

AndroidDeviceInfo? androidDeviceInfo;

Future<dynamic> onSelectNotification(String? payload) async {
  if (payload != null) {
    String coinId = payload.split("_").first;
    String coinName = payload.split("_").last;
    navKey.currentState!.push(MaterialPageRoute(
        builder: (_) => DetailScreenBase(coinId: coinId))
    );
  }
}

///Migrates the database from the old version to the new
///Version 2: save MarketData objects in the MarketDataBox instead of CoinDetail objects
///Version 3: add transaction actualPrice. Set actual price to coinPrice in old transactions
///Version 4: add multiple portfolios. Wallets now have int ids
Future<void> migrateDatabase(int oldVersion) async {
  if (oldVersion == 1) {
    Hive.deleteBoxFromDisk(Constants.MARKETDATABOXNAME);
    oldVersion++;
  }

  if (oldVersion == 2) {
    final box = await Hive.openBox<Transaction>(Constants.TRANSACTIONSBOXNAME);
    List<Transaction> transactions = box.values.toList();
    transactions.forEach((element) {
      element.actualCoinPrice = element.coinPrice;
      element.save();
    });
    oldVersion++;
  }

  if (oldVersion == 3) {
    await Hive.openBox<Transaction>(Constants.TRANSACTIONSBOXNAME);
    await Hive.openBox<Portfolio>(Constants.PORTFOLIOBOXNAME);
    await Hive.openBox(Constants.SETTINGSBOXNAME);
    await Hive.openBox<ExplorerTransactionUrl>(Constants.EXPLORERSTRANSACTIONBOXNAME);
    final walletBox = await Hive.openBox<Wallet>(Constants.WALLETBOXNAME);

    if (walletBox.isNotEmpty) { //only add new portfolio if wallet data exists
      final newPortfolio = Portfolio(
          Util.generateId(),
          "Portfolio 1",
          currency: SettingsDAO().getCurrencyHive()
      );

      WalletDao walletDao = WalletDao();
      List<Wallet> wallets = walletDao.getAllWallets();
      wallets.forEach((wallet) {
        wallet.id = Util.generateId();
        wallet.connectedPortfolioId = newPortfolio.id;
      });

      PortfolioDao().insertPortfolio(newPortfolio);

      await walletBox.clear();
      walletDao.insertMultipleWallets(wallets);

      final transactions = TransactionDao().getAllTransactions();
      newPortfolio.insertMultipleTransaction(transactions);
    }

    //bittorrent coinId changed
    ExplorerTransactionUrlDao explorerTransactionUrlDao = ExplorerTransactionUrlDao();
    if (explorerTransactionUrlDao.getCoinExplorerTransactionUrl("bittorrent-2") != null) {
      final url = explorerTransactionUrlDao.getCoinExplorerTransactionUrl("bittorrent-2")!;
      url.coinId = "bittorrent";
      explorerTransactionUrlDao.deleteCoinExplorerTransactionUrl("bittorrent-2");
      explorerTransactionUrlDao.insertCoinExplorerTransactionUrl(url);
    }

    oldVersion++;
  }
}

class MyApp extends StatefulWidget {
  final String initialRoute;

  MyApp({Key? key, required this.initialRoute});

  @override
  _MyAppState createState() => _MyAppState();
}

class _MyAppState extends State<MyApp> with WidgetsBindingObserver {
  late ThemeData _lightTheme;

  late ThemeData _darkTheme;

  @override
  void initState() {
    super.initState();
    WidgetsBinding.instance!.addObserver(this);

    //rebuild app when theme or language changes
    currentTheme.addListener(() {
      setState(() {});
    });
    languageChange.addListener(() {
      setState(() {});
    });

    _lightTheme = ThemeData(
      primaryColor: Constants.defaultPrimaryColor,
      textButtonTheme: TextButtonThemeData(style: TextButton.styleFrom(primary: Constants.accentColorLight!)),
      colorScheme: ColorScheme.light().copyWith(
        primary: Constants.defaultPrimaryColor,
        secondary: Constants.accentColorLight,
        onSecondary: Colors.white
      ),
    );

    _darkTheme = ThemeData(
      primaryColor: Constants.defaultPrimaryColor,
      textButtonTheme: TextButtonThemeData(style: TextButton.styleFrom(primary: Constants.accentColorDark!)),
      colorScheme: ColorScheme.dark().copyWith(
        surface: Constants.defaultPrimaryColor,
        primary: Constants.accentColorDark,
        secondary: Constants.accentColorDark
      ),
    );
  }

  @override
  void dispose() {
    WidgetsBinding.instance!.removeObserver(this);
    super.dispose();
  }

  @override
  void didChangeAppLifecycleState(AppLifecycleState state) async {
    switch (state) {
      case AppLifecycleState.paused: {
        await _backupData();
        break;
      }

      default: {
        return;
      }
    }
  }

  @override
  Widget build(BuildContext context) {
    return MaterialApp(
      onGenerateTitle: (context) => S.of(context).appName,
      localizationsDelegates: [
        S.delegate,
        GlobalMaterialLocalizations.delegate,
        GlobalWidgetsLocalizations.delegate,
        GlobalCupertinoLocalizations.delegate
      ],
      supportedLocales: S.delegate.supportedLocales,
      locale: languageChange.currentLocale(), //use system locale (if supported) if null
      theme: _lightTheme,
      darkTheme: _darkTheme,
      themeMode: currentTheme.currentTheme(),
      navigatorKey: navKey,
      initialRoute: widget.initialRoute,
      routes: {
        //only use named routes when tapping notification
        HomePage.routeName: (_) => HomePage(),
        CoinDetailScreen.routeName: (_) => DetailScreenBase(
            coinId: (routeArguments as CoinDetailScreenArguments).coinId
        ),
        PriceAlertEditWidget.routeName: (_) => PriceAlertEditWidget(
          alertId: (routeArguments as PriceAlertEditWidgetArguments).alertId,
        ),
        MigrationAssistant.routeName: (_) => MigrationAssistant(),
      },
    );
  }

  ///Creates a backup file of all data in the internal app directory
  Future<void> _backupData() async {
    Directory appDocDir = await getApplicationDocumentsDirectory();
    String internalBackupFilePath = appDocDir.path + "/${Constants.BACKUPFILENAME}.json";
    await ImportExport().exportData(internalBackupFilePath);
    print("backup complete");
  }
}
