import 'package:crypto_prices/constants.dart';
import 'package:flutter/material.dart';
import 'package:hive/hive.dart';
import 'package:shared_preferences/shared_preferences.dart';

import '../main.dart';

///DAO for the settings data key-value pairs. Contains methods to get values from preferences or from Hive database.
///Only use Hive in the app and use preferences for background work
class SettingsDAO {

  ///Sets the locale value
  void setLocale(String locale) async {
    SharedPreferences prefs = await SharedPreferences.getInstance();
    prefs.setString(Constants.LOCALEKEY, locale);
    Hive.box(Constants.SETTINGSBOXNAME).put(Constants.LOCALEKEY, locale);
  }

  ///Gets the locale value from preferences (use in background work)
  Future<String> getLocalePrefs() async {
    SharedPreferences prefs = await SharedPreferences.getInstance();
    if (!prefs.containsKey(Constants.LOCALEKEY)) {
      return Constants.LOCALEDEFAULT;
    } else {
      return prefs.getString(Constants.LOCALEKEY)!;
    }
  }

  ///Gets the locale value from Hive (use in the app process)
  String getLocaleHive() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return box.get(Constants.LOCALEKEY, defaultValue: Constants.LOCALEDEFAULT);
  }

  ///Sets the currency value
  void setCurrency(String currency) async {
    SharedPreferences prefs = await SharedPreferences.getInstance();
    prefs.setString(Constants.CURRENCYKEY, currency);
    Hive.box(Constants.SETTINGSBOXNAME).put(Constants.CURRENCYKEY, currency);
  }

  ///Gets the currency value from preferences (use in background work)
  Future<String> getCurrencyPrefs() async {
    SharedPreferences prefs = await SharedPreferences.getInstance();
    if (!prefs.containsKey(Constants.CURRENCYKEY)) {
      return Constants.CURRENCYDEFAULT;
    } else {
      return prefs.getString(Constants.CURRENCYKEY)!;
    }
  }

  ///Gets the currency value from Hive (use in the app process)
  String getCurrencyHive() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return box.get(Constants.CURRENCYKEY, defaultValue: Constants.CURRENCYDEFAULT);
  }

  ///Sets the darkModeEnabled value
  void setDarkModeEnabled(bool darkModeEnabled) {
    Hive.box(Constants.SETTINGSBOXNAME).put(Constants.DARKMODEKEY, darkModeEnabled);
  }

  ///Gets the darkModeEnabled value from Hive (use in the app process)
  bool getDarkModeEnabledHive() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return box.get(Constants.DARKMODEKEY, defaultValue: Constants.DARKMODEENABLEDDEFAULT);
  }

  ///Sets the systemDarkMode value
  void setSystemDarkMode(bool useSystemDarkMode) {
    Hive.box(Constants.SETTINGSBOXNAME).put(Constants.SYSTEMDARKMODE, useSystemDarkMode);
  }

  ///Gets the systemDarkMode value from Hive (use in the app process)
  bool getUseSystemDarkModeHive() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    if (!box.containsKey(Constants.SYSTEMDARKMODE)) {
      int _sdkNumber = androidDeviceInfo?.version.sdkInt ?? 28;
      if (_sdkNumber >= 29) { //if < 29 don't use system default (light theme)
        return true;
      } else {
        return false;
      }
    } else {
      return Hive.box(Constants.SETTINGSBOXNAME).get(Constants.SYSTEMDARKMODE);
    }
  }

  ///Sets the marketDataLastUpdated value of a coin
  void setMarketDataLastUpdated(String coinId, DateTime marketDataLastUpdated) {
    Hive.box(Constants.SETTINGSBOXNAME).put(coinId + Constants.MARKETDATALASTUPDATEDKEY, marketDataLastUpdated);
  }

  ///Gets the marketDataLastUpdated value for a coin from Hive (use in the app process)
  DateTime? getMarketDataLastUpdatedHive(String coinId) {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return box.get(coinId + Constants.MARKETDATALASTUPDATEDKEY);
  }

  ///Sets the coinsLastUpdated value
  void setCoinsLastUpdated(DateTime coinsLastUpdated) {
    Hive.box(Constants.SETTINGSBOXNAME).put(Constants.COINSLASTUPDATEDKEY, coinsLastUpdated);
  }

  ///Gets the coinsLastUpdated value from Hive (use in the app process)
  DateTime? getCoinsLastUpdatedHive() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return box.get(Constants.COINSLASTUPDATEDKEY);
  }

  ///Sets the favoritesLastUpdated value
  void setFavoritesLastUpdated(DateTime favoritesLastUpdated) {
    Hive.box(Constants.SETTINGSBOXNAME).put(Constants.FAVORITESLASTUPDATEDKEY, favoritesLastUpdated);
  }

  ///Gets the favoritesLastUpdated value from Hive (use in the app process)
  DateTime? getFavoritesLastUpdatedHive() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return box.get(Constants.FAVORITESLASTUPDATEDKEY);
  }

  ///Sets the availableCoinsLastUpdated value
  void setAvailableCoinsLastUpdated(DateTime availableCoinsLastUpdated) {
    Hive.box(Constants.SETTINGSBOXNAME).put(Constants.AVAILABLECOINSLASTUPDATEDKEY, availableCoinsLastUpdated);
  }

  ///Gets the availableCoinsLastUpdated value from Hive (use in the app process)
  DateTime? getAvailableCoinsLastUpdated() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return box.get(Constants.AVAILABLECOINSLASTUPDATEDKEY);
  }

  ///Sets the walletsLastUpdated value
  void setWalletsLastUpdated(DateTime WalletsLastUpdated) {
    Hive.box(Constants.SETTINGSBOXNAME).put(Constants.WALLETSLASTUPDETEDKEY, WalletsLastUpdated);
  }

  ///Gets the walletsLastUpdated value from Hive (use in the app process)
  DateTime? getWalletsLastUpdatedHive() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return box.get(Constants.WALLETSLASTUPDETEDKEY);
  }

  ///Sets the globalMarketInfoLastUpdated value
  void setGlobalMarketInfoLastUpdated(DateTime globalMarketInfoLastUpdated) {
    Hive.box(Constants.SETTINGSBOXNAME).put(Constants.GLOBALMARKETINFOLASTUPDATEDKEY, globalMarketInfoLastUpdated);
  }

  ///Gets the globalMarketInfoLastUpdated value from Hive (use in the app process)
  DateTime? getGlobalMarketInfoLastUpdatedHive() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return box.get(Constants.GLOBALMARKETINFOLASTUPDATEDKEY);
  }

  ///Sets the LargeMoveAlertThreshold value
  void setLargeMoveAlertThreshold(double threshold) async {
    SharedPreferences prefs = await SharedPreferences.getInstance();
    prefs.setDouble(Constants.LARGEMOVEALERTLIMITKEY, threshold);
    Hive.box(Constants.SETTINGSBOXNAME).put(Constants.LARGEMOVEALERTLIMITKEY, threshold);
  }

  ///Gets the LargeMoveAlertThreshold value from preferences (use in background work)
  Future<double> getLargeMoveAlertThresholdPrefs() async {
    SharedPreferences prefs = await SharedPreferences.getInstance();
    if (!prefs.containsKey(Constants.LARGEMOVEALERTLIMITKEY)) {
      return Constants.LARGEMOVEALERTTHRESHOLDDEFAULT;
    } else {
      return prefs.getDouble(Constants.LARGEMOVEALERTLIMITKEY)!;
    }
  }

  ///Gets the LargeMoveAlertThreshold value from Hive (use in the app process)
  double getLargeMoveAlertThresholdHive() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return box.get(Constants.LARGEMOVEALERTLIMITKEY, defaultValue: Constants.LARGEMOVEALERTTHRESHOLDDEFAULT)!;
  }

  ///Sets the LargeMoveAlertsEnabled value
  void setLargeMoveAlertsEnabled(bool enabled) async {
    SharedPreferences prefs = await SharedPreferences.getInstance();
    prefs.setBool(Constants.LARGEMOVEALERTENABLEDKEY, enabled);
    Hive.box(Constants.SETTINGSBOXNAME).put(Constants.LARGEMOVEALERTENABLEDKEY, enabled);
  }

  ///Gets the LargeMoveAlertsEnabled value from preferences (use in background work)
  Future<bool> getLargeMoveAlertsEnabledPrefs() async {
    SharedPreferences prefs = await SharedPreferences.getInstance();
    if (!prefs.containsKey(Constants.LARGEMOVEALERTENABLEDKEY)) {
      return Constants.LARGEMOVEALERTENABLEDDEFAULT;
    } else {
      return prefs.getBool(Constants.LARGEMOVEALERTENABLEDKEY)!;
    }
  }

  ///Gets the LargeMoveAlertsEnabled value from Hive (use in the app process)
  bool getLargeMoveAlertsEnabledHive() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return box.get(Constants.LARGEMOVEALERTENABLEDKEY, defaultValue: Constants.LARGEMOVEALERTENABLEDDEFAULT)!;
  }

  ///Sets the primary color value
  void setPrimaryColor(Color color) {
    Hive.box(Constants.SETTINGSBOXNAME).put(Constants.PRIMARYCOLORKEY, color.value);
  }

  ///Gets the primary color from Hive (use in the app process)
  Color getPrimaryColorHive() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    int colorValue = box.get(Constants.PRIMARYCOLORKEY, defaultValue: Constants.defaultPrimaryColor!.value);
    return Color(colorValue);
  }

  ///Sets the value of collapseAll to true if all entries should be collapsed
  ///and to false if all entries should be expanded
  void setCollapseAll(bool collapseAll) {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    box.put(Constants.COLLAPSEALLKEY, collapseAll);
  }

  ///Gets the collapseAll value from Hive (use in the app process)
  bool getCollapseAllHive() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return box.get(Constants.COLLAPSEALLKEY, defaultValue: Constants.COLLAPSEALLDEFAULT);
  }

  ///Returns a stream that listens to changes of the collapseAll value
  Stream<BoxEvent> getCollapseAllWatcher() {
    return Hive.box(Constants.SETTINGSBOXNAME).watch(key: Constants.COLLAPSEALLKEY);
  }

  ///Sets the database version
  void setDatabaseVersion(int version) async {
    SharedPreferences prefs = await SharedPreferences.getInstance();
    prefs.setInt(Constants.DATABASEVERSIONKEY, version);
  }

  ///Gets the database version from the preferences
  Future<int> getDatabaseVersionPrefs() async {
    SharedPreferences prefs = await SharedPreferences.getInstance();
    if (!prefs.containsKey(Constants.DATABASEVERSIONKEY)) {
      return 1; //first version
    } else {
      return prefs.getInt(Constants.DATABASEVERSIONKEY)!;
    }
  }

  ///Sets the add wallet to favorites value
  void setAddWalletToFavorites(bool addToFav) {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    box.put(Constants.ADDWALLETTOFAVORITESKEY, addToFav);
  }

  ///Gets the add wallet to favorites value from Hive
  bool getAddWalletToFavoritesHive() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return box.get(Constants.ADDWALLETTOFAVORITESKEY, defaultValue: Constants.ADDWALLETTOFAVORITESDEFAULT)!;
  }

  ///Sets the current app version
  void setCurrentAppVersion(String currentVersion) {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    box.put(Constants.APPVERSIONKEY, currentVersion);
  }

  ///Gets the current app version from Hive
  String getCurrentAppVersionHive() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return box.get(Constants.APPVERSIONKEY, defaultValue: Constants.APPVERSIONDEFAULT);
  }

  ///Sets the selected time period of the wallet
  void setPortfolioSelectedPeriod(TimePeriod period) {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    box.put(Constants.PORTFOLIOSELECTEDPERIODKEY, period.index);
  }

  ///Gets the selected time period of the portfolio from Hive
  TimePeriod getPortfolioSelectedPeriod() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    if (!box.containsKey(Constants.PORTFOLIOSELECTEDPERIODKEY)) {
      return TimePeriod.day;
    } else {
      return TimePeriod.values[box.get(Constants.PORTFOLIOSELECTEDPERIODKEY)];
    }
  }

  ///Sets if the coin price entered by the user should be used
  ///for coin value calculation in transaction list
  void setUseOwnCoinPrice(bool useOwnCoinPrice) {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    box.put(Constants.USEOWNCOINPRICEKEY, useOwnCoinPrice);
  }

  ///gets if the coin price entered by the user should be used
  ///for coin value calculation in transaction list
  bool getUseOwnCoinPriceHive() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return box.get(Constants.USEOWNCOINPRICEKEY, defaultValue: Constants.USEOWNCOINPRICEDEFAULT);
  }

  ///Sets if the transaction value should be displayed in the portfolio currency
  void setDisplayInPortfolioCurrency(bool displayInPortfolioCurrency) {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    box.put(Constants.DISPLAYINPORTFOLIOCURRENCYKEY, displayInPortfolioCurrency);
  }

  ///Gets if the transaction value should be displayed in the portfolio currency
  bool getDisplayInPortfolioCurrency() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return box.get(Constants.DISPLAYINPORTFOLIOCURRENCYKEY, defaultValue: Constants.DISPLAYINPORTFOLIOCURRENCYDEFAULT);
  }

  ///Sets the id of the default portfolio
  void setDefaultPortfolioId(int portfolioId) {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    box.put(Constants.DEFAULTPORTFOLIOIDKEY, portfolioId);
  }

  ///Gets the id of the default portfolio or 0 if none exists
  int getDefaultPortfolioIdHive() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return box.get(Constants.DEFAULTPORTFOLIOIDKEY, defaultValue: Constants.NODEFAULTPORTFOLIOID);
  }

  ///Sets the start screen of the app
  void setStartScreen(StartScreen startScreen) {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    box.put(Constants.STARTSCREENKEY, startScreen.index);
  }

  ///Gets the start screen of the app
  StartScreen getStartScreen() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return StartScreen.values[box.get(Constants.STARTSCREENKEY, defaultValue: Constants.DEFAULTSTARTSCREEN.index)];
  }

  ///Sets the thumbnailsLastUpdated value
  void setThumbnailsLastUpdated(DateTime thumbnailsLastUpdated) {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    box.put(Constants.thumbnailsLastUpdatedKey, thumbnailsLastUpdated);
  }

  ///Gets the thumbnailsLastUpdated value
  DateTime? getThumbnailsLastUpdated() {
    var box = Hive.box(Constants.SETTINGSBOXNAME);
    return box.get(Constants.thumbnailsLastUpdatedKey);
  }
}