import 'package:crypto_prices/database/settings_dao.dart';
import 'package:flutter/material.dart';
import 'package:flutter/scheduler.dart';

///Change Notifier that changes the theme of the app and notifies listeners
class ThemeChange extends ChangeNotifier {
  bool _isDark = true;
  bool _useSystemDarkMode = true;
  SettingsDAO _settingsDAO = SettingsDAO();

  ThemeChange() {
    _isDark = _settingsDAO.getDarkModeEnabledHive();
    _useSystemDarkMode = _settingsDAO.getUseSystemDarkModeHive();
  }

  ///Returns the current theme mode
  ThemeMode currentTheme() {
    if (_useSystemDarkMode)
      return ThemeMode.system;

    return _isDark ? ThemeMode.dark : ThemeMode.light;
  }

  ///Checks if the current theme is the dark theme
  bool isThemeDark() {
    if (_useSystemDarkMode) {
      if (SchedulerBinding.instance!.window.platformBrightness == Brightness.dark) {
        return true;
      } else {
        return false;
      }
    } else {
      return _isDark;
    }
  }

  ///Switches between light and dark theme and notifies listeners
  void switchTheme() {
    _isDark = ! _isDark;
    _settingsDAO.setDarkModeEnabled(_isDark);
    notifyListeners();
  }

  ///Changes the theme mode of the app (light, dark, system)
  void changeTheme(ThemeMode mode) {
    switch (mode) {
      case ThemeMode.system: {
        _useSystemDarkMode = true;
        _settingsDAO.setSystemDarkMode(_useSystemDarkMode);
        break;
      }
      case ThemeMode.light: {
        _useSystemDarkMode = false;
        _settingsDAO.setSystemDarkMode(_useSystemDarkMode);
        _isDark = false;
        _settingsDAO.setDarkModeEnabled(_isDark);
        break;
      }
      case ThemeMode.dark: {
        _useSystemDarkMode = false;
        _settingsDAO.setSystemDarkMode(_useSystemDarkMode);
        _isDark = true;
        _settingsDAO.setDarkModeEnabled(_isDark);
        break;
      }
    }
    notifyListeners();
  }
}