package com.wa2c.android.cifsdocumentsprovider.tools.string_converter.repository

import com.wa2c.android.cifsdocumentsprovider.tools.string_converter.model.CsvRow
import com.wa2c.android.cifsdocumentsprovider.tools.string_converter.model.StringItem
import org.redundent.kotlin.xml.PrintOptions
import org.redundent.kotlin.xml.XmlVersion
import org.redundent.kotlin.xml.xml
import java.io.File

/**
 * Resource file repository
 */
class ResourceRepository {

    private val printOptions = PrintOptions(
        pretty = true,
        singleLineTextElements = true,
        indent = " ".repeat(4),
    )

    /**
     * Replace special characters
     */
    private fun String.replaceText(): String {
        return replace(Regex("\\r\\n|\\r|\\n"), "\\\\n")
            .replace(Regex("\'"), "\\\\'")
            .replace(Regex("\""), "\\\\\"")
    }

    /**
     * Save resource files
     */
    fun saveMultiLanguage(csvList: List<CsvRow>, resourceDirPath: String) {
        val stringMap = csvList.first().langText.keys.map { it to mutableListOf<StringItem>() }.toMap()
        var group = ""
        csvList.filter { it.resourceId.isNotEmpty() }.forEach { row ->
            if (row.title.isNotEmpty()) group = row.title
            row.langText.forEach { (langCode, text) ->
                stringMap[langCode]?.add(
                    StringItem(
                        group = group,
                        resourceId = row.resourceId,
                        value = text
                    )
                )
            }
        }

        stringMap.entries.map { entry ->
            val document = xml("resources", encoding = "utf-8", version= XmlVersion.V10) {
                comment("This file is generated by string converter tool.")
                comment("Source: https://docs.google.com/spreadsheets/d/1y71DyM31liwjcAUuPIk3CuIqxJD2l9Y2Q-YZ0I0XE_E/edit#gid=0")

                var currentGroup = ""

                entry.value.forEach { item ->
                    if (currentGroup != item.group) {
                        currentGroup = item.group
                        comment(currentGroup)
                    }
                    "string" {
                        attribute("name", item.resourceId)
                        -item.value.replaceText()
                    }
                }
            }

            val suffix = if (entry.key == "en") "" else "-${entry.key}"
            val file = File(resourceDirPath + "/values${suffix}" , "strings.xml")
            if (!file.exists()) file.createNewFile()

            file.writeText(document.toString(printOptions) + "\n", XML_CHARSET)
            println("Updated: " + file.canonicalPath)
        }
    }

    companion object {
        private val XML_CHARSET = Charsets.UTF_8
    }
}
