'use strict';

import { useEffect, useRef } from 'react';
import { findNodeHandle } from 'react-native';
import { useEvent } from './useEvent';
import { useSharedValue } from './useSharedValue';
import { isWeb } from '../PlatformChecker';
const IS_WEB = isWeb();
const scrollEventNames = ['onScroll', 'onScrollBeginDrag', 'onScrollEndDrag', 'onMomentumScrollBegin', 'onMomentumScrollEnd'];

/**
 * Lets you synchronously get the current offset of a `ScrollView`.
 *
 * @param animatedRef - An [animated ref](https://docs.swmansion.com/react-native-reanimated/docs/core/useAnimatedRef) attached to an Animated.ScrollView component.
 * @returns A shared value which holds the current offset of the `ScrollView`.
 * @see https://docs.swmansion.com/react-native-reanimated/docs/scroll/useScrollViewOffset
 */
export function useScrollViewOffset(animatedRef, initialRef) {
  const offsetRef = useRef(
  // eslint-disable-next-line react-hooks/rules-of-hooks
  initialRef !== undefined ? initialRef : useSharedValue(0));
  const eventHandler = useEvent(event => {
    'worklet';

    offsetRef.current.value = event.contentOffset.x === 0 ? event.contentOffset.y : event.contentOffset.x;
  }, scrollEventNames
  // Read https://github.com/software-mansion/react-native-reanimated/pull/5056
  // for more information about this cast.
  );

  useEffect(() => {
    const component = animatedRef.current;
    const viewTag = IS_WEB ? component : findNodeHandle(component);
    eventHandler.workletEventHandler.registerForEvents(viewTag);
    return () => {
      var _eventHandler$worklet;
      (_eventHandler$worklet = eventHandler.workletEventHandler) === null || _eventHandler$worklet === void 0 ? void 0 : _eventHandler$worklet.unregisterFromEvents();
    };
    // React here has a problem with `animatedRef.current` since a Ref .current
    // field shouldn't be used as a dependency. However, in this case we have
    // to do it this way.
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [animatedRef, animatedRef.current, eventHandler]);
  return offsetRef.current;
}
//# sourceMappingURL=useScrollViewOffset.js.map