package com.torrents_csv_android.ui.components

import android.content.ActivityNotFoundException
import android.content.Intent
import android.net.Uri
import android.widget.Toast
import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.combinedClickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.LazyListState
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.text.KeyboardActions
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Search
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.ListItem
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.material3.TextFieldDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.focus.onFocusChanged
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.input.key.NativeKeyEvent
import androidx.compose.ui.input.key.onKeyEvent
import androidx.compose.ui.platform.LocalClipboardManager
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalSoftwareKeyboardController
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.AnnotatedString
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.tooling.preview.Preview
import com.torrents_csv_android.R
import com.torrents_csv_android.Torrent
import com.torrents_csv_android.formatSize
import com.torrents_csv_android.magnetLink
import com.torrents_csv_android.sampleTorrent1
import com.torrents_csv_android.ui.theme.DEFAULT_PADDING
import kotlinx.coroutines.delay
import java.util.Date
import java.util.Locale

@Composable
fun TorrentListView(
    torrents: List<Torrent>,
    listState: LazyListState,
) {
    Column {
        LazyColumn(
            state = listState,
            modifier = Modifier.fillMaxHeight(),
        ) {
            items(
                items = torrents,
                key = { it.infohash },
            ) { torrent ->
                TorrentView(torrent)
                HorizontalDivider()
            }
        }
    }
}

@OptIn(ExperimentalFoundationApi::class)
@Composable
fun TorrentView(torrent: Torrent) {
    val ctx = LocalContext.current
    val magnet = magnetLink(torrent.infohash, torrent.name)
    val intent = Intent(Intent.ACTION_VIEW, Uri.parse(magnet))
    val localClipboardManager = LocalClipboardManager.current

    val sdf = java.text.SimpleDateFormat("yyyy-MM-dd", Locale.US)
    val created = sdf.format(Date(torrent.created_unix.toLong() * 1000))
//  val scraped = sdf.format(Date(torrent.scraped_date.toLong() * 1000))

    val noTorrentAppInstalledStr = stringResource(R.string.no_torrent_app_installed)
    val magnetLinkCopiedStr = stringResource(R.string.magnet_link_copied)
    Column(
        Modifier.combinedClickable(
            onClick = {
                try {
                    ctx.startActivity(intent)
                } catch (_: ActivityNotFoundException) {
                    Toast
                        .makeText(
                            ctx,
                            noTorrentAppInstalledStr,
                            Toast.LENGTH_SHORT,
                        ).show()
                }
            },
            onLongClick = {
                localClipboardManager.setText(AnnotatedString(magnet))
                Toast
                    .makeText(
                        ctx,
                        magnetLinkCopiedStr,
                        Toast.LENGTH_SHORT,
                    ).show()
            },
        ),
    ) {
        ListItem(
            headlineContent = {
                Text(
                    torrent.name,
                    maxLines = 4,
                    overflow = TextOverflow.Ellipsis,
                )
            },
            supportingContent = {
                Row(
                    horizontalArrangement = Arrangement.spacedBy(DEFAULT_PADDING),
                ) {
                    Text(
                        text = formatSize(torrent.size_bytes),
                    )
                    Text(
                        text = created,
                        fontWeight = FontWeight.Light,
                    )
                }
            },
            trailingContent = {
                Text(
                    text = torrent.seeders.toString(),
                )
            },
        )
    }
}

@Composable
fun SearchField(
    text: String,
    onSearchChange: (String) -> Unit,
    onSubmit: () -> Unit,
) {
    val isValid = text.count() >= 3

    val focusRequester = remember { FocusRequester() }
    var focus by remember { mutableStateOf(false) }
    val kbController = LocalSoftwareKeyboardController.current

    LaunchedEffect(Unit) {
        delay(300)
        kbController?.show()
        focusRequester.requestFocus()
    }

    TextField(
        colors =
            TextFieldDefaults.colors(
                focusedContainerColor = Color.Transparent,
                unfocusedContainerColor = Color.Transparent,
                errorContainerColor = Color.Transparent,
            ),
        value = text,
        modifier =
            Modifier
                .focusRequester(focusRequester)
                .fillMaxWidth()
                .onFocusChanged {
                    if (focus != it.isFocused) {
                        focus = it.isFocused
                    } else {
                        kbController?.hide()
                    }
                }.onKeyEvent {
                    if (it.nativeKeyEvent.keyCode == NativeKeyEvent.KEYCODE_ENTER) {
                        onSubmit()
                    }
                    false
                },
        onValueChange = onSearchChange,
        placeholder = {
            Text(stringResource(R.string.search))
        },
        trailingIcon = {
            Icon(Icons.Filled.Search, stringResource(R.string.search))
        },
        singleLine = true,
        keyboardActions =
            KeyboardActions(onDone = {
                onSubmit()
                kbController?.hide()
            }),
        isError = !isValid,
    )
}

@Preview
@Composable
fun TorrentPreview() {
    TorrentView(sampleTorrent1)
}
