/*
 * Copyright (C) 2016 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.google.android.exoplayer2.text.ssa;

import static com.google.common.truth.Truth.assertThat;
import static com.google.common.truth.Truth.assertWithMessage;

import android.text.Layout;
import androidx.test.core.app.ApplicationProvider;
import androidx.test.ext.junit.runners.AndroidJUnit4;
import com.google.android.exoplayer2.testutil.TestUtil;
import com.google.android.exoplayer2.text.Cue;
import com.google.android.exoplayer2.text.Subtitle;
import com.google.common.collect.Iterables;
import java.io.IOException;
import java.util.ArrayList;
import org.junit.Test;
import org.junit.runner.RunWith;

/** Unit test for {@link SsaDecoder}. */
@RunWith(AndroidJUnit4.class)
public final class SsaDecoderTest {

  private static final String EMPTY = "ssa/empty";
  private static final String TYPICAL = "ssa/typical";
  private static final String TYPICAL_HEADER_ONLY = "ssa/typical_header";
  private static final String TYPICAL_DIALOGUE_ONLY = "ssa/typical_dialogue";
  private static final String TYPICAL_FORMAT_ONLY = "ssa/typical_format";
  private static final String OVERLAPPING_TIMECODES = "ssa/overlapping_timecodes";
  private static final String POSITIONS = "ssa/positioning";
  private static final String INVALID_TIMECODES = "ssa/invalid_timecodes";
  private static final String INVALID_POSITIONS = "ssa/invalid_positioning";
  private static final String POSITIONS_WITHOUT_PLAYRES = "ssa/positioning_without_playres";

  @Test
  public void testDecodeEmpty() throws IOException {
    SsaDecoder decoder = new SsaDecoder();
    byte[] bytes = TestUtil.getByteArray(ApplicationProvider.getApplicationContext(), EMPTY);
    Subtitle subtitle = decoder.decode(bytes, bytes.length, false);

    assertThat(subtitle.getEventTimeCount()).isEqualTo(0);
    assertThat(subtitle.getCues(0).isEmpty()).isTrue();
  }

  @Test
  public void testDecodeTypical() throws IOException {
    SsaDecoder decoder = new SsaDecoder();
    byte[] bytes = TestUtil.getByteArray(ApplicationProvider.getApplicationContext(), TYPICAL);
    Subtitle subtitle = decoder.decode(bytes, bytes.length, false);

    assertThat(subtitle.getEventTimeCount()).isEqualTo(6);
    // Check position, line, anchors & alignment are set from Alignment Style (2 - bottom-center).
    Cue firstCue = subtitle.getCues(subtitle.getEventTime(0)).get(0);
    assertWithMessage("Cue.textAlignment")
        .that(firstCue.textAlignment)
        .isEqualTo(Layout.Alignment.ALIGN_CENTER);
    assertWithMessage("Cue.positionAnchor")
        .that(firstCue.positionAnchor)
        .isEqualTo(Cue.ANCHOR_TYPE_MIDDLE);
    assertWithMessage("Cue.position").that(firstCue.position).isEqualTo(0.5f);
    assertWithMessage("Cue.lineAnchor").that(firstCue.lineAnchor).isEqualTo(Cue.ANCHOR_TYPE_END);
    assertWithMessage("Cue.lineType").that(firstCue.lineType).isEqualTo(Cue.LINE_TYPE_FRACTION);
    assertWithMessage("Cue.line").that(firstCue.line).isEqualTo(0.95f);

    assertTypicalCue1(subtitle, 0);
    assertTypicalCue2(subtitle, 2);
    assertTypicalCue3(subtitle, 4);
  }

  @Test
  public void testDecodeTypicalWithInitializationData() throws IOException {
    byte[] headerBytes =
        TestUtil.getByteArray(ApplicationProvider.getApplicationContext(), TYPICAL_HEADER_ONLY);
    byte[] formatBytes =
        TestUtil.getByteArray(ApplicationProvider.getApplicationContext(), TYPICAL_FORMAT_ONLY);
    ArrayList<byte[]> initializationData = new ArrayList<>();
    initializationData.add(formatBytes);
    initializationData.add(headerBytes);
    SsaDecoder decoder = new SsaDecoder(initializationData);
    byte[] bytes =
        TestUtil.getByteArray(ApplicationProvider.getApplicationContext(), TYPICAL_DIALOGUE_ONLY);
    Subtitle subtitle = decoder.decode(bytes, bytes.length, false);

    assertThat(subtitle.getEventTimeCount()).isEqualTo(6);
    assertTypicalCue1(subtitle, 0);
    assertTypicalCue2(subtitle, 2);
    assertTypicalCue3(subtitle, 4);
  }

  @Test
  public void testDecodeOverlappingTimecodes() throws IOException {
    SsaDecoder decoder = new SsaDecoder();
    byte[] bytes =
        TestUtil.getByteArray(ApplicationProvider.getApplicationContext(), OVERLAPPING_TIMECODES);
    Subtitle subtitle = decoder.decode(bytes, bytes.length, false);

    assertThat(subtitle.getEventTime(0)).isEqualTo(1_000_000);
    assertThat(subtitle.getEventTime(1)).isEqualTo(2_000_000);
    assertThat(subtitle.getEventTime(2)).isEqualTo(4_230_000);
    assertThat(subtitle.getEventTime(3)).isEqualTo(5_230_000);
    assertThat(subtitle.getEventTime(4)).isEqualTo(6_000_000);
    assertThat(subtitle.getEventTime(5)).isEqualTo(8_440_000);
    assertThat(subtitle.getEventTime(6)).isEqualTo(9_440_000);
    assertThat(subtitle.getEventTime(7)).isEqualTo(10_720_000);
    assertThat(subtitle.getEventTime(8)).isEqualTo(13_220_000);
    assertThat(subtitle.getEventTime(9)).isEqualTo(14_220_000);
    assertThat(subtitle.getEventTime(10)).isEqualTo(15_650_000);

    String firstSubtitleText = "First subtitle - end overlaps second";
    String secondSubtitleText = "Second subtitle - beginning overlaps first";
    String thirdSubtitleText = "Third subtitle - out of order";
    String fourthSubtitleText = "Fourth subtitle - same timings as fifth";
    String fifthSubtitleText = "Fifth subtitle - same timings as fourth";
    String sixthSubtitleText = "Sixth subtitle - fully encompasses seventh";
    String seventhSubtitleText = "Seventh subtitle - nested fully inside sixth";
    assertThat(Iterables.transform(subtitle.getCues(1_000_010), cue -> cue.text.toString()))
        .containsExactly(firstSubtitleText);
    assertThat(Iterables.transform(subtitle.getCues(2_000_010), cue -> cue.text.toString()))
        .containsExactly(firstSubtitleText, secondSubtitleText);
    assertThat(Iterables.transform(subtitle.getCues(4_230_010), cue -> cue.text.toString()))
        .containsExactly(secondSubtitleText);
    assertThat(Iterables.transform(subtitle.getCues(5_230_010), cue -> cue.text.toString()))
        .isEmpty();
    assertThat(Iterables.transform(subtitle.getCues(6_000_010), cue -> cue.text.toString()))
        .containsExactly(thirdSubtitleText);
    assertThat(Iterables.transform(subtitle.getCues(8_440_010), cue -> cue.text.toString()))
        .containsExactly(fourthSubtitleText, fifthSubtitleText);
    assertThat(Iterables.transform(subtitle.getCues(9_440_010), cue -> cue.text.toString()))
        .isEmpty();
    assertThat(Iterables.transform(subtitle.getCues(10_720_010), cue -> cue.text.toString()))
        .containsExactly(sixthSubtitleText);
    assertThat(Iterables.transform(subtitle.getCues(13_220_010), cue -> cue.text.toString()))
        .containsExactly(sixthSubtitleText, seventhSubtitleText);
    assertThat(Iterables.transform(subtitle.getCues(14_220_010), cue -> cue.text.toString()))
        .containsExactly(sixthSubtitleText);
    assertThat(Iterables.transform(subtitle.getCues(15_650_010), cue -> cue.text.toString()))
        .isEmpty();
  }

  @Test
  public void testDecodePositions() throws IOException {
    SsaDecoder decoder = new SsaDecoder();
    byte[] bytes = TestUtil.getByteArray(ApplicationProvider.getApplicationContext(), POSITIONS);
    Subtitle subtitle = decoder.decode(bytes, bytes.length, false);

    // Check \pos() sets position & line
    Cue firstCue = Iterables.getOnlyElement(subtitle.getCues(subtitle.getEventTime(0)));
    assertWithMessage("Cue.position").that(firstCue.position).isEqualTo(0.5f);
    assertWithMessage("Cue.lineType").that(firstCue.lineType).isEqualTo(Cue.LINE_TYPE_FRACTION);
    assertWithMessage("Cue.line").that(firstCue.line).isEqualTo(0.25f);

    // Check the \pos() doesn't need to be at the start of the line.
    Cue secondCue = Iterables.getOnlyElement(subtitle.getCues(subtitle.getEventTime(2)));
    assertWithMessage("Cue.position").that(secondCue.position).isEqualTo(0.25f);
    assertWithMessage("Cue.line").that(secondCue.line).isEqualTo(0.25f);

    // Check only the last \pos() value is used.
    Cue thirdCue = Iterables.getOnlyElement(subtitle.getCues(subtitle.getEventTime(4)));
    assertWithMessage("Cue.position").that(thirdCue.position).isEqualTo(0.25f);

    // Check \move() is treated as \pos()
    Cue fourthCue = Iterables.getOnlyElement(subtitle.getCues(subtitle.getEventTime(6)));
    assertWithMessage("Cue.position").that(fourthCue.position).isEqualTo(0.5f);
    assertWithMessage("Cue.line").that(fourthCue.line).isEqualTo(0.25f);

    // Check alignment override in a separate brace (to bottom-center) affects textAlignment and
    // both line & position anchors.
    Cue fifthCue = Iterables.getOnlyElement(subtitle.getCues(subtitle.getEventTime(8)));
    assertWithMessage("Cue.position").that(fifthCue.position).isEqualTo(0.5f);
    assertWithMessage("Cue.line").that(fifthCue.line).isEqualTo(0.5f);
    assertWithMessage("Cue.positionAnchor")
        .that(fifthCue.positionAnchor)
        .isEqualTo(Cue.ANCHOR_TYPE_MIDDLE);
    assertWithMessage("Cue.lineAnchor").that(fifthCue.lineAnchor).isEqualTo(Cue.ANCHOR_TYPE_END);
    assertWithMessage("Cue.textAlignment")
        .that(fifthCue.textAlignment)
        .isEqualTo(Layout.Alignment.ALIGN_CENTER);

    // Check alignment override in the same brace (to top-right) affects textAlignment and both line
    // & position anchors.
    Cue sixthCue = Iterables.getOnlyElement(subtitle.getCues(subtitle.getEventTime(10)));
    assertWithMessage("Cue.position").that(sixthCue.position).isEqualTo(0.5f);
    assertWithMessage("Cue.line").that(sixthCue.line).isEqualTo(0.5f);
    assertWithMessage("Cue.positionAnchor")
        .that(sixthCue.positionAnchor)
        .isEqualTo(Cue.ANCHOR_TYPE_END);
    assertWithMessage("Cue.lineAnchor").that(sixthCue.lineAnchor).isEqualTo(Cue.ANCHOR_TYPE_START);
    assertWithMessage("Cue.textAlignment")
        .that(sixthCue.textAlignment)
        .isEqualTo(Layout.Alignment.ALIGN_OPPOSITE);
  }

  @Test
  public void testDecodeInvalidPositions() throws IOException {
    SsaDecoder decoder = new SsaDecoder();
    byte[] bytes =
        TestUtil.getByteArray(ApplicationProvider.getApplicationContext(), INVALID_POSITIONS);
    Subtitle subtitle = decoder.decode(bytes, bytes.length, false);

    // Negative parameter to \pos() - fall back to the positions implied by middle-left alignment.
    Cue firstCue = Iterables.getOnlyElement(subtitle.getCues(subtitle.getEventTime(0)));
    assertWithMessage("Cue.position").that(firstCue.position).isEqualTo(0.05f);
    assertWithMessage("Cue.lineType").that(firstCue.lineType).isEqualTo(Cue.LINE_TYPE_FRACTION);
    assertWithMessage("Cue.line").that(firstCue.line).isEqualTo(0.5f);

    // Negative parameter to \move() - fall back to the positions implied by middle-left alignment.
    Cue secondCue = Iterables.getOnlyElement(subtitle.getCues(subtitle.getEventTime(2)));
    assertWithMessage("Cue.position").that(secondCue.position).isEqualTo(0.05f);
    assertWithMessage("Cue.lineType").that(secondCue.lineType).isEqualTo(Cue.LINE_TYPE_FRACTION);
    assertWithMessage("Cue.line").that(secondCue.line).isEqualTo(0.5f);

    // Check invalid alignment override (11) is skipped and style-provided one is used (4).
    Cue thirdCue = Iterables.getOnlyElement(subtitle.getCues(subtitle.getEventTime(4)));
    assertWithMessage("Cue.positionAnchor")
        .that(thirdCue.positionAnchor)
        .isEqualTo(Cue.ANCHOR_TYPE_START);
    assertWithMessage("Cue.lineAnchor").that(thirdCue.lineAnchor).isEqualTo(Cue.ANCHOR_TYPE_MIDDLE);
    assertWithMessage("Cue.textAlignment")
        .that(thirdCue.textAlignment)
        .isEqualTo(Layout.Alignment.ALIGN_NORMAL);

    // No braces - fall back to the positions implied by middle-left alignment
    Cue fourthCue = Iterables.getOnlyElement(subtitle.getCues(subtitle.getEventTime(6)));
    assertWithMessage("Cue.position").that(fourthCue.position).isEqualTo(0.05f);
    assertWithMessage("Cue.lineType").that(fourthCue.lineType).isEqualTo(Cue.LINE_TYPE_FRACTION);
    assertWithMessage("Cue.line").that(fourthCue.line).isEqualTo(0.5f);
  }

  @Test
  public void testDecodePositionsWithMissingPlayResY() throws IOException {
    SsaDecoder decoder = new SsaDecoder();
    byte[] bytes =
        TestUtil.getByteArray(
            ApplicationProvider.getApplicationContext(), POSITIONS_WITHOUT_PLAYRES);
    Subtitle subtitle = decoder.decode(bytes, bytes.length, false);

    // The dialogue line has a valid \pos() override, but it's ignored because PlayResY isn't
    // set (so we don't know the denominator).
    Cue firstCue = Iterables.getOnlyElement(subtitle.getCues(subtitle.getEventTime(0)));
    assertWithMessage("Cue.position").that(firstCue.position).isEqualTo(Cue.DIMEN_UNSET);
    assertWithMessage("Cue.lineType").that(firstCue.lineType).isEqualTo(Cue.LINE_TYPE_FRACTION);
    assertWithMessage("Cue.line").that(firstCue.line).isEqualTo(Cue.DIMEN_UNSET);
  }

  @Test
  public void testDecodeInvalidTimecodes() throws IOException {
    // Parsing should succeed, parsing the third cue only.
    SsaDecoder decoder = new SsaDecoder();
    byte[] bytes =
        TestUtil.getByteArray(ApplicationProvider.getApplicationContext(), INVALID_TIMECODES);
    Subtitle subtitle = decoder.decode(bytes, bytes.length, false);

    assertThat(subtitle.getEventTimeCount()).isEqualTo(2);
    assertTypicalCue3(subtitle, 0);
  }

  private static void assertTypicalCue1(Subtitle subtitle, int eventIndex) {
    assertThat(subtitle.getEventTime(eventIndex)).isEqualTo(0);
    assertThat(subtitle.getCues(subtitle.getEventTime(eventIndex)).get(0).text.toString())
        .isEqualTo("This is the first subtitle.");
    assertThat(subtitle.getEventTime(eventIndex + 1)).isEqualTo(1230000);
  }

  private static void assertTypicalCue2(Subtitle subtitle, int eventIndex) {
    assertThat(subtitle.getEventTime(eventIndex)).isEqualTo(2340000);
    assertThat(subtitle.getCues(subtitle.getEventTime(eventIndex)).get(0).text.toString())
        .isEqualTo("This is the second subtitle \nwith a newline \nand another.");
    assertThat(subtitle.getEventTime(eventIndex + 1)).isEqualTo(3450000);
  }

  private static void assertTypicalCue3(Subtitle subtitle, int eventIndex) {
    assertThat(subtitle.getEventTime(eventIndex)).isEqualTo(4560000);
    assertThat(subtitle.getCues(subtitle.getEventTime(eventIndex)).get(0).text.toString())
        .isEqualTo("This is the third subtitle, with a comma.");
    assertThat(subtitle.getEventTime(eventIndex + 1)).isEqualTo(8900000);
  }
}
